/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens
{
  public abstract class SimpleScope : Object
  {
    public Unity.Scope scope { get; protected set; }

    protected abstract int num_results_without_search { get; }
    protected abstract int num_results_global_search { get; }
    protected abstract int num_results_lens_search { get; }

    protected abstract async void perform_search (LensSearch search,
                                                  SearchType search_type,
                                                  owned List<FilterParser> filters,
                                                  int max_results = -1,
                                                  Cancellable? cancellable = null);

    public SimpleScope ()
    {
    }

    protected void initialize ()
    {
      /* Listen for filter changes */
      scope.filters_changed.connect (() => {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      scope.active_sources_changed.connect (() => {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      /* No need to search if only the whitespace changes */
      scope.generate_search_key.connect ((lens_search) => {
        return lens_search.search_string.strip ();
      });

      /* Listen for changes to the lens search entry */
      scope.search_changed.connect ((search, search_type, cancellable) => {
        update_search_async.begin (search, search_type, cancellable);
      });
    }
    
    private async void update_search_async (LensSearch search,
                                            SearchType search_type,
                                            Cancellable cancellable)
    {
      int max_results;
      /*
       * results for a global search from just hitting Super.
       * Here we want to return a smaller number of results with 0 filters.
       */
      if (search_type == SearchType.GLOBAL)
        max_results = num_results_global_search;
      else if (is_search_empty (search))
        max_results = num_results_without_search;
      else
        max_results = num_results_lens_search;

      yield prepare_search (search, search_type, max_results, cancellable);
    }

    private async void prepare_search (LensSearch search,
                                       SearchType search_type,
                                       int max_results,
                                       Cancellable cancellable)
    {
      var results_model = search.results_model;

      List<FilterParser> filters = new List<FilterParser> ();
      // don't apply filters to global searches
      if (search_type != SearchType.GLOBAL)
      {
        Filter filter = scope.get_filter ("genre");
        if (filter.filtering)
          filters.append (new GenreFilterParser (filter as CheckOptionFilterCompact));

        filter = scope.get_filter ("decade");
        if (filter.filtering)
          filters.append (new DecadeFilterParser (filter as MultiRangeFilter));
      }

      results_model.clear ();

      // don't perform search is all sources are inactive
      if (scope.sources.options.length () > 0 && scope.sources.filtering)
      {
        bool any_active = false;
        foreach (var source in scope.sources.options)
        {
          if (source.active) any_active = true;
        }
        if (!any_active)
        {
          search.finished ();
          return;
        }
      }

      yield perform_search (search, search_type, (owned) filters, max_results, cancellable);

      if (results_model.get_n_rows () == 0)
      {
        search.set_reply_hint ("no-results-hint",
          _("Sorry, there is no music that matches your search."));
      }

      search.finished ();
    }

    protected bool is_search_empty (LensSearch search)
      requires (search.search_string != null)
    {
      return search.search_string.strip () == "";
    }
  }
}
