import os
import dbus
import logging
from dbus.mainloop.glib import DBusGMainLoop
from usbcreator.backends.base import Backend
from usbcreator.misc import *

DISKS_IFACE = 'org.freedesktop.DeviceKit.Disks'
DEVICE_IFACE = 'org.freedesktop.DeviceKit.Disks.Device'
PROPS_IFACE = 'org.freedesktop.DBus.Properties'

class DeviceKitBackend(Backend):
    def __init__(self, bus=None):
        Backend.__init__(self)
        self.mounted_source = ''
        logging.debug('DeviceKitBackend')
        DBusGMainLoop(set_as_default=True)
        if bus:
            self.bus = bus
        else:
            self.bus = dbus.SystemBus()
        devkit_obj = self.bus.get_object('org.freedesktop.DeviceKit.Disks',
                                         '/org/freedesktop/DeviceKit/Disks')
        self.devkit = dbus.Interface(devkit_obj,
                                     'org.freedesktop.DeviceKit.Disks')
        self.helper = self.bus.get_object('com.ubuntu.USBCreator',
                                          '/com/ubuntu/USBCreator')
        self.helper = dbus.Interface(self.helper, 'com.ubuntu.USBCreator')

    # Device detection and processing functions.

    def detect_devices(self):
        '''Start looking for new devices to add.  Devices added will be sent to
        the fronted using frontend.device_added.  Devices will only be added as
        they arrive if a main loop is present.'''
        logging.debug('detect_devices')
        self.bus.add_signal_receiver(self._device_added,
            'DeviceAdded',
            'org.freedesktop.DeviceKit.Disks',
            'org.freedesktop.DeviceKit.Disks',
            '/org/freedesktop/DeviceKit/Disks')
        self.bus.add_signal_receiver(self._device_changed,
            'DeviceChanged',
            'org.freedesktop.DeviceKit.Disks',
            'org.freedesktop.DeviceKit.Disks',
            '/org/freedesktop/DeviceKit/Disks')
        self.bus.add_signal_receiver(self._device_removed,
            'DeviceRemoved',
            'org.freedesktop.DeviceKit.Disks',
            'org.freedesktop.DeviceKit.Disks',
            '/org/freedesktop/DeviceKit/Disks')
        def handle_reply(res):
            for r in res:
                self._device_added(r)
        def handle_error(err):
            logging.error('Unable to enumerate devices: %s' % str(err))
        self.devkit.EnumerateDevices(reply_handler=handle_reply,
                                     error_handler=handle_error)
        
    def _device_added(self, device):
        # Make sure we add it only once
        if device in self.sources:
            return
        elif device in self.targets:
            return

        logging.debug('device_added: %s' % device)
        devkit_obj = self.bus.get_object('org.freedesktop.DeviceKit.Disks',
                                         device)
        d = dbus.Interface(devkit_obj, 'org.freedesktop.DBus.Properties')

        if d.Get(device, 'device-is-optical-disc'):
            self._add_cd(device)
        if not d.Get(device, 'device-is-system-internal'):
            if d.Get(device, 'device-is-partition'):
                self._add_partition(device)
            elif d.Get(device, 'device-is-drive'):
                if not d.Get(device, 'device-is-optical-disc'):
                    self._add_drive(device)

    def _device_changed(self, device):
        # Make sure we add it only once
        if device in self.sources:
            return
        elif device in self.targets:
            return

        devkit_obj = self.bus.get_object('org.freedesktop.DeviceKit.Disks',
                                         device)
        d = dbus.Interface(devkit_obj, 'org.freedesktop.DBus.Properties')

        # The only changed device we are concerned with is optical disk
        if d.Get(device, 'device-is-optical-disc'):
            logging.debug('device_changed: %s' % device)
            self._add_cd(device)

    def _add_cd(self, device):
        logging.debug('cd added: %s' % device)
        dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', device)
        def get(prop):
            return dk.Get(device, prop, dbus_interface=PROPS_IFACE)
        label = get('id-label')
        if not get('device-is-mounted'):
            try:
                mp = dk.FilesystemMount('', [], dbus_interface=DEVICE_IFACE)
            except dbus.DBusException, e:
                logging.exception('Could not mount the device:')
                return
        mount = get('device-mount-paths')[0]
        device_file = get('device-file')
        total, free = fs_size(mount)
        self.sources[device] = {
            'device' : device_file,
            'size' : total,
            'label' : label,
            'type' : SOURCE_CD,
        }
        if callable(self.source_added_cb):
            self.source_added_cb(device)

    def _add_partition(self, device):
        logging.debug('partition added: %s' % device)
        dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', device)
        def get(prop):
            return dk.Get(device, prop, dbus_interface=PROPS_IFACE)

        fstype = get('id-type')
        logging.debug('id-type: %s' % fstype)
        if fstype == 'vfat':
            status = CAN_USE
        else:
            status = NEED_FORMAT
        label = get('id-label')
        logging.debug('id-label: %s' % label)
        if fstype == 'vfat' and not get('device-is-mounted'):
            try:
                mp = dk.FilesystemMount('', [], dbus_interface=DEVICE_IFACE)
            except dbus.DBusException, e:
                logging.exception('Could not mount the device:')
                return
        mount = get('device-mount-paths')
        if mount:
            mount = mount[0]
            total, free = fs_size(mount)
        else:
            # FIXME evand 2009-09-11: This is going to have weird side effects.
            # If the device cannot be mounted, but is a vfat filesystem, that
            # is.  Is this really the right approach?
            total = get('partition-size')
            free = -1
            # mount cannot be left as a dbus Array else we get crashes later - rgreening 2009-09-23
            mount = ''
        logging.debug('mount: %s' % mount)
        device_file = get('device-file')
        if total > 0:
            self.targets[unicode(device)] = {
                'label' : unicode(label),
                'free' : free,
                'device' : unicode(device_file),
                'capacity' : total,
                'status' : status,
                'mountpoint' : mount,
                'persist' : 0,
            }
            self._update_free(unicode(device))
            if callable(self.target_added_cb):
                self.target_added_cb(device)
        else:
            logging.debug('not adding device: 0 byte partition.')

    def _add_drive(self, device):
        logging.debug('disk added: %s' % device)
        dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', device)
        def get(prop):
            return dk.Get(device, prop, dbus_interface=PROPS_IFACE)
        device_file = get('device-file')
        size = get('device-size')
        if size > 0:
            self.targets[unicode(device)] = {
                'label' : '',
                'free' : -1,
                'device' : unicode(device_file),
                'capacity' : size,
                'status' : NEED_FORMAT,
                'mountpoint' : None,
                'persist' : 0,
            }
            if callable(self.target_added_cb):
                self.target_added_cb(device)
        else:
            logging.debug('not adding device: 0 byte disk.')

    def _device_removed(self, device):
        logging.debug('Device has been removed from the system: %s' % device)
        if device in self.sources:
            if callable(self.source_removed_cb):
                self.source_removed_cb(device)
            self.sources.pop(device)
        elif device in self.targets:
            if callable(self.target_removed_cb):
                self.target_removed_cb(device)
            self.targets.pop(device)

    # Device manipulation functions.
    def _is_casper_cd(self, filename):
        cmd = ['isoinfo', '-J', '-i', filename, '-x', '/.disk/info']
        try:
            output = popen(cmd, stderr=None)
            if output:
                return output
        except USBCreatorProcessException:
            # TODO evand 2009-07-26: Error dialog.
            logging.error('Could not extract .disk/info.')
        return None

    def _unmount_all(self, parent):
        '''Unmounts all the partitions that are children of parent.'''
        devices = self.devkit.EnumerateDevices()
        for device in devices:
            dev = self.bus.get_object('org.freedesktop.DeviceKit.Disks', device)
            if dev.Get(device, 'partition-slave', dbus_interface=PROPS_IFACE) == parent:
                try:
                    logging.debug('Unmounting %s' % device)
                    dev.FilesystemUnmount([], dbus_interface=DEVICE_IFACE)
                except dbus.DBusException, e:
                    #TODO
                    print e

    def open(self, udi):
        mp = self.targets[udi]['mountpoint']
        if not mp:
            try:
                dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', udi)
                mp = dk.FilesystemMount('', [], dbus_interface=DEVICE_IFACE)
            except dbus.DBusException:
                logging.exception('Could not mount the device:')
                return ''
        try:
            popen(['mount', '-o', 'remount,rw', mp])
        except USBCreatorProcessException:
            logging.exception('Could not mount the device:')
            return ''
        return mp
    
    def format(self, device):
        # TODO evand 2009-08-25: Needs a confirmation dialog.
        try:
            dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', device)
            device = dk.Get(device, 'device-file', dbus_interface=PROPS_IFACE)
            self.helper.Format(device, timeout=MAX_DBUS_TIMEOUT)
        except dbus.DBusException:
            logging.exception('Could not format the device:')

    def install(self, source, target, persist):
        # XXX evand 2009-07-29: Leave mounted, so we don't rip the
        # mountpoint out from under the user, if they've decided to use it.
        # FIXME: rgreening 2009-09-01 leaving mount with root perms prevents users from being able to safely remove stick
        # TODO evand 2009-07-31: Lock source and target.
        logging.debug('install source: %s' % source)
        logging.debug('install target: %s' % target)
        logging.debug('install persistence: %d' % persist)
        stype = self.sources[source]['type']
        if stype == SOURCE_CD:
            dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', source)
            def get(prop):
                return dk.Get(source, prop, dbus_interface=PROPS_IFACE)
            if not get('device-is-mounted'):
                source = dk.FilesystemMount('', [], dbus_interface=DEVICE_IFACE)
            else:
                source = get('device-mount-paths')[0]
        elif stype == SOURCE_ISO:
            isofile = self.sources[source]['device']
            source = self.helper.Mount(isofile)
            self.mounted_source = source
        
        dk = self.bus.get_object('org.freedesktop.DeviceKit.Disks', target)
        def get(prop):
            return dk.Get(target, prop, dbus_interface=PROPS_IFACE)
        dev = get('device-file')
        if stype == SOURCE_IMG:
            target = None
            self._unmount_all(target)
        else:
            if not get('device-is-mounted'):
                target = dk.FilesystemMount('', [], dbus_interface=DEVICE_IFACE)
            else:
                target = get('device-mount-paths')[0]
            # XXX evand 2009-09-16: In my opinion, it should be possible to do
            # this:
            # dk.FilesystemMount('', ['remount,rw'], dbus_interface=DEVICE_IFACE)
            # 
            # But that returns "org.freedesktop.DeviceKit.Disks.Error.Busy:
            # /dev/sdb1 is mounted", so we'll have to work around it for now.
            self.helper.RemountRW(dev)
        Backend.install(self, source, target, persist, device=dev)

    def cancel_install(self):
        Backend.cancel_install(self)
        self.unmount()

    def unmount(self):
        if self.mounted_source:
            self.helper.Unmount(self.mounted_source)
        try:
            self.helper.Shutdown()
        except dbus.DBusException:
            logging.exception('Could not shut down the dbus service.')
