/**
 * @file geis_backend_test_fixture.c
 * @brief GEIS mock back end test fixture implementation
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#include "geis_backend.h"
#include "geis_backend_protected.h"

#include "geis_attr.h"
#include "geis_class.h"
#include "geis_device.h"
#include "geis_event.h"
#include "geis_filter.h"
#include "geis_filter_term.h"
#include "geis_frame.h"
#include "geis_group.h"
#include "geis_logging.h"
#include "geis_private.h"
#include "geis_subscription.h"
#include "geis_touch.h"
#include <stdlib.h>


static inline GeisSize
_min(GeisSize a, GeisSize b)
{
  return (a < b) ? a : b;
}


static inline GeisSize
_max(GeisSize a, GeisSize b)
{
  return (a > b) ? a : b;
}

static GeisSize   g_min_touches = 1;
static GeisSize   g_max_touches = 5;

typedef struct _GeisBackendTestFixture
{
  struct _GeisBackend  tf_base;
  Geis                 tf_geis;
} *GeisBackendTestFixture;


static void 
_finalize(GeisBackend be)
{
  GeisBackendTestFixture tf __attribute__((unused)) = (GeisBackendTestFixture)be;
}


static void
_translate_class_term_to_xcb(GeisFilterTerm  term,
                             GeisString     *class_name,
                             GeisSize       *min_touches,
                             GeisSize       *max_touches)
{
  GeisAttr   attr = geis_filter_term_attr(term);
  GeisString name = geis_attr_name(attr);
  GeisFilterOperation op = geis_filter_term_operation(term);

  if (0 == strcmp(name, GEIS_CLASS_ATTRIBUTE_NAME)
      && op == GEIS_FILTER_OP_EQ)
  {
    *class_name = geis_attr_value_to_string(attr);
  }
  else if (0 == strcmp(name, GEIS_GESTURE_ATTRIBUTE_TOUCHES))
  {
    GeisSize touches = geis_attr_value_to_integer(attr);
    switch (op)
    {
      case GEIS_FILTER_OP_GT:
	*min_touches = _max(*min_touches, touches+1);
	break;
      case GEIS_FILTER_OP_GE:
	*min_touches = _max(*min_touches, touches);
	break;
      case GEIS_FILTER_OP_LT:
	*max_touches = _min(touches-1, *max_touches);
	break;
      case GEIS_FILTER_OP_LE:
	*max_touches = _min(touches, *max_touches);
	break;
      case GEIS_FILTER_OP_EQ:
	*min_touches = _max(*min_touches, touches);
	*max_touches = _min(touches, *max_touches);
	break;
      case GEIS_FILTER_OP_NE:
	break;
    }
  }
  else
  {
    geis_warning("invalid filter term");
  }

}


static GeisStatus _subscribe(GeisBackend  be, GeisSubscription sub);

static GeisStatus 
_unsubscribe(GeisBackend      be  __attribute__((unused)),
             GeisSubscription sub __attribute__((unused)))
{
  geis_debug("begins");
  GeisStatus status = GEIS_STATUS_SUCCESS;
  geis_debug("ends");
  return status;
}

static struct _GeisBackendVtable tf_vtbl = {
  _finalize,
  _subscribe,
  _unsubscribe
};


static void _create_test_devices(GeisBackendTestFixture tf)
{
  GeisDevice device = geis_device_new("abs-test-device", 0);
  GeisEvent  event = geis_event_new(GEIS_EVENT_DEVICE_AVAILABLE);
  GeisAttr   attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_DEVICE,
                                  GEIS_ATTR_TYPE_POINTER,
                                  device);
  geis_event_add_attr(event, attr);
  geis_post_event(tf->tf_geis, event);
}


static void _create_test_classes(GeisBackendTestFixture tf)
{
  GeisAttr    attr;
  GeisInteger i1 = 1;
  GeisGestureClass gesture_class = geis_gesture_class_new("poke", 0);
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                       GEIS_ATTR_TYPE_INTEGER,
                       &i1);
  geis_gesture_class_add_attr(gesture_class, attr);

  GeisEvent  event = geis_event_new(GEIS_EVENT_CLASS_AVAILABLE);
  attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_CLASS,
                       GEIS_ATTR_TYPE_POINTER,
                       gesture_class);
  geis_event_add_attr(event, attr);
  geis_post_event(tf->tf_geis, event);
}


static void
_create_gesture_events(GeisBackendTestFixture tf)
{
  geis_debug("begins");
  GeisSize     i;

  GeisFloat    attr_float;
  GeisInteger  attr_int;
  GeisAttr     attr = NULL;

  GeisEvent    event = geis_event_new(GEIS_EVENT_GESTURE_BEGIN);
  GeisGroupSet groupset = geis_groupset_new();
  GeisGroup    group = geis_group_new(1);
  GeisAttr     group_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_GROUPSET,
                                          GEIS_ATTR_TYPE_POINTER,
                                          groupset);
  GeisTouchSet touchset = geis_touchset_new();
  GeisAttr     touch_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_TOUCHSET,
                                          GEIS_ATTR_TYPE_POINTER,
                                          touchset);

  GeisFrame frame = geis_frame_new(1);

  attr_int = 13;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_DEVICE_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &attr_int);
  geis_frame_add_attr(frame, attr);

  attr_int = 1;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TIMESTAMP,
                       GEIS_ATTR_TYPE_INTEGER,
                       &attr_int);
  geis_frame_add_attr(frame, attr);

  attr_int = 2;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_ROOT_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &attr_int);
  geis_frame_add_attr(frame, attr);

  attr_int = 3;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &attr_int);
  geis_frame_add_attr(frame, attr);

  attr_int = 4;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_CHILD_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &attr_int);
  geis_frame_add_attr(frame, attr);

  attr_float = 123.456;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_FOCUS_X,
                       GEIS_ATTR_TYPE_FLOAT,
                       &attr_float);
  geis_frame_add_attr(frame, attr);

  attr_float = 987.654;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_FOCUS_Y,
                       GEIS_ATTR_TYPE_FLOAT,
                       &attr_float);
  geis_frame_add_attr(frame, attr);

  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_GESTURE_NAME,
                       GEIS_ATTR_TYPE_STRING,
                       "mock gesture");
  geis_frame_add_attr(frame, attr);

  geis_debug("min_touches=%d", g_min_touches);
  attr_int = g_min_touches;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                       GEIS_ATTR_TYPE_INTEGER,
                       &attr_int);
  geis_frame_add_attr(frame, attr);

  for (i = 0; i < g_min_touches; ++i)
  {
    GeisTouch    touch = geis_touch_new(1);
    geis_touchset_insert(touchset, touch);
    geis_frame_add_touchid(frame, geis_touch_id(touch));
  }

  geis_group_insert_frame(group, frame);

  geis_groupset_insert(groupset, group);

  geis_event_add_attr(event, group_attr);
  geis_event_add_attr(event, touch_attr);

  geis_post_event(tf->tf_geis, event);
  geis_debug("ends");
}


GeisBackend
geis_backend_new_test_fixture(Geis geis,
                              GeisBoolean track_devices,
                              GeisBoolean track_classes)
{
  GeisBackendTestFixture tf = calloc(1, sizeof(struct _GeisBackendTestFixture));
  if (!tf)
  {
    geis_error("failed to allocate GEIS back end test fixture");
    goto nomem;
  }

  geis_backend_init_base(&tf->tf_base, &tf_vtbl, "GEIS2 test fixture");
  tf->tf_geis = geis;

  if (track_devices)
  {
    _create_test_devices(tf);
  }

  if (track_classes)
  {
    _create_test_classes(tf);
  }

  geis_debug("%s back end created", geis_backend_name(&tf->tf_base));

nomem:
  return (GeisBackend)tf;
}

GeisStatus 
_subscribe(GeisBackend      be,
           GeisSubscription sub)
{
  geis_debug("begins");
  GeisBackendTestFixture tf = (GeisBackendTestFixture)be;
  GeisStatus status = GEIS_STATUS_SUCCESS;
  GeisString class_name;
  GeisSize   i, j;
  int num_device_filters, num_class_filters, num_region_filters;

  for (i = 0; i < geis_subscription_filter_count(sub); ++i)
  {
    GeisFilter filter = geis_subscription_filter(sub, i);
    geis_debug("processing filter '%s'", geis_filter_name(filter));
    GeisSize term_count = geis_filter_term_count(filter);
    for (j = 0; j < term_count; ++j)
    {
      GeisFilterTerm term = geis_filter_term(filter, j);
      geis_debug("  TERM \"%s\" (%d) \"%s\"",
                 geis_attr_name(geis_filter_term_attr(term)),
                 geis_filter_term_operation(term),
                 geis_attr_value_to_string(geis_filter_term_attr(term)));
      switch (geis_filter_term_facility(term))
      {
	case GEIS_FILTER_DEVICE:
	  ++num_device_filters;
	  break;
	case GEIS_FILTER_CLASS:
	  _translate_class_term_to_xcb(term, &class_name,
				       &g_min_touches, &g_max_touches);
	  ++num_class_filters;
	  break;
	case GEIS_FILTER_REGION:
	  ++num_region_filters;
	  break;
	default:
	  geis_warning("-- unknown term --");
	  break;
      }
    }
  }
  _create_gesture_events(tf);
  geis_debug("ends");
  return status;
}



