/**
 * @file geis_error.h
 * @brief internal GEIS error facilities
 *
 * This file is the internal interface to the GEIS API object.  It provides the
 * implementation hooks for back ends to pass information through the API.
 *
 * Copyright 2010, 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#ifndef GEIS_PRIVATE_H_
#define GEIS_PRIVATE_H_

#include <geis/geis.h>

#include "geis_error.h"
#include "geis_backend_multiplexor.h"
#include "geis_subscription.h"


/**
 * Increases the reference count of an API instance object.
 *
 * @param[in] geis  The API instance.
 *
 * @returns the same API instance.
 */
Geis geis_ref(Geis geis);

/**
 * Decremenets the reference count of an API instance object.
 *
 * @param[in] geis  The API instance.
 *
 * If the reference count on the API instance object drops to zero, the object
 * is destroyed.
 */
void geis_unref(Geis geis);

/**
 * Gets the error stack from the geis object.
 *
 * @param[in] geis  The API instance.
 */
GeisErrorStack *geis_error_stack(Geis geis);

/**
 * Gets the subscription container from the geis object.
 *
 * @param[in] geis  The API instance.
 */
GeisSubBag geis_subscription_bag(Geis geis);

/**
 * Adds a back end file descriptor to multiplex.
 *
 * @param[in] geis     The API instance.
 * @param[in] fd       The file descriptor to add.
 * @param[in] callback The file descriptor event callback.
 * @param[in] context  A contextual datum.
 */
void geis_multiplex_fd(Geis                        geis,
                       int                         fd,
                       GeisBackendFdEventCallback  callback,
                       void                       *context);

/**
 * Removes a back end file descriptor from the multiplex.
 *
 * @param[in] geis     The API instance.
 * @param[in] fd       The file descriptor to remove.
 */
void geis_demultiplex_fd(Geis geis, int fd);

/**
 * Posts a new event through the API.
 *
 * @param[in] geis  The API instance.
 * @param[in] event The GEIS event.
 */
void geis_post_event(Geis geis, GeisEvent event);

/**
 * Activates a subscription in the back end.
 *
 * @param[in] geis  The API instance.
 * @param[in] sub   The subscription.
 */
GeisStatus geis_activate_subscription(Geis geis, GeisSubscription sub);

/**
 * Activates a subscription in the back end.
 *
 * @param[in] geis  The API instance.
 * @param[in] sub   The subscription.
 */
GeisStatus geis_deactivate_subscription(Geis geis, GeisSubscription sub);

/**
 * Gets the type of a named device attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the device attr.
 *
 * Gets the type of a device attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all device attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_device_attr_type(Geis geis, GeisString attr_name);

/**
 * Gets the type of a named class attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the class attr.
 *
 * Gets the type of a class attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all class attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_class_attr_type(Geis geis, GeisString attr_name);

/**
 * Gets the type of a named region attr.
 *
 * @param[in]  geis      The API instance.
 * @param[in]  attr_name The name of the class attr.
 *
 * Gets the type of a region attr by name, assuming the attr is known.
 *
 * There is a basic assumption that all region attrs of the same name are of the
 * same type.
 *
 * @returns the type of the attr, GEIS_ATTR_TYPE_UNKNOWN if the attr is unknown.
 */
GeisAttrType geis_get_region_attr_type(Geis geis, GeisString attr_name);

#endif /* GEIS_PRIVATE_H_ */
