/**
 * internal unit test for the back end event posting interface
 */
#include <check.h>

#include <errno.h>
#include "geis/geis.h"
#include "libutouch-geis/geis_event.h"
#include "libutouch-geis/geis_private.h"
#include <string.h>
#include <sys/select.h>
#include <unistd.h>


/* fixtures */
Geis g_geis;

static void
construct_geis()
{
  g_geis = geis_new(GEIS_INIT_UTOUCH_MOCK_ENGINE, NULL);
}

static void
destroy_geis()
{
  geis_delete(g_geis);
}

static void
testcase_event_callback(int fd,
                        GeisBackendMultiplexorEvent event CK_ATTRIBUTE_UNUSED,
                        void *context)
{
  char buf[2];
  ssize_t ssize CK_ATTRIBUTE_UNUSED;

  ssize = read(fd, buf, 1);
  *(int *)context += 1; 
}

START_TEST(backend_post)
{
  GeisStatus status;
  GeisEvent event_in = geis_event_new(GEIS_EVENT_GESTURE_END);
  GeisEvent event_out;

  geis_post_event(g_geis, event_in);
  status = geis_dispatch_events(g_geis);
  fail_unless(status == GEIS_STATUS_SUCCESS,
              "unexpected status from geis_dispatch_events");
  status = geis_next_event(g_geis, &event_out);
  fail_unless(status == GEIS_STATUS_SUCCESS,
              "unexpected status from geis_next_event");
  fail_unless(geis_event_type(event_in) == geis_event_type(event_out),
              "event in and event out types do not match");

  geis_post_event(g_geis, event_in);
  geis_post_event(g_geis, event_in);
  status = geis_dispatch_events(g_geis);
  fail_unless(status == GEIS_STATUS_SUCCESS,
              "unexpected status from geis_dispatch_events");
  status = geis_next_event(g_geis, &event_out);
  fail_unless(status == GEIS_STATUS_CONTINUE,
              "expected CONTINUE status from geis_next_event");
  status = geis_next_event(g_geis, &event_out);
  fail_unless(status == GEIS_STATUS_SUCCESS,
              "expected SUCCESS status from geis_next_event");

  geis_event_delete(event_in);
}
END_TEST

START_TEST(multiplex_fd)
{
  int pfd[2];
  int geis_fd;
  int called = 0;
  int status = 0;
  int first_time = 1;
  GeisStatus gstat = geis_get_configuration(g_geis,
                                            GEIS_CONFIGURATION_FD,
                                            &geis_fd);
  fail_unless(gstat == GEIS_STATUS_SUCCESS, "unable to get GEIS fd");

  fail_unless(pipe(pfd) == 0, "error %d creating self-pipe: %d",
              errno, strerror(errno));
  geis_multiplex_fd(g_geis, pfd[0], testcase_event_callback, &called);

  while (1)
  {
    fd_set fds;
    FD_ZERO(&fds);
    FD_SET(geis_fd, &fds);

    struct timeval tm;
    tm.tv_sec = 0;
    tm.tv_usec = 5000;

    status = select(geis_fd+1, &fds, NULL, NULL, &tm);
    fail_if(status < 0, "error in select");
    if (status < 0)
    {
      break;
    }
    else if (0 == status)
    {
      fail_unless(first_time, "select timed out before read");
      if (!first_time)
      {
	break;
      }
      first_time = 0;
      ssize_t ssize = write(pfd[1], "1", 1);
      fail_unless(ssize == 1, "error writing to self-pipe");
    }
    else
    {
      geis_dispatch_events(g_geis);
      break;
    }
  }

  fail_if(called == 0, "event callback not called");
  fail_if(called >  1, "event callback called too many times");
}
END_TEST

/* boilerplate */
Suite *
make_backend_event_posting_suite()
{
  Suite *s = suite_create("utouch-geis2-geis-private");

  TCase *usage = tcase_create("backend-event-posting");
  tcase_add_checked_fixture(usage, construct_geis, destroy_geis);
  tcase_add_test(usage, backend_post);
  tcase_add_test(usage, multiplex_fd);
  suite_add_tcase(s, usage);

  return s;
}

