/**
 * Gtest test suite for GEIS v2 attributes.
 *
 * Copyright 2012 Canonical Ltd.
 */

#include <functional>
#include "geis/geis.h"
#include "gtest_evemu_device.h"
#include "gtest_geis_fixture.h"
#include <gtest/gtest.h>
#include <mutex>
#include <sys/select.h>
#include <sys/time.h>


namespace
{

static const std::string TEST_DEVICE_PROP_FILE(
    TEST_ROOT_DIR "recordings/touchscreen_a/device.prop");

/**
 * Fixture for testing expected attributes.  This has to be a separate class
 * because of the way Java reflection is used in jUnit.
 */
class GeisAttributeTests
: public GTestGeisFixture
{
public:
  GeisAttributeTests()
  : evemu_device_(TEST_DEVICE_PROP_FILE)
  { }

protected:
  /* FIXME: Move this to test body so we can use different devices. Requires a
   * fix for lp:944822. */
  Testsuite::EvemuDevice evemu_device_;
};


/*
 * Regression test for lp:394398: rotation angle is swapped with rotation angle
 * delta.
 *
 * The captive "rotate90" events whould produce a rotation gesture with a final
 * rotation angle of somewhere around -pi/2 radians.
 */
TEST_F(GeisAttributeTests, rotate_90)
{
  static const std::string TEST_DEVICE_EVENTS_FILE(
      TEST_ROOT_DIR "recordings/touchscreen_a/rotate90.record");

  GeisBoolean found_angle = GEIS_FALSE;
  GeisSubscription sub = geis_subscription_new(geis_,
                                               "rotate_90",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";

  GeisFilter filter = geis_filter_new(geis_, "rotate");
  EXPECT_TRUE(filter != NULL) << "can not create filter";

  GeisStatus fs = geis_filter_add_term(filter,
               GEIS_FILTER_CLASS,
               GEIS_CLASS_ATTRIBUTE_NAME, GEIS_FILTER_OP_EQ, GEIS_GESTURE_ROTATE,
               GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_FILTER_OP_GT, 1,
               NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter";

  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter";

  EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
              << "can not activate subscription";

  GeisStatus dispatch_status= geis_dispatch_events(geis_);
  while (dispatch_status == GEIS_STATUS_CONTINUE
      || dispatch_status == GEIS_STATUS_SUCCESS)
  {
    GeisEvent  event;
    GeisStatus event_status = geis_next_event(geis_, &event);
    while (event_status == GEIS_STATUS_CONTINUE
        || event_status == GEIS_STATUS_SUCCESS)
    {
      if (geis_event_type(event) == GEIS_EVENT_INIT_COMPLETE)
      {
        evemu_device_.play(TEST_DEVICE_EVENTS_FILE);
      }
      else if (geis_event_type(event) == GEIS_EVENT_GESTURE_END)
      {
        GeisAttr attr = geis_event_attr_by_name(event, GEIS_EVENT_ATTRIBUTE_GROUPSET);
        EXPECT_TRUE(attr != NULL) << "event is missing groupset attr";

        GeisGroupSet groupset = (GeisGroupSet)geis_attr_value_to_pointer(attr);
        EXPECT_TRUE(groupset != NULL) << "event is missing groupset";

        for (GeisSize i = 0; i < geis_groupset_group_count(groupset); ++i)
        {
          GeisGroup group = geis_groupset_group(groupset, i);
          EXPECT_TRUE(group != NULL) << "group " << i << " not found in groupset";

          for (GeisSize j = 0; j < geis_group_frame_count(group); ++j)
          {
            GeisFrame frame = geis_group_frame(group, j);
            EXPECT_TRUE(frame != NULL) << "frame " << j << " not found in group";

            GeisAttr attr = geis_frame_attr_by_name(frame,
                                                    GEIS_GESTURE_ATTRIBUTE_ANGLE);
            EXPECT_TRUE(attr != NULL) << "angle attribute not found in frame";

            GeisFloat angle = geis_attr_value_to_float(attr);
            EXPECT_NEAR(angle, -1.6f, 0.4f);
            found_angle = GEIS_TRUE;
            goto test_done;
          }
        }
      }
      geis_event_delete(event);

      event_status = geis_next_event(geis_, &event);
    }

    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET(geis_fd(), &read_fds);
    timeval tmo = { 5, 0 };
    int sstat = select(geis_fd() + 1, &read_fds, NULL, NULL, &tmo);
    EXPECT_GT(sstat, -1) << "error in select";
    if (sstat == 0)
      break;
    dispatch_status = geis_dispatch_events(geis_);
  }

test_done:
  EXPECT_TRUE(found_angle);

  geis_filter_delete(filter);
  geis_subscription_delete(sub);
}

/*
 * Regression test for lp:967267: Position delta attributes incorrect when touch
 * count changes.
 *
 * Some gestures continue when the number of touches changes. When this happens,
 * much of the gesture state is reset. This includes the original center and
 * cumulative transformation matrix.
 *
 * This test ensures the position delta attributes stay in a reasonable range
 * after a two to one touch drag transition.
 */
TEST_F(GeisAttributeTests, TouchChangePositionDelta)
{
  static const std::string TEST_DEVICE_EVENTS_FILE(
      TEST_ROOT_DIR "recordings/touchscreen_a/drag_2_to_1.record");

  GeisBoolean found_delta = GEIS_FALSE;
  GeisSubscription sub = geis_subscription_new(geis_,
                                               "drag changing touches",
                                               GEIS_SUBSCRIPTION_NONE);
  EXPECT_TRUE(sub != NULL) << "can not create subscription";

  GeisFilter filter = geis_filter_new(geis_, "drag");
  EXPECT_TRUE(filter != NULL) << "can not create filter";

  GeisStatus fs = geis_filter_add_term(filter,
               GEIS_FILTER_CLASS,
               GEIS_CLASS_ATTRIBUTE_NAME, GEIS_FILTER_OP_EQ, GEIS_GESTURE_DRAG,
               GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_FILTER_OP_GE, 1,
               NULL);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not add class to filter";

  fs = geis_subscription_add_filter(sub, filter);
  EXPECT_EQ(fs, GEIS_STATUS_SUCCESS) << "can not subscribe filter";

  EXPECT_EQ(GEIS_STATUS_SUCCESS, geis_subscription_activate(sub))
      << "can not activate subscription";

  GeisStatus dispatch_status= geis_dispatch_events(geis_);
  while (dispatch_status == GEIS_STATUS_CONTINUE
      || dispatch_status == GEIS_STATUS_SUCCESS)
  {
    GeisEvent  event;
    GeisStatus event_status = geis_next_event(geis_, &event);
    while (event_status == GEIS_STATUS_CONTINUE
        || event_status == GEIS_STATUS_SUCCESS)
    {
      switch (geis_event_type(event))
      {
        case GEIS_EVENT_INIT_COMPLETE:
          evemu_device_.play(TEST_DEVICE_EVENTS_FILE);
          break;

        case GEIS_EVENT_GESTURE_BEGIN:
        case GEIS_EVENT_GESTURE_UPDATE:
        case GEIS_EVENT_GESTURE_END:
          GeisAttr attr = geis_event_attr_by_name(event, GEIS_EVENT_ATTRIBUTE_GROUPSET);
          EXPECT_TRUE(attr != NULL) << "event is missing groupset attr";

          GeisGroupSet groupset = (GeisGroupSet)geis_attr_value_to_pointer(attr);
          EXPECT_TRUE(groupset != NULL) << "event is missing groupset";

          for (GeisSize i = 0; i < geis_groupset_group_count(groupset); ++i)
          {
            GeisGroup group = geis_groupset_group(groupset, i);
            EXPECT_TRUE(group != NULL) << "group " << i << " not found in groupset";

            for (GeisSize j = 0; j < geis_group_frame_count(group); ++j)
            {
              GeisFrame frame = geis_group_frame(group, j);
              EXPECT_TRUE(frame != NULL) << "frame " << j << " not found in group";

              GeisAttr attr = geis_frame_attr_by_name(frame,
                                                      GEIS_GESTURE_ATTRIBUTE_DELTA_X);
              EXPECT_TRUE(attr != NULL) << "angle attribute not found in frame";

              GeisFloat delta = geis_attr_value_to_float(attr);

              /* The drag is nearly vertical. All X deltas should be within 0.2%
               * of the screen width. On failure, the delta can vary up to 20%
               * of the screen width. */
              int width = WidthOfScreen(DefaultScreenOfDisplay(Display()));
              EXPECT_NEAR(delta, 0, 0.01 * width);
              found_delta = GEIS_TRUE;
            }
          }
          break;
      }
      geis_event_delete(event);

      event_status = geis_next_event(geis_, &event);
    }

    fd_set read_fds;
    FD_ZERO(&read_fds);
    FD_SET(geis_fd(), &read_fds);
    timeval tmo = { 5, 0 };
    int sstat = select(geis_fd() + 1, &read_fds, NULL, NULL, &tmo);
    EXPECT_GT(sstat, -1) << "error in select";
    if (sstat == 0)
      break;
    dispatch_status = geis_dispatch_events(geis_);
  }

  EXPECT_TRUE(found_delta);

  geis_filter_delete(filter);
  geis_subscription_delete(sub);
}

} // anonymous namespace

