/*****************************************************************************
 *
 * grail - Gesture Recognition And Instantiation Library
 *
 * Copyright (C) 2010-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#ifndef GRAIL_BITS_H
#define GRAIL_BITS_H

#ifdef __cplusplus
extern "C" {
#endif

/* Macros that set symbol visibilities in shared libraries properly.
 * Adapted from http://gcc.gnu.org/wiki/Visibility
 */

#if defined _WIN32 || defined __CYGWIN__
        #ifdef BUILDING_GRAIL
                #define GRAIL_PUBLIC __declspec(dllexport)
        #else
                #define GRAIL_PUBLIC __declspec(dllimport)
        #endif
        #define GRAIL_PRIVATE
#else
        #if defined __GNUC__
                #define GRAIL_PUBLIC __attribute__ ((visibility("default")))
                #define GRAIL_PRIVATE  __attribute__ ((visibility("hidden")))
        #else
                #pragma message ("Compiler does not support symbol visibility.")
                #define GRAIL_PUBLIC
                #define GRAIL_PRIVATE
        #endif
#endif

typedef unsigned char grail_mask_t;

static inline void grail_mask_set(grail_mask_t *mask, int i)
{
	mask[i >> 3] |= (1 << (i & 7));
}

static inline void grail_mask_clear(grail_mask_t *mask, int i)
{
	mask[i >> 3] &= ~(1 << (i & 7));
}

static inline void grail_mask_modify(grail_mask_t *mask, int i, int v)
{
	if (v)
		grail_mask_set(mask, i);
	else
		grail_mask_clear(mask, i);
}

static inline int grail_mask_get(const grail_mask_t *mask, int i)
{
	return (mask[i >> 3] >> (i & 7)) & 1;
}

void GRAIL_PUBLIC grail_mask_set_mask(grail_mask_t *a, const grail_mask_t *b,
				      int bytes);
void GRAIL_PUBLIC grail_mask_clear_mask(grail_mask_t *a, const grail_mask_t *b,
					int bytes);

int GRAIL_PUBLIC grail_mask_count(const grail_mask_t *mask, int bytes);
int GRAIL_PUBLIC grail_mask_get_first(const grail_mask_t *mask, int bytes);
int GRAIL_PUBLIC grail_mask_get_next(int i, const grail_mask_t *mask,
				     int bytes);

#define grail_mask_foreach(i, mask, bytes)				\
	for (i = grail_mask_get_first(mask, bytes);			\
	     i >= 0;							\
	     i = grail_mask_get_next(i, mask, bytes))

#ifdef __cplusplus
}
#endif

#endif
