/* -*- Mode: C++ -*-
 * Worldvisions Weaver Software:
 *   Copyright (C) 1997-2002 Net Integration Technologies, Inc.
 * 
 * Support for monikers, which are strings that you can pass to a magic
 * factory to get objects supporting a particular interface, without actually
 * knowing anything about the constructor for those objects.
 */
#ifndef __WVMONIKER_H
#define __WVMONIKER_H

#include "wvxplc.h"
#include "wvstring.h"

class WvMonikerRegistry;

typedef IObject *WvMonikerCreateFunc(WvStringParm parms,
				     IObject *obj, void *userdata);

/**
 * WvMonikerBase is an auto-registration class for putting things into
 * a WvMonikerRegistry.  When a WvMonikerBase instance is created, it
 * registers a moniker prefix ("test:", "ssl:", "ini:", etc) and a factory
 * function that can be used to create an IObject using that prefix.
 * 
 * When the instance is destroyed, it auto-unregisters the moniker prefix
 * from the registry.
 * 
 * You can't actually create one of these, because it's not typesafe.  See
 * WvMoniker<T> instead.
 */
class WvMonikerBase
{
protected:
    WvMonikerBase(const XUUID &iid, WvStringParm _id,
		  WvMonikerCreateFunc *func);
    ~WvMonikerBase();
    
public:
    WvString id;
    WvMonikerRegistry *reg;
};


/**
 * A type-safe version of WvMonikerBase that lets you provide create functions
 * for object types other than IObject.  (The objects themselves have to
 * be derived from IObject, however.)
 * 
 * See WvMonikerBase for details.
 * 
 * Example:
 *    static IWvStream *createfunc(WvStringParm s, IObject *obj,
 *                                 void *userdata)
 *    {
 *        return new WvStream;
 *    }
 * 
 *    static WvMoniker<IWvStream> registration("ssl", createfunc);
 */
template <class T>
class WvMoniker : public WvMonikerBase
{
public:
    typedef T *CreateFunc(WvStringParm parms, IObject *obj, void *userdata);
    IObject *silly;
    
    WvMoniker(WvStringParm _id, CreateFunc *_func)
	: WvMonikerBase(XIID<T>::get(), _id, (WvMonikerCreateFunc *)_func)
    { 
	// this looks pointless, but it ensures that T* can be safely,
	// automatically downcast to IObject*.  That means T is really derived
	// from IObject, which is very important.
	silly = (T *)NULL;
    };
};


/**
 * Create an object registered in a WvMonikerRegistry.  The iid specifies
 * which registry to look in, and s, obj, and userdata are the parameters to
 * supply to the object's factory.  Most factories need only 's', which is the
 * moniker itself.
 * 
 * Most people don't use this function.  See the templated, type-safe version
 * of wvcreate() below.
 */
IObject *wvcreate(const XUUID &iid,
		  WvStringParm s, IObject *obj = NULL, void *userdata = NULL);


/**
 * Create an object registered in a WvMonikerRegistry.  Exactly which
 * registry is determined by the template type T.
 * 
 * s, obj, and userdata are the parameters to supply to the object's
 * factory.  Most factories need only 's', which is the moniker itself.
 * 
 * Example:
 *    IWvStream *s = wvcreate<IWvStream>("tcp:localhost:25");
 *    IWvStream *s_ssl = wvcreate<IWvStream>("ssl:", s);
 */
template <class T>
inline T *wvcreate(WvStringParm s, IObject *obj = NULL, void *userdata = NULL)
{
    return mutate<T>(wvcreate(XIID<T>::get(), s, obj, userdata));
}


#endif // __WVMONIKER_H
