/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.component;

import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.parameters.Parameterizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLoggable;
import org.apache.avalon.framework.logger.Loggable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.LogKitLogger;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.avalon.excalibur.pool.ObjectFactory;
import org.apache.avalon.excalibur.logger.LogKitManager;
import org.apache.avalon.excalibur.logger.LogKitManageable;

/**
 * Factory for Avalon components.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:paul@luminas.co.uk">Paul Russell</a>
 * @version CVS $Revision: 1.10 $ $Date: 2001/12/11 09:53:27 $
 * @since 4.0
 */
public class DefaultComponentFactory
    extends AbstractLoggable
    implements ObjectFactory, ThreadSafe
{
    /** The class which this <code>ComponentFactory</code>
     * should create.
     */
    private Class                   m_componentClass;

    /** The Context for the component
     */
    private Context                 m_context;

    /** The component manager for this component.
     */
    private ComponentManager        m_componentManager;

    /** The configuration for this component.
     */
    private Configuration           m_configuration;

    /** The RoleManager for child ComponentSelectors
     */
    private RoleManager             m_roles;

    /** The LogKitManager for child ComponentSelectors
     */
    private LogKitManager           m_logkit;

    /** The org.apache.avalon.framework.logger.Logger instance.
     */
    private org.apache.avalon.framework.logger.Logger m_logEnabledLogger;

    /**
     * Construct a new component factory for the specified component.
     *
     * @param componentClass the class to instantiate (must have a default constructor).
     * @param configuration the <code>Configuration</code> object to pass to new instances.
     * @param componentManager the component manager to pass to <code>Composable</code>s.
     * @param context the <code>Context</code> to pass to <code>Contexutalizable</code>s.
     * @param roles the <code>RoleManager</code> to pass to <code>DefaultComponentSelector</code>s.
     */
    public DefaultComponentFactory( final Class componentClass,
                                    final Configuration configuration,
                                    final ComponentManager componentManager,
                                    final Context context,
                                    final RoleManager roles,
                                    final LogKitManager logkit )
    {
        m_componentClass = componentClass;
        m_configuration = configuration;
        m_componentManager = componentManager;
        m_context = context;
        m_roles = roles;
        m_logkit = logkit;
    }

    public Object newInstance()
        throws Exception
    {
        final Object component = m_componentClass.newInstance();

        if (getLogger().isDebugEnabled())
        {
            getLogger().debug( "ComponentFactory creating new instance of " +
                               m_componentClass.getName() + "." );
        }

        if( component instanceof LogEnabled )
        {
            if( null == m_logkit || null == m_configuration )
            {
                ((LogEnabled)component).enableLogging( getLogEnabledLogger() );
            }
            else
            {
                final String logger = m_configuration.getAttribute( "logger", null );
                if( null == logger )
                {
                    getLogger().debug( "no logger attribute available, using standard logger" );
                    ((LogEnabled)component).enableLogging( getLogEnabledLogger() );
                }
                else
                {
                    getLogger().debug( "logger attribute is " + logger );
                    ((LogEnabled)component).enableLogging( new LogKitLogger( m_logkit.getLogger( logger ) ) );
                }
            }
        }

        if( component instanceof Loggable )
        {
            if( null == m_logkit || null == m_configuration )
            {
                ((Loggable)component).setLogger( getLogger() );
            }
            else
            {
                final String logger = m_configuration.getAttribute( "logger", null );
                if( null == logger )
                {
                    getLogger().debug( "no logger attribute available, using standard logger" );
                    ((Loggable)component).setLogger( getLogger() );
                }
                else
                {
                    getLogger().debug( "logger attribute is " + logger );
                    ((Loggable)component).setLogger( m_logkit.getLogger( logger ) );
                }
            }
        }

        if( component instanceof Contextualizable )
        {
            ((Contextualizable)component).contextualize( m_context );
        }

        if( component instanceof Composable )
        {
            ((Composable)component).compose( m_componentManager );
        }

        if ( component instanceof RoleManageable )
        {
            ((RoleManageable)component).setRoleManager( m_roles );
        }

        if ( component instanceof LogKitManageable )
        {
            ((LogKitManageable)component).setLogKitManager( m_logkit );
        }

        if( component instanceof Configurable )
        {
            ((Configurable)component).configure( m_configuration );
        }

        if( component instanceof Parameterizable )
        {
            ((Parameterizable)component).
                parameterize( Parameters.fromConfiguration( m_configuration ) );
        }

        if( component instanceof Initializable )
        {
            ((Initializable)component).initialize();
        }

        if( component instanceof Startable )
        {
            ((Startable)component).start();
        }

        return component;
    }

    public final Class getCreatedClass()
    {
        return m_componentClass;
    }

    public final void decommission( final Object component )
        throws Exception
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug( "ComponentFactory decommissioning instance of " +
                               m_componentClass.getName() + "." );
        }

        if( component instanceof Startable )
        {
            ((Startable)component).stop();
        }

        if( component instanceof Disposable )
        {
            ((Disposable)component).dispose();
        }
    }

    protected org.apache.avalon.framework.logger.Logger getLogEnabledLogger()
    {
        if ( null == m_logEnabledLogger )
        {
            m_logEnabledLogger = new LogKitLogger( getLogger() );
        }

        return m_logEnabledLogger;
    }
}
