/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.i18n;

import java.util.Locale;
import java.io.FileNotFoundException;

import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.ComponentSelector;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.activity.Initializable;

/**
 * Used to map locale information to URI space, to find the relevant bundle.
 *
 * @author <a href="mailto:neeme@apache.org">Neeme Praks</a>
 * @version CVS $Revision: 1.1 $ $Date: 2002/01/02 19:04:56 $ $Author: neeme $
 */

public class DefaultBundleFactory extends AbstractLogEnabled implements BundleFactory, Configurable, Composable, Initializable {

    public static final class ConfigurationKeys {
        public static final String BUNDLE = "bundle";
        public static final String MAPPER = "mapper";
        public static final String BUNDLE_CONF = "bundle-conf";
    }

    /** FQCN of the returned bundles */
    private String bundleClassName = null;

    /** Bundle configuration */
    private Configuration bundleConf = null;

    /** Mapper lookup hint **/
    private String mapperHint = null;

    /** Component Manager */
    protected ComponentManager manager = null;

    public void compose(ComponentManager manager) {
        this.manager = manager;
    }

    /**
     * Configure the component.
     *
     * @param configuration the configuration
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        if (bundleClassName == null) bundleClassName = configuration.getAttribute(ConfigurationKeys.BUNDLE);
        if (mapperHint == null) mapperHint = configuration.getAttribute(ConfigurationKeys.MAPPER);
        if (bundleConf == null) bundleConf = configuration.getChild(ConfigurationKeys.BUNDLE_CONF);

        if (getLogger().isDebugEnabled()) {
            getLogger().debug(
                "BundleLoader configured with bundle=" + bundleClassName +
                ", mapper=" + mapperHint +
            "");
        }
    }

    public void initialize() throws Exception {
        // TODO: implement the preloading of bundles
    }

    /**
     * Load a bundle, based on bundleInfo.
     *
     * @return      the bundle
     */
    public Bundle createInstance(BundleInfo bi) {
        Bundle bundle = null;
        Bundle parentBundle = null;
        BundleInfo parentBundleInfo = bi.getParent();
        ComponentSelector mapperSelector = null;
        BundleInfoMapper mapper = null;
        ComponentSelector bundleSelector = null;
        try {
            mapperSelector = (ComponentSelector) manager.lookup(BundleInfoMapper.ROLE + "Selector");
            mapper = (BundleInfoMapper) mapperSelector.select(mapperHint);
            bundleSelector = (ComponentSelector) manager.lookup(Bundle.ROLE + "Selector");
            if (getLogger().isDebugEnabled()) getLogger().debug("Loading bundle: " + bi);
            if (parentBundleInfo != null)
                parentBundle = (Bundle) bundleSelector.select(parentBundleInfo);
            bundle = getBundleInstance();
            if (bundle instanceof LogEnabled) ((LogEnabled)bundle).enableLogging(getLogger());
            if (bundle instanceof Configurable) ((Configurable)bundle).configure(bundleConf);
            bundle.setBundleInfo(bi);
            bundle.setMapper(mapper);
            bundle.setParent(parentBundle);
            if (bundle instanceof Composable) ((Composable)bundle).compose(this.manager);
            bundle.setLastModified(System.currentTimeMillis());
            if (bundle instanceof Initializable) ((Initializable)bundle).initialize();
        }
        catch (FileNotFoundException e) {
            getLogger().warn("File not found while loading bundle: " + bi);
            bundle = null;
        }
        catch (Exception e) {
            getLogger().error("Error while loading bundle: " + bi, e);
            bundle = null;
        }
        finally {
            if (mapperSelector != null) {
                if (mapper != null) mapperSelector.release((Component) mapper);
                manager.release(mapperSelector);
            }
            if (bundleSelector != null) manager.release(bundleSelector);
        }
        return bundle;
    }

    private Bundle getBundleInstance() {
        try {
            return (Bundle) Thread.currentThread().getContextClassLoader().loadClass(bundleClassName).newInstance();
        } catch (InstantiationException e) {
            getLogger().error("Could not create bundle instance!", e);
        } catch (IllegalAccessException e) {
            getLogger().error("Could not create bundle instance!", e);
        } catch (ClassNotFoundException e) {
            getLogger().error("Could not create bundle instance!", e);
        }
        return null;
    }
}
