/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.system;

import org.apache.avalon.framework.activity.*;
import org.apache.avalon.framework.context.*;
import org.apache.avalon.framework.component.*;
import org.apache.avalon.framework.configuration.*;
import org.apache.avalon.framework.parameters.*;
import org.apache.avalon.framework.logger.*;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.excalibur.logger.LogKitLoggerManager;
import org.apache.avalon.excalibur.util.ComponentStateValidator;
import org.apache.avalon.excalibur.source.*;
import org.apache.avalon.excalibur.command.*;
import org.apache.avalon.excalibur.pool.*;

import java.io.File;

/**
 * The ContainerManager is a single point of contact to manage your Container
 * resources.  It takes care of creating the other managers that a Container
 * needs to use, as well as initializing the Container.  It is designed to be
 * directly instantiated by whatever class needs to initialize your system.
 *
 * <p>
 *   The ContainerManager provides some constants used in the initial
 *   <code>Parameters</code> passed into the ContainerManager.  The ContainerManager
 *   uses these values to create all the pieces necessary for the Container.
 *   Below is a table that describes what those options are.
 * </p>
 *
 * <table>
 *   <tr>
 *     <th>Constant</th>
 *     <td>Description</th>
 *   </tr>
 *   <tr>
 *     <td><code>CONTAINER_CLASS</code></td>
 *     <td>
 *       <code>String</code> fully qualified class name for the container class.
 *       this is absolutely necessary for the ContainerManager to create the
 *       Container class instance.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>CONTEXT_DIRECTORY</code></td>
 *     <td>
 *       <code>String</code> path to the Context's root directory.  This is
 *       used to resolve all relative paths, and the "context:" psuedo-protocol.
 *       Defaults to "./".
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>WORK_DIRECTORY</code></td>
 *     <td>
 *       <code>String</code> path to a work area on the file system.  Defaults
 *       to "/tmp".
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>XML_PARSER</code></td>
 *     <td>
 *       <code>String</code> fully qualified class name of the Parser component
 *       implementation.  It defaults to
 *       "<code>org.apache.avalon.excalibur.xml.JaxpParser</code>".
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>LOG_CATEGORY</code></td>
 *     <td>
 *       <code>String</code> root category name for the container and its manager.
 *       If no value is given, the default (null) is used.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>LOGKIT_CONFIG</code></td>
 *     <td>
 *       <code>String</code> path to the LogKitLoggerManager configuration file.
 *       If this is not set, ContainerManager tries the uri
 *       <code>"context://logkit.xconf"</code>.  If nothing is there, the
 *       LogKitLoggerManager is not configured.  <strong>Note:</strong> If you
 *       passed in a reference to a LoggerManager yourself, ContainerManager
 *       ignores this parameter altogether.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>ROLE_CONFIG</code></td>
 *     <td>
 *       <code>String</code> path to the RoleManager configuration file.  If
 *       this is not set, ContainerManager does not create a RoleManager at all.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>CONTAINER_CONFIG</code></td>
 *     <td>
 *       <code>String</code> path the points to the container's Configuration
 *       file.  This value defaults to the uri "context://container.xconf".
 *       If the Container is Configurable and no entry can be found, then the
 *       Container is given an empty Configuration object.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>THREADS_CPU</code></td>
 *     <td>
 *       <code>Integer</code> that holds the value of the number of threads used
 *       by the ThreadManager per CPU.  This defaults to 2.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td><code>CPU_COUNT</code></td>
 *     <td>
 *       <code>Integer</code> representing the number of CPUs on a machine.  The
 *       value is used in efficiency calculations, and determining the number of
 *       threads to have running for the CommandManager.  Defaults to <code>1</code>.
 *     </td>
 *   </tr>
 * </table>
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.10 $ $Date: 2002/01/28 19:29:47 $
 */
public class ContainerManager
{
    public static final String CONTEXT_DIRECTORY = Container.CONTEXT_DIRECTORY;
    public static final String WORK_DIRECTORY    = Container.WORK_DIRECTORY;
    public static final String CONTAINER_CLASS   = "container.class";
    public static final String XML_PARSER        = "container.xmlParser";
    public static final String LOGKIT_CONFIG     = "container.loggerConfig";
    public static final String ROLE_CONFIG       = "container.roles";
    public static final String THREADS_CPU       = "container.threadsPerCPU";
    public static final String CONTAINER_CONFIG  = "container.configFile";
    public static final String LOG_CATEGORY      = "container.logCategory";
    public static final String CPU_COUNT         = "os.arch.cpus";

    private static final Configuration EMPTY_CONFIG;

    static
    {
        DefaultConfiguration config = new DefaultConfiguration("", "", "", "");
        config.makeReadOnly();
        EMPTY_CONFIG = config;
    }

    private final DefaultConfigurationBuilder    m_configBuilder =
                                                     new DefaultConfigurationBuilder();
    private final DefaultConfigurationSerializer m_configSerialzer =
                                                     new DefaultConfigurationSerializer();

    private final Parameters              m_initialParameters;
    private final ClassLoader             m_contextClassLoader;
    private final File                    m_contextDirectory;
    private final File                    m_workDirectory;
    private       LoggerManager           m_logManager;
    private final PoolManager             m_poolManager;
    private final ThreadManager           m_threadManager;
    private final CommandManager          m_commandManager;
    private       ComponentManager        m_componentManager;
    private       Configuration           m_containerConfig;
    private       Configuration           m_logKitConfig;
    private       Configuration           m_roleConfig;
    private       Context                 m_containerContext;
    private       Container               m_containerInstance;
    private       ComponentStateValidator m_validator;
    private       RoleManager             m_roleManager;

    /**
     * This constructor creates a new ContainerManager with a LogKitLoggerManager
     * for managing your logging categories.  If you want another LoggerManager
     * implementation, then use the constructor with two arguments.
     */
    public ContainerManager( final Parameters initialParameters )
        throws InitializationException
    {
        this( initialParameters, null );
    }

    /**
     * Create a ContainerManager with all the information in the Context, and the
     * supplied LoggerManager.
     */
    public ContainerManager( final Parameters initialParameters, final LoggerManager defaultLogManager )
        throws InitializationException
    {
        this( initialParameters, defaultLogManager, Thread.currentThread().getContextClassLoader() );
    }

    /**
     * Create a ContainerManager with all the information in the Context, and the
     * supplied LoggerManager.
     */
    public ContainerManager( final Parameters initialParameters,
                             final LoggerManager defaultLogManager,
                             final ClassLoader rootClassLoader )
        throws InitializationException
    {
        m_initialParameters = initialParameters;
        m_contextClassLoader = rootClassLoader;
        m_contextDirectory = new File(initialParameters.getParameter( CONTEXT_DIRECTORY, "./" ) );
        m_workDirectory = new File( initialParameters.getParameter( CONTEXT_DIRECTORY, "/tmp" ) );
        m_commandManager = new CommandManager();
        m_threadManager = new TPCThreadManager( initialParameters );
        m_threadManager.register( m_commandManager );
        m_poolManager = new DefaultPoolManager( m_commandManager.getCommandQueue() );

        if ( null != defaultLogManager )
        {
            m_logManager = defaultLogManager;
        }

        recycleContainer();
    }

    /**
     * Get a reference to your Container.  Typically, you would cast this to
     * whatever interface you will use to interact with it.  The actual
     * initialization process
     */
    Container getContainer()
    {
        return m_containerInstance;
    }

    /**
     * Get a reference to the initial ComponentManager used by the ContainerManager
     * to hold the Components used for parsing the config files and setting up the
     * environment.
     */
    public ComponentManager getComponentManager()
    {
        if ( null == m_componentManager )
        {
            DefaultComponentManager manager = new DefaultComponentManager();

            try
            {
                SourceResolverImpl resolver = new SourceResolverImpl();
                resolver.enableLogging( getLogger() );
                resolver.contextualize( getContext() );
                resolver.compose( manager );
                manager.put( resolver.ROLE, resolver );

                DefaultComponentSelector selector = new DefaultComponentSelector();
                ResourceSourceFactory resource = new ResourceSourceFactory();
                resource.enableLogging( getLogger() );
                selector.put("resource", resource);

                manager.put( resource.ROLE + "Selector", selector );
            }
            catch ( Exception e )
            {
                getLogger().warn("Could not set up the initial components", e);
            }

            manager.makeReadOnly();
            m_componentManager = manager;
        }

        return m_componentManager;
    }

    /**
     * Override this if you have any special needs for the container (such as
     * wanting to use your own class).
     */
    protected void recycleContainer()
        throws InitializationException
    {
        if ( null != m_containerInstance )
        {
            if ( m_containerInstance instanceof Startable )
            {
                try
                {
                    ( (Startable) m_containerInstance ).stop();
                }
                catch (Exception e)
                {
                    if ( getLogger().isWarnEnabled() )
                    {
                        getLogger().warn("Caught an exception when stopping the Container, continuing with shutdown", e);
                    }
                }
            }

            if ( m_containerInstance instanceof Disposable )
            {
                ( (Disposable) m_containerInstance ).dispose();
            }

            m_containerInstance = null;
        }

        Container instance = null;
        try
        {
            instance = (Container) m_contextClassLoader
                    .loadClass( m_initialParameters.getParameter( CONTAINER_CLASS ) )
                    .newInstance();
        }
        catch ( Exception e )
        {
            instance = null;
            if ( getLogger().isFatalErrorEnabled() )
            {
                getLogger().fatalError( "Cannot set up the Container, this is an error I cannot recover from.", e );
            }
            return;
        }

        try
        {
            if ( instance instanceof Contextualizable )
            {
                ( (Contextualizable) instance ).contextualize( getContext() );
            }

            if ( instance instanceof LogEnabled )
            {
                ( (LogEnabled) instance ).enableLogging( getLogger() );
            }

            if ( instance instanceof Composable )
            {
                ( (Composable) instance ).compose( getComponentManager() );
            }

            if ( instance instanceof Configurable )
            {
                ( (Configurable) instance ).configure( getContainerConfig() );
            }

            if ( instance instanceof Parameterizable )
            {
                ( (Parameterizable) instance ).parameterize( Parameters.fromConfiguration( getContainerConfig() ) );
            }

            if ( instance instanceof Initializable )
            {
                ( (Initializable) instance ).initialize();
            }

            if ( instance instanceof Startable )
            {
                ( (Startable) instance ).start();
            }
        }
        catch ( Exception e )
        {
            instance = null;
            if ( getLogger().isFatalErrorEnabled() )
            {
                getLogger().fatalError( "Cannot set up the Container, this is an error I cannot recover from.", e );
            }
        }

        m_containerInstance = instance;
    }

    /**
     * Override this if you have any special needs for the Container's Context.
     * Typically, this will f
     */
    protected Context getContext()
    {
        if ( null == m_containerContext )
        {
            DefaultContext context = new DefaultContext();
            context.put( CONTEXT_DIRECTORY, m_contextDirectory );
            context.put( WORK_DIRECTORY, m_workDirectory );
            context.put( CPU_COUNT,
                    new Integer(m_initialParameters.getParameterAsInteger( CPU_COUNT, 1 ) )
            );
            context.put( LOG_CATEGORY,
                    m_initialParameters.getParameter(LOG_CATEGORY, null)
            );
            context.put( Container.CONTEXT_CLASSLOADER, m_contextClassLoader );
            context.put( Container.ROLE_MANAGER, getRoleManager() );

            context.put( Container.COMMAND_QUEUE, m_commandManager.getCommandQueue() );
            context.put( Container.POOL_MANAGER, m_poolManager );

            context.makeReadOnly();
            m_containerContext = context;
        }

        return m_containerContext;
    }

    /**
     * Get the LoggerManager. Will set up a LogKitLoggerManager if none is
     * supplied.  This can be overridden if you don't want a LogKitLoggerManager.
     */
    public LoggerManager getLoggerManager()
    {
        if ( null == m_logManager )
        {
            LogKitLoggerManager logManager = new LogKitLoggerManager(
                    m_initialParameters.getParameter(LOG_CATEGORY, null)
            );

            try
            {
                logManager.contextualize( getContext() );
                logManager.configure( getLogKitConfig() );
            }
            catch (Exception e)
            {
                getLogger().warn("Could not completely set up LogKitLoggerManager", e);
            }

            m_logManager = logManager;
        }

        return m_logManager;
    }

    /**
     * Get a reference the default Logger.
     */
    public Logger getLogger()
    {
        return getLoggerManager().getDefaultLogger();
    }

    /**
     * Sets up the RoleManager.  You may override this method to implement your
     * own logic.  This is the RoleManager given to the Container so that the
     * Container can interpret it's Configuration.
     */
    protected RoleManager getRoleManager()
    {
        if ( null == m_roleManager )
        {
            if ( null == m_roleConfig )
            {
                m_roleManager = new ExcaliburRoleManager();
            }
            else
            {
                ExcaliburRoleManager erm = new ExcaliburRoleManager();
                ConfigurableRoleManager crm = new ConfigurableRoleManager( erm );

                try
                {
                    crm.configure( getRoleConfig() );
                    m_roleManager = crm;
                }
                catch ( Exception e )
                {
                    getLogger().warn("There was a problem with the role configuration, defaulting to ExcaliburComponentManager.", e);
                    m_roleManager = erm;
                }
            }
        }

        return m_roleManager;
    }

    /**
     * Get the Container Configuration hierarchy.
     */
    protected Configuration getContainerConfig()
    {
        if ( null == m_containerConfig )
        {
            String configFile = m_initialParameters.getParameter( CONTAINER_CONFIG, "" );

            try
            {
                m_containerConfig = m_configBuilder.buildFromFile( configFile );
            }
            catch (Exception e)
            {
                if ( getLogger().isWarnEnabled() )
                {
                    getLogger().warn("Could not read configuration file: " + configFile, e);
                    m_containerConfig = EMPTY_CONFIG;
                }
            }
        }

        return m_containerConfig;
    }

    /**
     * Get the Container Configuration hierarchy.
     */
    protected Configuration getRoleConfig()
    {
        if ( null == m_containerConfig )
        {
            String configFile = m_initialParameters.getParameter( ROLE_CONFIG, "" );

            try
            {
                m_roleConfig = m_configBuilder.buildFromFile( configFile );
            }
            catch (Exception e)
            {
                if ( getLogger().isWarnEnabled() )
                {
                    getLogger().warn("Could not read configuration file: " + configFile, e);
                    m_roleConfig = EMPTY_CONFIG;
                }
            }
        }

        return m_roleConfig;
    }

    /**
     * Get the Container Configuration hierarchy.
     */
    protected Configuration getLogKitConfig()
    {
        if ( null == m_containerConfig )
        {
            String configFile = m_initialParameters.getParameter( LOGKIT_CONFIG, "" );

            try
            {
                m_logKitConfig = m_configBuilder.buildFromFile( configFile );
            }
            catch (Exception e)
            {
                if ( getLogger().isWarnEnabled() )
                {
                    getLogger().warn("Could not read configuration file: " + configFile, e);
                    m_logKitConfig = EMPTY_CONFIG;
                }
            }
        }

        return m_logKitConfig;
    }
}
