//
// File:        IOR.java
// Package:     gov.llnl.babel.backend
// Revision:    @(#) $Id: IOR.java 4462 2005-03-23 19:29:24Z leek2 $
// Description: common sidl to IOR routines shared by code generators
//
// Copyright (c) 2000-2004, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend;

import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import gov.llnl.babel.BabelConfiguration;
import java.util.Iterator;
import java.util.List;
import java.util.Collections;
import java.util.Comparator;

/**
 * Class <code>IOR</code> contains common SIDL to IOR translation
 * routines shared by the backend code generators.  This class simply
 * collects many common IOR language bindings into one place.
 */
public class IOR {
  public final static int MAJOR_VERSION = 0;
  public final static int MINOR_VERSION = 9;

   private final static String s_types[] = {
      "void",
      "sidl_bool",
      "char",
      "struct sidl_dcomplex",
      "double",
      "struct sidl_fcomplex",
      "float",
      "int32_t",
      "int64_t",
      "void*",
      "char*"
   };

   private final static String s_array_types[] = {
      null,
      "struct sidl_bool__array*",
      "struct sidl_char__array*",
      "struct sidl_dcomplex__array*",
      "struct sidl_double__array*",
      "struct sidl_fcomplex__array*",
      "struct sidl_float__array*",
      "struct sidl_int__array*",
      "struct sidl_long__array*",
      "struct sidl_opaque__array*",
      "struct sidl_string__array*"
   };

  private final static String [] s_builtinMethods = {
    "_cast",                    // the CAST method
    "_delete",                  // the DELETE method
    "_exec",                    // the reflexive EXEC method
    "_ctor",                    // the CONSTRUCTOR method
    "_dtor",                    // the DESTRUCTOR method
    "_load",                    // the LOAD method
    "_checks",                  // the CHECKS method
    "_iceptors",		// the INTERCEPTORS method
    "_check_error",             // the CHECK_ERROR method
    "_dump_stats",              // the DUMP_STATS method
  };

  private final static String[] s_builtin_comments = {
    "Cast method for interface and class type conversions.",
    "Delete method called automatically by IOR to destroy object.",
    "Select and execute a method by name",
    "Class constructor called when the class is created.",
    "Class destructor called when the class is deleted.",
    "Static class initializer called exactly once before any user-defined method is dispatched",
    "Method to set the level of checking SIDL-specified assertions.",
    "Method to set whether or not method interceptors should be invoked.",
    "Method to handle assertion violations.",
    "Method to dump assertion checking statistics.",
  };

  /**
   * The number of builtin methods that an interface has. Builtin methods
   * are implicitly defined methods that are required for the inner
   * workings of the IOR or to support the language bindings. The names
   * of the builtins are numbers
   * <code>0...INTERFACE_BUILTIN_METHODS-1</code> and are available from
   * the method <code>getBuiltinMethod</code>.
   *
   * @see #getBuiltinMethod
   */
  public static final int INTERFACE_BUILTIN_METHODS = 3;

  /**
   * The number of builtin methods that a class has. Builtin methods are
   * implicitly defined methods that are required for the inner
   * workings of the IOR or to support the language bindings. The names
   * of the builtins are numbers
   * <code>0...CLASS_BUILTIN_METHODS-1</code> and are available from
   * the method <code>getBuiltinMethod</code>.
   */
  public static final int CLASS_BUILTIN_METHODS = 10;

  /**
   * The index of the builtin method for casting.
   */
  public static final int CAST = 0;

  /**
   * The index of the builtin method for deleting an object.
   */
  public static final int DELETE = 1;

  /**
   * The index of the builtin method for executing a named method.
   */
  public static final int EXEC = 2;

  /**
   * The index of the builtin method for constructing a
   * class instance
   */
  public static final int CONSTRUCTOR = 3;

  /**
   * The index of the builtin method for destructing a
   * class instance
   */
  public static final int DESTRUCTOR = 4;

  /**
   * The index of the builtin method for initializing a class
   * (before first instance, or static method is called).
   */
  public static final int LOAD = 5;

  /**
   * The index of the builtin method for setting the assertion checking.
   */
  public static final int CHECKS = 6;

  /**
   * The index of the builtin method for enabling/disabling interceptor
   * execution.
   */
  public static final int INTERCEPTORS = 7;

  /**
   * The index of the builtin method for assertion checking 
   * implementation.
   */
  public static final int CHECK_ERROR = 8;

  /**
   * The index of the builtin method for dumping assertion checking
   * data.
   */
  public static final int DUMP_STATS = 9;

  private static final int BUILTIN_MIN = 0;
  private static final int BUILTIN_MAX = 9;

  private static final String s_default_version    = MAJOR_VERSION + "." 
                                                     + MINOR_VERSION + ".0";

  private static SymbolID s_exceptionID            = null;
  private static String s_exceptionFundamentalType = null;
  private static String s_objectType               = null;
  private static String s_interfaceType            = null;

  public static String FUND_EXCEPTION_CALL_PREFIX  = 
                        BabelConfiguration.FUND_EXCEPTION.replace('.', '_');
  public static String PRECONDITION_CALL_PREFIX    = 
                        BabelConfiguration.PRE_EXCEPTION.replace('.', '_');
  public static String POSTCONDITION_CALL_PREFIX   = 
                        BabelConfiguration.POST_EXCEPTION.replace('.', '_');
  public static String INVARIANT_CALL_PREFIX       = 
                        BabelConfiguration.INV_EXCEPTION.replace('.', '_');

  static {
    s_exceptionID = new SymbolID(BabelConfiguration.getBaseExceptionType(), 
                                 new Version(s_default_version));
    s_exceptionFundamentalType = getObjectName(s_exceptionID) + " *";
    SymbolID id = new SymbolID(BabelConfiguration.getBaseClass(),
                      new Version());
    s_objectType = getObjectName(id) + " *";
    id = new SymbolID(BabelConfiguration.getBaseInterface(),
                      new Version());
    s_interfaceType = getObjectName(id) + " *";
  }

  /*
   * Now for some assertion and interceptor EPV basics.  In the interest of
   * quickly/efficiently switching at run-time between wrapped, or decorated,
   * server-side methods, support for several additional EPV structures _may_
   * be generated in the IOR, if necessary.
   */
   public final static boolean SUPPORT_ASSERTIONS =
     BabelConfiguration.getInstance().generateAssertions();
   public final static String  ASSERTION_LEVEL    =
     BabelConfiguration.getInstance().getAssertionLevel();
   public final static boolean SUPPORT_INTERCEPTORS =
     BabelConfiguration.getInstance().generateInterceptors();

   private final static String[] s_epv_type = { "", "b"};
   public  final static int PUBLIC_EPV      = 0;
   public  final static int BASE_EPV        = 1;
   private final static int MIN_EPV_TYPE    = 0;
   private final static int MAX_EPV_TYPE    = 1;

   public  final static int    SET_PUBLIC       = 0;
   public  final static int    SET_ASSERTIONS   = 1;
   public  final static int    SET_INTERCEPTORS = 2;
   private final static int    SET_MINIMUM      = 0;
   private final static int    SET_MAXIMUM      = 2;
   private final static String s_SET_EPV_NAMES[] = {
     "",
     "assertions",
     "interceptors"
   };

   public final static int   EPV_NEW     = 0;
   public final static int   EPV_OLD     = 1;
   public final static int   EPV_REMOTE  = 2;
   public final static int   EPV_STATIC  = 3;
   private final static int   EPV_MINIMUM = 0;
   private final static int   EPV_MAXIMUM = 3;
   private final static String s_EPV_DESC[] = {
     "new",
     "old",
     "rem",
     "stc",
   };

   public final static String GENERIC_PRE_SUFFIX    = "_pre";
   public final static String GENERIC_POST_SUFFIX   = "_post";
   public final static int    GENERIC_SUFFIX_MAXLEN = 5;

   public final static String D_CALLS               = "calls";
   public final static String D_CHECKS              = "checking_level";
   public final static String D_CONTROLS            = "d_controls";
   public final static String D_COUNTDOWN           = "countdown";
   public final static String D_DATA                = "d_data";
   public final static String D_INTERCEPTORS        = "use_iceptors";
   public final static String D_METHOD_CONTROLS     = "method_controls";
   public final static String D_METHOD_EXCEPT       = "method_exceptions";
   public final static String D_SUCCESSES           = "successes";
   public final static String D_RATE                = "check_rate";
   public final static String D_TRIES               = "tries";

   public final static String S_CONTROLS            = "s_control_stats";
   public final static String S_DUMP_FILE           = "s_dump_fptr";

   /**
    * Some assertion checking defaults...at least until we provide another
    * mechanism.
    */
   public final static String DEFAULT_CHECK_LEVEL         =
                                 "CHECK_ALL_TYPES | CHECK_ALWAYS";
   public final static String DEFAULT_OPTION_INTERCEPTORS = "0";
   public final static String DEFAULT_THRESHOLD           = "0.03";

   /*
    * Note: the following must match the macro used in sidlAsserts.h
    */
   public final static String S_FULL_STATS_MACRO    = "SIDL_FULL_ASTATS";
   public final static String S_TEXT_STATS_MACRO    = "SIDL_TEXT_ASTATS";

   public final static String s_static_suffix       = "_static";

   /*------------------------------------------------------------------*/

    /** A CPP Macro Name */
    public static String getLockStaticGlobalsMacroName() { 
	return "LOCK_STATIC_GLOBALS";
    }
    /** A CPP macro name */
    public static String getUnlockStaticGlobalsMacroName() { 
	return "UNLOCK_STATIC_GLOBALS";
    }
    /** A CPP macro name */
    public static String getHaveLockStaticGlobalsMacroName() { 
	return "HAVE_LOCKED_STATIC_GLOBALS";
    }

  /**
   * Return the method description of a particular builtin method. This will
   * raise an <code>ArrayIndexOutOfBoundsException</code> if
   * <code>index</code> is less than zero or greater than or equal to the
   * number of builtin methods.
   *
   * @param index       the index of the builtin method that is
   *                    desired. Generally, one of <code>CAST</code>,
   *                    <code>DELETE</code>, <code>CONSTRUCTOR</code>,
   *                    or <code>DESTRUCTOR</code> though others possible.
   * @param id          the name of the symbol
   * @param sVersion    TRUE if the static version is desired; else FALSE
   * @return a description the method.
   * @exception java.lang.ArrayIndexOutOfBoundsException
   *    this runtime exception is thrown if <code>index</code> is out of
   *    bounds.
   */
  public static Method getBuiltinMethod(int      index,
                                        SymbolID id,
                                        boolean  sVersion)
    throws CodeGenerationException
  {
    Method m = new Method();
    m.setMethodName(getBuiltinName(index));
    if ((index==LOAD) || (sVersion)) { 
	m.setDefinitionModifier(Method.STATIC);
    } else { 
	m.setDefinitionModifier(Method.NORMAL);
    }

    String[] s = new String[1];
    if (sVersion) {
      s[0] = "Static " + s_builtin_comments[index];
    } else {
      s[0] = s_builtin_comments[index];
    }
    m.setComment(new Comment(s));

    Argument a = null;
    switch (index) {
       case CAST: 
          a = new Argument(false, Argument.IN, new Type(Type.STRING), "name");
          m.addArgument(a);
          m.setReturnType(new Type(Type.OPAQUE));
          break;
       case EXEC:
          a = new Argument(false, Argument.IN, new Type(Type.STRING), 
                           "methodName");
          m.addArgument(a);
          Symbol tmpSym = Utilities.lookupSymbol("sidl.io.Deserializer");
          a = new Argument(false, Argument.IN, new Type(tmpSym.getSymbolID()), 
                           "inArgs");
          m.addArgument(a);
          tmpSym = Utilities.lookupSymbol("sidl.io.Serializer");
          a = new Argument(false, Argument.IN, new Type(tmpSym.getSymbolID()), 
                           "outArgs");
          m.addArgument(a);
          m.setReturnType(new Type(Type.VOID));
          break;
       case CHECKS:
          a = new Argument(false, Argument.IN, new Type(Type.INT), "level");
          m.addArgument(a);
          a = new Argument(false, Argument.IN, new Type(Type.DOUBLE), 
                           "rate");
          m.addArgument(a);
          a = new Argument(false, Argument.IN, new Type(Type.INT), 
                           "resetCounters");
          m.addArgument(a);
          m.setReturnType(new Type(Type.VOID));
          break;
       case INTERCEPTORS:
          a = new Argument(false, Argument.IN, new Type(Type.INT), "on");
          m.addArgument(a);
          m.setReturnType(new Type(Type.VOID));
          break;
       case CHECK_ERROR:
          a = new Argument(false, Argument.IN, new Type(Type.STRING), "msg");
          m.addArgument(a);
          m.setReturnType(new Type(Type.VOID));
          break;
       case DUMP_STATS:
          a = new Argument(false, Argument.IN, new Type(Type.STRING), 
                           "filename");
          m.addArgument(a);
          m.setReturnType(new Type(Type.VOID));
          break;
       default:
          m.setReturnType(new Type(Type.VOID));
          break;
    }

    return m;
  }

  /**
   * Return the method description of a particular non-static builtin method. 
   * This will raise an <code>ArrayIndexOutOfBoundsException</code> if
   * <code>index</code> is less than zero or greater than or equal to the
   * number of builtin methods.
   *
   * @param index       the index of the desired builtin method.
   * @param id          the name of the symbol
   * @return a description the method.
   * @exception java.lang.ArrayIndexOutOfBoundsException
   *    this runtime exception is thrown if <code>index</code> is out of
   *    bounds.
   */
  public static Method getBuiltinMethod(int index, SymbolID id) 
    throws CodeGenerationException
  {
    return getBuiltinMethod(index, id, false);
  }

  /**
   * Return TRUE if there is a builtin static version of the method; FALSE 
   * otherwise.
   *
   * @param index       the index of the desired builtin method.
   */
  public static boolean hasStaticBuiltin(int index) {
    boolean has = false;
    if (  (index == CHECKS)      || (index == INTERCEPTORS)
       || (index == CHECK_ERROR) || (index == DUMP_STATS) ) {
      has = true;
    }
    return has;
  }

  /**
   * Return the name of the specified version of the builtin method.
   *
   * @param index       the index of the builtin method that is
   *                    desired. Generally, one of <code>CAST</code>,
   *                    <code>DELETE</code>, <code>CONSTRUCTOR</code>,
   *                    or <code>DESTRUCTOR</code> though others possible.
   * @param sVersion    TRUE if the static version is desired; FALSE otherwise.
   * @exception java.lang.ArrayIndexOutOfBoundsException
   *    this runtime exception is thrown if <code>index</code> is out of
   *    bounds.
   */
  public static String getBuiltinName(int index, boolean sVersion) {
    return s_builtinMethods[index] + (sVersion ? s_static_suffix : "");
  }

  /**
   * Return the normal name of the builtin method.
   *
   * @param index       the index of the desired builtin method.
   * @exception java.lang.ArrayIndexOutOfBoundsException
   *    this runtime exception is thrown if <code>index</code> is out of
   *    bounds.
   */
  public static String getBuiltinName(int index) {
    return getBuiltinName(index, false);
  }

  /**
   * Return TRUE if the method name is one of the built-in methods, FALSE
   * otherwise.
   *
   * @param methodName  the name of the method being checked
   * @param sVersion    TRUE if the static version is desired; FALSE otherwise.
   */
  public static boolean isBuiltinMethod(String methodName, boolean sVersion) {
    boolean isBuiltin = false;
    for (int i=BUILTIN_MIN; (i<=BUILTIN_MAX) && !isBuiltin; i++) {
      if (  hasStaticBuiltin(i) 
         && methodName.equals(getBuiltinName(i, sVersion))  ) {
        isBuiltin = true;
      }
    }
    return isBuiltin;
  }

  /**
   * Return TRUE if the method name is one of the non-static built-in methods, 
   * FALSE otherwise.
   *
   * @param methodName  the name of the method being checked
   */
  public static boolean isBuiltinMethod(String methodName) {
    boolean isBuiltin = false;
    for (int i=BUILTIN_MIN; (i<=BUILTIN_MAX) && !isBuiltin; i++) {
      if (methodName.equals(getBuiltinName(i, false))) {
        isBuiltin = true;
      }
    }
    return isBuiltin;
  }

  /**
   * Return TRUE if the index is associated with a basic built-in method,
   * FALSE otherwise.
   */
  public static boolean isBuiltinBasic(int ind) {
    boolean is = false;
    if ( (CAST <= ind) && (ind <= DESTRUCTOR) ) {
      is = true;
    }
    return is;
  }

  /**
   * Return TRUE if the index is associated with an assertion related built-in 
   * method, FALSE otherwise.
   */
  public static boolean isBuiltinAssert(int ind) {
    boolean is = false;
    if (  (ind == CHECKS) || (ind == CHECK_ERROR) || (ind == DUMP_STATS) ) {
      is = true;
    }
    return is;
  }

  /**
   * Generate the name of an entry in the entry point vector or the
   * static entry point vector.
   *
   * @param methodName  the name of the method that is an element
   *                    in the entry point vector.
   */
  public static String getVectorEntry(String methodName) {
    StringBuffer buf = new StringBuffer(2 + methodName.length());
    buf.append("f_").append(methodName);
    return buf.toString();
  }

   /**
    * Generate the header filename associated with a symbol identifier.
    * Replace the "." scope separators in the symbol by underscores and
    * append the suffix "_IOR.h".
    */
   public static String getHeaderFile(SymbolID id) {
      return id.getFullName().replace('.', '_') + "_IOR.h";
   }

   /**
    * Generate the source filename associated with a symbol identifier.
    * Replace the "." scope separators in the symbol by underscores and
    * append the suffix "_IOR.c".
    */
   public static String getSourceFile(SymbolID id) {
      return id.getFullName().replace('.', '_') + "_IOR.c";
   }

   /**
    * Convert a symbol name into an IOR identifier.  This method replaces
    * the "." scope separators in the symbol by underscores.
    */
   public static String getSymbolName(SymbolID id) {
      return id.getFullName().replace('.', '_');
   }

   /**
    * Convert a SIDL enumerated type into its symbol name, which is
    * "enum " followed by the symbol name followed by "__enum".
    */
   public static String getEnumName(SymbolID id) {
      return C.getEnumName(id);
/*
 * ToDo...Clean this up if it works.
 *
      return "enum " + getSymbolName(id) + "__enum";
*/
   }

  /**
   * Get struct name for extern entry point structure.
   */
  public static String getExternalName(SymbolID id)
  {
    return "struct " + getSymbolName(id) + "__external";
  }

  /**
   * Get the name of the function that returns the structure of
   * external entry points.
   */
  public static String getExternalFunc(SymbolID id)
  {
    return getSymbolName(id) + "__externals";
  }

   /**
    * Convert a SIDL interface or class into its symbol name, which is
    * "struct " followed by the symbol name followed by "__object".
    */
   public static String getObjectName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__object";
   }

   /**
    * Convert a SIDL symbol into its array representation, which is
    * "struct " followed by the symbol name followed by "__array".
    * Passing <code>null</code> to this function causes it to 
    * return the generic (typeless) SIDL array.
    */
   public static String getArrayName(SymbolID id) {
      return (null != id) 
        ? ("struct " + getSymbolName(id) + "__array")
        : "struct sidl__array";
   }

  /** 
   * Get the sidl array name for a given type.  Use the arrayType from the array.
   */
  public static String getArrayName(int intType) {
    return s_array_types[intType];
  }

  /** 
   * Gets the sidl array name for a given type, and removes the trailing
   * asterix.  Use the arrayType from the array.
   */
  public static String getArrayNameWithoutAsterix(int intType) {
    return s_array_types[intType].substring(0,s_array_types[intType].length()-1);
  }

  /** 
   * Gets the sidl array name for a given type, for use in C functions.
   */
  public static String getArrayNameForFunctions(int intType) {
    return s_array_types[intType].substring(7,s_array_types[intType].length()-1);
  }

   /**
    * Return TRUE if the extendable is a SIDL symbol; FALSE otherwise.
    */
   public static boolean isSIDLSymbol(SymbolID id) {
      return id.getFullName().toUpperCase().startsWith("sidl");
   }

   /**
    * Return TRUE if the extendable is a SIDL symbol; FALSE otherwise.
    */
   public static boolean isSIDLSymbol(Extendable ext) {
      return isSIDLSymbol(ext.getSymbolID());
   }

   /**
    * Return TRUE if assertion checking needs to be supported; FALSE otherwise.
    */
   public static boolean supportAssertions(Extendable ext) {
     return SUPPORT_ASSERTIONS && ext.hasAssertions() && (!isSIDLSymbol(ext));
   }

   /**
    * Return TRUE if interceptors need to be supported; FALSE otherwise.
    */
   public static boolean supportInterceptors(Extendable ext) {
     return SUPPORT_INTERCEPTORS && (!isSIDLSymbol(ext));
   }

   /**
    * Return TRUE if interceptors need to be supported; FALSE otherwise.
    */
   public static boolean supportInterceptors(SymbolID id) {
     return SUPPORT_INTERCEPTORS && (!isSIDLSymbol(id));
   }

   /**
    * Return TRUE if the base EPV attribute needs to be supported; FALSE 
    * otherwise.
    */
   public static boolean supportBaseEPVAttr(Extendable ext) {
     return supportInterceptors(ext) && supportAssertions(ext);
   }

   /**
    * Return the name of the method description data structure name.
    */
  public static String getMethodDescDataStruct(SymbolID id) {
    return "struct " + getSymbolName(id) + "__method";
  }

   /**
    * Return the name of the static variable associated with the method
    * description data.
    */
  public static String getMethodDescDataName(SymbolID id) {
    return "s_ior_" + getSymbolName(id) + "_method";
  }

   /**
    * Convert a SIDL symbol into its control structure.
    */
   public static String getControlsStruct(SymbolID id) {
      return "struct " + getSymbolName(id) + "__controls";
   }

   /**
    * Convert a SIDL symbol into its method control structure.
    */
   public static String getMethodControlsStruct(SymbolID id) {
      return "struct " + getSymbolName(id) + "__method_controls";
   }

   /**
    * Convert a SIDL symbol into its method entry point vector (EPV) name.  
    */
   public static String getEPVName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__epv";
   }

   /**
    * Convert a SIDL symbol into its static entry point vector (SEPV) name.
    */
   public static String getSEPVName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__sepv";
   }

   /**
    * Return the type associated with the specified EPV type index,
    * or an empty string if the index is out of range.
    */
   public static String getEPVType(int type) {
      String ret = "";
      if ( (MIN_EPV_TYPE <= type) && (type <= MAX_EPV_TYPE) ) {
        ret = s_epv_type[type];
      }
      return ret;
   }

   /**
    * Return the standard method entry point vector (EPV) variable.  
    */
   public static String getEPVVar(int type) {
      String etype = getEPVType(type);
      return (etype.equals("")) ? "d_epv" : "d_" + etype + "epv";
   }

   /**
    * Returns the name of the set EPV type.
    */
   public static String getSetEPVTypeName(int type) {
     int t = SET_PUBLIC;
     if ( (SET_MINIMUM <= type) && (type <= SET_MAXIMUM) ) {
       t = type;
     }
     return s_SET_EPV_NAMES[t];
   }

   /**
    * Convert a SIDL symbol into the name of its associated constructor,
    * which is the symbol name appended with "__new".
    */
   public static String getNewName(SymbolID id) {
      return getSymbolName(id) + "__new";
   }

   /**
    * Convert a SIDL symbol into the name of its associated remote
    * constructor, which is the symbol name appended with "__remote".
    */
   public static String getRemoteName(SymbolID id) {
      return getSymbolName(id) + "__remote";
   }

   /**
    * Convert a SIDL symbol into the name of its set EPV method.
    */
   public static String getSetEPVName(SymbolID id) {
      return getSymbolName(id) + "__set_epv";
   }

   /**
    * Convert a SIDL symbol into the name of its set static EPV method.
    */
   public static String getSetSEPVName(SymbolID id) {
      return getSymbolName(id) + "__set_sepv";
   }

  /**
   * Return the static EPV prefix string or, if invalid, the one with the
   * minimum type value.
   */
  public static String getEPVPrefix(int epvType) {
    int type = EPV_MINIMUM;
    if ( (EPV_MINIMUM <= epvType) && (epvType <= EPV_MAXIMUM) ) {
      type = epvType;
    }
    return "s_" + s_EPV_DESC[type] + "_epv";
  }

  /**
   * Return the name of the specified static EPV variable.
   */
  public static String getStaticEPVVariable(SymbolID id, int epvType, 
                                            int setType) 
  {
    String name = getSymbolName(id).toLowerCase();
    String type = getSetEPVTypeName(setType);
    return getEPVPrefix(epvType) + (type.equals("") ? "" : "_" + type)
           + "__" + name;
  }

  /**
   * Generate the static EPV variables for the specified extendable and
   * EPV type.
   */
  public static void generateStaticEPVVariables(LanguageWriterForC lw, 
                                                Extendable ext,
                                                boolean has_static, boolean is_remote, 
                                                int setType)
  {
    SymbolID id        = ext.getSymbolID();
    String   sType     = "static " + getEPVName(id);
    String   sTypeStr  = (has_static ? sType + "  " : sType + " ");
    String   ssTypeStr = "static " + getSEPVName(id) + " ";

    if (!ext.isInterface() && !is_remote) {
      lw.print(sTypeStr);
      lw.println(getStaticEPVVariable(id, EPV_NEW, setType) + ";");
    }

    if(is_remote) {
      lw.print(sTypeStr);
      lw.println(getStaticEPVVariable(id, EPV_REMOTE, setType) + ";");
    }

    if (has_static) {
      lw.print(ssTypeStr);
      lw.println(getStaticEPVVariable(id, EPV_STATIC, setType) + ";");
    }
    lw.println();
  }

    /**
     * Convert a sidl symbol into the name of its associated _call_load method
     * which is the symbol name appended with "__call_load"
     */
    public static String getCallLoadName(SymbolID id ) { 
	return getSymbolName(id) + "__call_load";
    }

   /**
    * Convert a SIDL symbol into the name of its associated local 
    * statics method.
    */
   public static String getLocalStaticsName(SymbolID id) {
      return getSymbolName(id) + "__get_static_epv";
   }

   /**
    * Convert a SIDL symbol into the name of its associated statics
    * method, which is the symbol name appended with "__statics".
    */
   public static String getStaticsName(SymbolID id) {
      return getSymbolName(id) + "__statics";
   }

   /**
    * Convert a SIDL symbol into the name of its associated init
    * method, which is the symbol name appended with "__init".
    */
   public static String getInitName(SymbolID id) {
      return getSymbolName(id) + "__init";
   }
   /**
    * Convert a SIDL symbol into the name of its associated fini
    * method, which is the symbol name appended with "__fini".
    */
   public static String getFiniName(SymbolID id) {
      return getSymbolName(id) + "__fini";
   }

   /**
    * Convert a SIDL symbol into the name of its associated fini
    * method, which is the symbol name appended with "__fini".
    */
   public static String getVersionName(SymbolID id) {
      return getSymbolName(id) + "__IOR_version";
   }

  /**
   * Return the name of the type of the implicit exception argument;
   * namely, sidl_BaseInterface__object.  This is deemed necessary
   * in order to minimize the impact on existing Impl codes due to
   * the memory layout of the epv.  That is, it is not necessary for
   * the implementation writer to cast a newly created exception 
   * (to the base exception interface) IF the pointer is declared
   * to be the start of the epv structure to begin with.
   */
  public static String getExceptionFundamentalType() {
    return s_exceptionFundamentalType;
  }

  /**
   * Return the name of the type of the implicit base class type.
   * The return value is of the form "struct X_Y_Z *" where X_Y_Z
   * depends on the name of the base class and its mapping
   * to a C struct name.
   */
  public static String getClassType() {
    return s_objectType;
  }

  /**
   * Return the name of the type of the base interface type.
   * The return value is of the form "struct X_Y_Z *" where X_Y_Z
   * depends on the name of the base interface and its mapping
   * to a C struct name.
   */
  public static String getInterfaceType() {
    return s_interfaceType;
  }

  public static String getPreconditionExceptType() {
    return "struct " + PRECONDITION_CALL_PREFIX + "__object *";
  }

  public static String getPostconditionExceptType() {
    return "struct " + POSTCONDITION_CALL_PREFIX + "__object *";
  }

  public static String getInvariantExceptType() {
    return "struct " + INVARIANT_CALL_PREFIX + "__object *";
  }

  public static String getSymbolType(Symbol sym) {
    if (sym.getSymbolType() == Symbol.ENUM) {
      return getEnumName(sym.getSymbolID());
    } else {
      return getObjectName(sym.getSymbolID()) + "*";
    }
  }

  /**
   * Generate a return string for the specified SIDL type.  Most of
   * the SIDL return strings are listed in the static structures defined
   * at the start of the class.  Symbol types and array types require
   * special processing.
   *
   * @param type   the <code>Type</code> whose return string is being built.
   */
  public static String getReturnString(Type type) throws CodeGenerationException
  {
    return getReturnString(type, true, false);
  }

  /**
   * Generate a C return string for the specified SIDL type.  Most of
   * the SIDL return strings are listed in the static structures defined
   * at the start of the class.  Symbol types and array types require
   * special processing.
   *
   * @param type   the <code>Type</code> whose return string is being built.
   * @param objPtr TRUE if the object pointer type should be returned; FALSE
   *               otherwise.
   * @param inStub TRUE is the string is for the stub; FALSE otherwise.
   */
  public static String getReturnString(Type type, boolean objPtr,
                                       boolean inStub)
    throws CodeGenerationException
  {
    /*
     * If the type is one of the primitive types, then just return
     * its string value from the lookup table.
     */
    int t = type.getType();
    if (t < s_types.length) {
      return s_types[t];
    }

   /*
     * If the type is a symbol, then look up the symbol type and return
     * the associated type name.
     */
    if (t == Type.SYMBOL) {
      Symbol symbol = Utilities.lookupSymbol(type.getSymbolID());
      SymbolID id   = symbol.getSymbolID();

      if (symbol.getSymbolType() == Symbol.ENUM) {
        return C.getEnumName(id);
      } else if (objPtr) {
        return C.getSymbolObjectPtr(id);
      } else {
        return C.getObjectName(id);
      }
    }

    /*
     * If the type is an array, then either return one of the primitive
     * array types or construct the corresponding array type.
     */
    if (t == Type.ARRAY) {
      Type atype = type.getArrayType();

      if (inStub && type.isRarray())
        return getReturnString(atype, objPtr, inStub) + "*";

      if (null != atype) {
        int  a     = atype.getType();

        if (a < s_array_types.length) {
          return s_array_types[a];
        } else {
          return getArrayName(atype.getSymbolID()) + "*";
        }
      } else {
        return getArrayName(null) + "*";
      }
    }
    return null;
  }

  /**
   * Generate a string containing only the specified method's arguments,
   * including exceptions, if any.
   */
  public static String getArgumentString(Method method)
    throws CodeGenerationException 
  {
     StringBuffer argstring = new StringBuffer();

     boolean has_throws = !method.getThrows().isEmpty();
     List args = method.getArgumentList();

     if ((args.size() > 0) || has_throws) {
        argstring.append(",");
     }

     for (Iterator a = args.iterator(); a.hasNext(); ) {
        Argument arg = (Argument) a.next();
        argstring.append(getArgumentString(arg, true, false, false));
        if (a.hasNext() || has_throws) {
           argstring.append(",");
        }
     }

     if (has_throws) {
       argstring.append(getExceptionFundamentalType());
       argstring.append('*');
     }

     return argstring.toString();
  }

   /**
    * Generate an argument string for the specified SIDL argument.
    * The formal argument name is not included.
    *
    * @param arg    the <code>Argument</code> whose string is being built.
    */
   public static String getArgumentString(Argument arg)
     throws CodeGenerationException
   {
     return getArgumentString(arg, false, false, false);
   }

   /**
    * Generate a C argument string for the specified SIDL argument.
    * The formal argument name is not included.
    *
    * @param arg    the <code>Argument</code> whose string is being built.
    * @param objPtr TRUE if the object pointer type should be returned; FALSE
    *               otherwise.
    * @param inStub TRUE is the string is for the stub; FALSE otherwise.
    * @param isExec TRUE if the string is for declaring variables in an Exec
    *               function, FALSE otherwise
    */
   public static String getArgumentString(Argument arg, boolean objPtr,
                                          boolean inStub, boolean isExec)
     throws CodeGenerationException
   {
     Type   type = arg.getType();
     String s    = getReturnString(type, objPtr, inStub);
 
     if (arg.getMode() == Argument.IN) {
       if (type.getType() == Type.STRING) {
         s = "const " + s;
       }
     } else if ( ((!inStub) || (!type.isRarray())) && !isExec ) {
       s = s + "*";
     }
 
     return s;
   }

   /**
    * Generate an argument string with the formal argument name.
    *
    * @param arg    the <code>Argument</code> whose string is being built.
    */
   public static String getArgumentWithFormal(Argument arg)
     throws CodeGenerationException
   {
     return getArgumentWithFormal(arg, false, false, false);
   }

   /**
    * Generate a C argument string with the formal argument name.
    *
    * @param arg    the <code>Argument</code> whose string is being built.
    * @param objPtr TRUE if the object pointer type should be returned; FALSE
    *               otherwise.
    * @param inStub TRUE if the string is for the stub; FALSE otherwise.
    * @param isExec TRUE if the string is generated for an exec function
    */
   public static String getArgumentWithFormal(Argument arg, boolean objPtr,
                                              boolean inStub, boolean isExec)
     throws CodeGenerationException
   {
     return getArgumentString(arg, objPtr, inStub, isExec) + " " + arg.getFormalName();
   }

   /**
    * Generate a cast string for the specified method.  The string
    * argument self represents the name of the object.  A code generation
    * exception is thrown if any of the required symbols do not exist in
    * the symbol table.
    */
   public static String getCast(Method method, String self)
     throws CodeGenerationException 
   {

      /*
       * Begin the cast string with the return type and self object reference.
       */
      StringBuffer cast = new StringBuffer();
      cast.append("(");
      cast.append(getReturnString(method.getReturnType(), true, false));
      cast.append(" (*)(");
      cast.append(self);

      /*
       * Add the method arguments to the cast clause as well as an
       * optional exception argument.
       */
      cast.append(getArgumentString(method));
      cast.append("))");

      return cast.toString();
   }

   /**
    * Return the static epv type option name.
    */
   public static String getStaticTypeOption(SymbolID id, int type) {
      int t = SET_PUBLIC;
      if ( (SET_MINIMUM <= type) && (type <= SET_MAXIMUM) ) {
         t = type;
      }
      String name = getSetEPVTypeName(t).toUpperCase();
      return "s_SEPV_" + getSymbolName(id).toUpperCase() + "_"
           + (name.equals("") ? "BASE" : name);
   }

   /**
    * Return the method index constant name associated with the specified 
    * method.
    */
  public static String getMethodIndex(SymbolID id, Method meth) {
     return "s_IOR_" + getSymbolName(id).toUpperCase() + "_" 
           + meth.getLongMethodName().toUpperCase();
  }

   /**
    * Return the method index constant name associated with the specified 
    * literal.
    */
  public static String getMethodIndex(SymbolID id, String lit) {
     return "s_IOR_" + getSymbolName(id).toUpperCase() + "_" 
           + lit.toUpperCase();
  }

  public static class CompareMethods implements Comparator {
    public int compare(Object o1, Object o2)
    {
      Method m1 = (Method)o1;
      Method m2 = (Method)o2;
      return m1.getLongMethodName().compareTo(m2.getLongMethodName());
    }
  }
}
