//
// File:        TypeModule.java
// Package:     gov.llnl.babel.backend.fortran
// Copyright:   (c) 2003 The Regents of the University of California
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.4 $
// Date:        $Date: 2003/09/03 15:08:42 $
// Description: Generate an F90 module containing a derived type
// 

package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;


/**
 * This class generates a FORTRAN 90 module holding the derived type for
 * client-side users.
 *
 * The type is held separately to avoid circular dependencies.
 */
public class TypeModule {
  static private final int s_maxArray = BabelConfiguration.getMaximumArray();

  /**
   * This holds the output device where the file will be written.
   */
  private LanguageWriterForFortran d_lw;

  /**
   * This is the extendable (i.e., class or interface) that is being
   * generated.
   */
  private Symbol d_sym;

  public TypeModule(LanguageWriterForFortran writer,
                    Symbol sym)
  {
    d_lw = writer;
    d_sym = sym;
  }

  private void describeFile()
  {
    d_lw.beginBlockComment(false);
    d_lw.println("This file contains a FORTRAN 90 derived type for the");
    d_lw.println("SIDL type " + d_sym.getFullName() + ".");
    d_lw.endBlockComment(false);
  }

  static void writeArrayType(LanguageWriter lw,
                             SymbolID id,
                             String implDataType)
  {
    for(int i = 1; i <= s_maxArray; ++i) {
      lw.println("type " + Fortran.getArrayName(id, i));
      lw.increaseTabLevel();
      lw.println("sequence");
      lw.println("integer (selected_int_kind(18)) :: d_array");
      if (implDataType != null){
        lw.println(implDataType + ", pointer, &");
        lw.increaseTabLevel();
        lw.print("dimension(:");
        for(int j = 1; j < i; ++j) {
          lw.print(",:");
        }
        lw.println(") :: d_data");
        lw.decreaseTabLevel();
      }
      
      lw.decreaseTabLevel();
      lw.println("end type " + Fortran.getArrayName(id, i));
      lw.println();
    }
  }

  private void writeModule()
  {
    final SymbolID id = d_sym.getSymbolID();
    d_lw.println("module " + Fortran.getTypeModule(id));
    d_lw.increaseTabLevel();
    if (d_sym instanceof Extendable) {
      d_lw.println("type " + Fortran.getTypeName(id));
      d_lw.increaseTabLevel();
      d_lw.println("sequence");
      d_lw.println("integer (selected_int_kind(18)) :: d_ior");
      d_lw.decreaseTabLevel();
      d_lw.println("end type " + Fortran.getTypeName(id));
      d_lw.println();
    }
    writeArrayType(d_lw, id, null);
    d_lw.decreaseTabLevel();
    d_lw.println("end module " + Fortran.getTypeModule(id));
                 
  }

  public void generateCode()
  {
    final SymbolID id = d_sym.getSymbolID();
    d_lw.writeBanner(d_sym, Fortran.getTypeFile(id), false,
                     CodeConstants.C_FORTRAN_TYPE_MODULE_PREFIX +
                     id.getFullName());
    d_lw.disableLineBreak();
    d_lw.printlnUnformatted("#include \"" +
                            Fortran.getStubNameFile(id) + "\"");
    d_lw.enableLineBreak();
    d_lw.println();
    describeFile();
    writeModule();
  }

  /**
   * Generate the FORTRAN 90 type module for a SIDL extendable (i.e., class
   * or interface).
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   * a catch all exception to indicate problems in the code generation
   * phase.
   */
  public static void generateCode(Symbol sym,
                                  LanguageWriterForFortran writer)
    throws CodeGenerationException
  {
    TypeModule tm = new TypeModule(writer, sym);
    tm.generateCode();
  }
}
