//
// File:        SymbolToDOM.java
// Package:     gov.llnl.babel.parsers.xml
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: SymbolToDOM.java,v 1.27 2003/03/27 22:46:47 dahlgren Exp $
// Description: convert SIDL symbols into an XML DOM document
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.parsers.xml;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.parsers.xml.DTDManager;
import gov.llnl.babel.parsers.xml.StringXML;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Metadata;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.xml.XMLUtilities;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

/**
 * Class <code>SymbolToDOM</code> converts a SIDL symbol into an XML DOM
 * document.  Utility function <code>convert</code>takes a symbol and returns
 * a DOM document.  The DOM representation is formatted with white space such
 * that a straight-forward DOM printer will generate pretty XML output.
 */
public class SymbolToDOM {
   private static final String EOL    = "\n";  // standard web end-of-line
   private static final String INDENT = "   "; // amount of each indent level

   private Document d_document;
   private int      d_indent;

   /**
    * This is a convenience utility function that converts the symbol into
    * a DOM document.  Since this method is static, it may be called without
    * explicitly creating an instance of object <code>SymbolToDOM</code>.
    */
   public static Document convert(Symbol symbol) {
      SymbolToDOM sym2dom = new SymbolToDOM(symbol);
      return sym2dom.getDocument();
   }

   /**
    * This is a convenience utilility function that converts the symbol
    * directly into a string.
    */
   public static String convertToString(Symbol symbol) {
      return XMLUtilities.getXMLString(convert(symbol));
   }

   /**
    * Create a symbol to DOM converter object.  The constructor creates a
    * DOM document node with the appropriate document type to validate the
    * symbol input.  The resulting document may be read by a call to method
    * <code>getDocument</code>.
    */
   public SymbolToDOM(Symbol symbol) {
      DOMImplementation dom = XMLUtilities.createDOMImplementation();
      DocumentType doctype = dom.createDocumentType(
         "Symbol", DTDManager.SYMBOL_PUBLIC_ID, "SIDL.dtd");
      d_document = dom.createDocument(null, "Symbol", doctype);

      if (symbol != null) {
         d_indent = 0;
         Element docsymbol = d_document.getDocumentElement();
         switch (symbol.getSymbolType()) {
         case Symbol.CLASS:
            convertClass((Class) symbol, docsymbol);
            break;
         case Symbol.ENUM:
            convertEnumeration((Enumeration) symbol, docsymbol);
            break;
         case Symbol.INTERFACE:
            convertInterface((Interface) symbol, docsymbol);
            break;
         case Symbol.PACKAGE:
            convertPackage((Package) symbol, docsymbol);
            break;
         }
         docsymbol.appendChild(d_document.createTextNode(EOL));
      }
   }

   /**
    * Return the DOM document for the symbol specified in the constructor.
    */
   public Document getDocument() {
      return d_document;
   }

   /**
    * Convert a SIDL class description into its DOM representation.
    * A class consists of an extends symbol, an implements block of
    * symbols, and a methods block.
    */
   private void convertClass(Class cls, Node parent) {
      convertSymbolID(cls.getSymbolID(), parent);
      convertMetadata(cls.getMetadata(), parent);
      convertComment (cls.getComment() , parent);

      Element class_element = createElement("Class");
      class_element.setAttribute("abstract",String.valueOf(cls.isAbstract()));

      /*
       * Write the extends block of a single class identifier.
       */
      Element extends_element = createElement("Extends");
      if (cls.getParentClass() != null) {
         convertSymbolID(cls.getParentClass().getSymbolID(), extends_element);
      }
      appendChild(extends_element, class_element);

      /*
       * Write the implements block of symbol identifiers.
       */
      Element implementsblock_element = createElement("ImplementsBlock");
      Collection parents = cls.getParentInterfaces(false);
      for (Iterator p = parents.iterator(); p.hasNext(); ) {
         Interface i = (Interface) p.next();
         convertSymbolID(i.getSymbolID(), implementsblock_element);
      }
      appendChild(implementsblock_element, class_element);

      /*
       * Write all parents of this class - all interfaces and classes.
       */
      Element all_classes_element = createElement("AllParentClasses");
      Class parent_class = cls.getParentClass();
      while (parent_class != null) {
         convertSymbolID(parent_class.getSymbolID(), all_classes_element);
         parent_class = parent_class.getParentClass();
      }
      appendChild(all_classes_element, class_element);
      
      Element all_interfaces_element = createElement("AllParentInterfaces");
      Collection allparents = cls.getParentInterfaces(true);
      for (Iterator p = allparents.iterator(); p.hasNext(); ) {
         Interface i = (Interface) p.next();
         convertSymbolID(i.getSymbolID(), all_interfaces_element);
      }
      appendChild(all_interfaces_element, class_element);

      /*
       * Write the methods block of class methods.
       */
      Element methodblock_element = createElement("MethodsBlock");
      for (Iterator m = cls.getMethods(false).iterator(); m.hasNext(); ) {
         convertMethod((Method) m.next(), methodblock_element);
      }
      appendChild(methodblock_element, class_element);
      
      appendChild(class_element, parent);
   }

   /**
    * Convert a SIDL enumeration symbol into its DOM representation.  An
    * enumeration consists of the standard symbol header followed by a list
    * of the enumerator symbols.
    */
   private void convertEnumeration(Enumeration e, Node parent) {
      convertSymbolID(e.getSymbolID(), parent);
      convertMetadata(e.getMetadata(), parent);
      convertComment (e.getComment() , parent);

      Element enumeration_element = createElement("Enumeration");
      for (Iterator i = e.getIterator(); i.hasNext(); ) {
         String n = (String) i.next();
         Element enumerator_element = createElement("Enumerator");
         enumerator_element.setAttribute("name", n);
         enumerator_element.setAttribute(
            "value", String.valueOf(e.getEnumeratorValue(n)));
         enumerator_element.setAttribute(
            "fromuser", String.valueOf(e.definedByUser(n)));
         Comment ecmt = e.getEnumeratorComment(n);
         if ((ecmt != null) && (!ecmt.isEmpty())) {
           convertComment(ecmt, enumerator_element);
         }
         appendChild(enumerator_element, enumeration_element);
      }
      appendChild(enumeration_element, parent);
   }

   /**
    * Convert a SIDL interface description into its DOM representation.
    * An interface consists of the standard symbol header followed by
    * an extends block and then a methods block.
    */
   private void convertInterface(Interface ifc, Node parent) {
      convertSymbolID(ifc.getSymbolID(), parent);
      convertMetadata(ifc.getMetadata(), parent);
      convertComment (ifc.getComment() , parent);

      Element interface_element = createElement("Interface");

      /*
       * Write the extends block of symbol identifiers.
       */
      Element extendsblock_element = createElement("ExtendsBlock");
      Collection parents = ifc.getParentInterfaces(false);
      for (Iterator p = parents.iterator(); p.hasNext(); ) {
         Interface i = (Interface) p.next();
         convertSymbolID(i.getSymbolID(), extendsblock_element);
      }
      appendChild(extendsblock_element, interface_element);

      /*
       * Write all parent interfaces of this interface.
       */
      Element all_interfaces_element = createElement("AllParentInterfaces");
      Collection allparents = ifc.getParentInterfaces(true);
      for (Iterator p = allparents.iterator(); p.hasNext(); ) {
         Interface i = (Interface) p.next();
         convertSymbolID(i.getSymbolID(), all_interfaces_element);
      }
      appendChild(all_interfaces_element, interface_element);

      /*
       * Write the methods block of interface methods.
       */
      Element methodblock_element = createElement("MethodsBlock");
      for (Iterator m = ifc.getMethods(false).iterator(); m.hasNext(); ) {
         convertMethod((Method) m.next(), methodblock_element);
      }
      appendChild(methodblock_element, interface_element);
      
      appendChild(interface_element, parent);
   }

   /**
    * Convert a SIDL package symbol into its DOM representation.  A package
    * consists of the standard symbol header followed by a list of the package
    * contents.
    */
   private void convertPackage(Package p, Node parent) {
      convertSymbolID(p.getSymbolID(), parent);
      convertMetadata(p.getMetadata(), parent);
      convertComment (p.getComment() , parent);

      Element package_element = createElement("Package");
      package_element.setAttribute("final",
                                   p.getFinal() ? "true" : "false");
      Collection children = p.getOrderedSymbolReferences();
      Map map = p.getSymbols();
      for (Iterator i = children.iterator(); i.hasNext(); ) {
         SymbolID entry = (SymbolID) i.next();
         Integer intValue =(Integer)map.get(entry);
         Element packagesymbol_element = createElement("PackageSymbol");
         packagesymbol_element.setAttribute(
            "name", entry.getShortName());
         packagesymbol_element.setAttribute
           ("type", StringXML.toSymbolXML(intValue.intValue()));
         packagesymbol_element.setAttribute
           ("version", entry.getVersion().getVersionString());
                                           
         appendChild(packagesymbol_element, package_element);
      }
      appendChild(package_element, parent);
   }

   /**
    * Convert a SIDL comment into its DOM representation.  Comments
    * consist of a number of comment lines.  Comments may contain XML
    * formatting elements if the comment is well-formed and follows
    * the comment DTD.
    */
   private void convertComment(Comment c, Node parent) {
      Element comment_element = null;

      /*
       * If the comment is empty, then return an empty comment element.
       */
      
      if (c.isEmpty()) {
         comment_element = createElement("Comment");
      } else {
         String[] lines = c.getComment();

        /*
         * First, check whether the comment is a well-formed as defined
         * by the comment DTD.  If the comment DTD does not exist or if
         * the comment is not valid, then write the comment as a text node.
         */

        StringBuffer buffer = new StringBuffer();
        buffer.append("<Comment>");
        for (int i = 0; i < lines.length; i++) {
          buffer.append(EOL);
          buffer.append(lines[i]);
        }
        buffer.append("</Comment>");

        Document document = XMLUtilities.validateXML
          (DTDManager.COMMENT_PUBLIC_ID, DTDManager.COMMENT_FILE,
           DTDManager.getInstance(), "Comment", buffer.toString());

        if (document != null) {
          Element root = document.getDocumentElement();
          comment_element =
            (Element) XMLUtilities.cloneDOM(root, d_document);
          d_indent++;
        }

        /*
         * If we have not been able to create a valid comment element as
         * a DOM structure, then create a text comment node.
         */
         
        if (comment_element == null) {
          comment_element = createElement("Comment");
          Text text = d_document.createTextNode("");
          for (int i = 0; i < lines.length; i++) {
            text.appendData(EOL);
            text.appendData(lines[i]);
          }
          comment_element.appendChild(text);
        }
      }

      appendChild(comment_element, parent);
   }

   /**
    * Convert metadata into its DOM representation.  The element name is
    * <code>Metadata</code> with attribute <code>date</code>.  There may
    * sub-elements that are (key,value) pairs.
    */
   private void convertMetadata(Metadata m, Node parent) {
      Element metadata_element = createElement("Metadata");
      if ( BabelConfiguration.getInstance().suppressTimestamps() ) {
        metadata_element.setAttribute("date", "20010101 00:00:00 GMT");
      } else { 
        metadata_element.setAttribute("date", m.getDateAsString());
      }

      Set entries = m.getMetadataDatabase().entrySet();
      for (Iterator i = entries.iterator(); i.hasNext(); ) {
         Map.Entry entry = (Map.Entry) i.next();
         Element data_element = createElement("MetadataEntry");
         data_element.setAttribute("key", (String) entry.getKey());
         data_element.setAttribute("value", (String) entry.getValue());
         appendChild(data_element, metadata_element);
      }
      appendChild(metadata_element, parent);
   }

   /**
    * Convert a SIDL method description into a DOM representation.
    * The element name is <code>Method</code> and contains a comment,
    * return type, argument list, and throws list.
    */
   private void convertMethod(Method m, Node parent) {
      /*
       * Create a method node and add attributes
       */
      Element method_element = createElement("Method");
      method_element.setAttribute("shortname", m.getShortMethodName());
      method_element.setAttribute("extension", m.getNameExtension());
      method_element.setAttribute(
         "copy", String.valueOf(m.isReturnCopy()));
      method_element.setAttribute(
         "definition", StringXML.toDefXML(m.getDefinitionModifier()));
      method_element.setAttribute(
         "communication", StringXML.toComXML(m.getCommunicationModifier()));

      /*
       * Create the comment and return type elements
       */
      convertComment(m.getComment(), method_element);
      convertType(m.getReturnType(), method_element);

      /*
       * Create the argument list element
       */
      Element arglist_element = createElement("ArgumentList");
      ArrayList args = m.getArgumentList();
      for (int a = 0; a < args.size(); a++) {
         convertArgument((Argument) args.get(a), arglist_element);
      }
      appendChild(arglist_element, method_element);

      /*
       * Create the throws list element
       */
      Element throwslist_element = createElement("ThrowsList");
      for (Iterator t = m.getThrows().iterator(); t.hasNext(); ) {
         convertSymbolID((SymbolID) t.next(), throwslist_element);
      }
      appendChild(throwslist_element, method_element);

      appendChild(method_element, parent);
   }

   /**
    * Convert a SIDL argument description into a DOM representation.
    * The element name is <code>Argument</code> with the attributes
    * <code>copy</code>, <code>mode</code>, and <code>name</code>.
    */
   private void convertArgument(Argument arg, Node parent) {
      Element arg_element = createElement("Argument");
      arg_element.setAttribute("copy", String.valueOf(arg.isCopy()));
      arg_element.setAttribute("mode", StringXML.toModeXML(arg.getMode()));
      arg_element.setAttribute("name", arg.getFormalName());
      convertType(arg.getType(), arg_element);
      appendChild(arg_element, parent);
   }

   /**
    * Convert a SIDL type description into a DOM representation.  The
    * element is <code>Type</code> with attribute <code>type</code>.
    */
   private void convertType(Type type, Node parent) {
      Element type_element = createElement("Type");
      int typeid = type.getType();
      type_element.setAttribute("type", StringXML.toTypeXML(typeid));

      if (typeid == Type.SYMBOL) {
         convertSymbolID(type.getSymbolID(), type_element);
      } else if (typeid == Type.ARRAY) {
         Element array_element = createElement("Array");
         array_element.setAttribute(
            "dim", String.valueOf(type.getArrayDimension()));
         array_element.setAttribute(
            "order", StringXML.toOrderXML(type.getArrayOrder()));
         convertType(type.getArrayType(), array_element);
         appendChild(array_element, type_element);
      }

      appendChild(type_element, parent);
   }

   /**
    * Convert a symbol identifier into a DOM representation.  The element
    * name is <code>SymbolName</code> with attributes <code>name</code>
    * and <code>version</code>.
    */
   private void convertSymbolID(SymbolID id, Node parent) {
      Element symbol_element = createElement("SymbolName");
      symbol_element.setAttribute("name", id.getFullName());
      symbol_element.setAttribute(
         "version", id.getVersion().getVersionString());
      appendChild(symbol_element, parent);
   }

   /**
    * Return the indent string, which is a character return plus spaces that
    * put the cursor at the appropriate indentation level.
    */
   private String getIndentString() {
      StringBuffer indent = new StringBuffer();
      indent.append(EOL);
      for (int i = 0; i < d_indent; i++) {
         indent.append(INDENT);
      }
      return indent.toString();
   }

   /**
    * Create a new element and increase the indentation level.
    */
   private Element createElement(String name) {
      d_indent++;
      return d_document.createElement(name);
   }

   /**
    * Append the child to the parent with the proper formatting.
    */
   private void appendChild(Element child, Node parent) {
      String indent = getIndentString();
      parent.appendChild(d_document.createTextNode(indent));
      if (child.hasChildNodes()) {
         child.appendChild(d_document.createTextNode(indent));
      }
      parent.appendChild(child);
      d_indent--;
   }
}
