//
// File:        Type.java
// Package:     gov.llnl.babel.symbols
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Type.java,v 1.14 2003/03/17 23:43:26 epperly Exp $
// Description: class that represents a SIDL type
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;

/**
 * The <code>Type</code> class defines a SIDL type, such as a primitive
 * type (boolean, char, dcomplex, double, fcomplex, float, int, long, opaque,
 * and string), arrays, and user-defined types (enum, interface, or class).
 * A <code>Type</code> object may also be created for a void return type
 * from a method.
 */
public class Type {
   public final static int VOID      =  0;
   public final static int BOOLEAN   =  1;
   public final static int CHAR      =  2;
   public final static int DCOMPLEX  =  3;
   public final static int DOUBLE    =  4;
   public final static int FCOMPLEX  =  5;
   public final static int FLOAT     =  6;
   public final static int INT       =  7;
   public final static int LONG      =  8;
   public final static int OPAQUE    =  9;
   public final static int STRING    = 10;
   public final static int ENUM      = Symbol.ENUM;
   public final static int CLASS     = Symbol.CLASS;
   public final static int INTERFACE = Symbol.INTERFACE;
   public final static int PACKAGE   = Symbol.PACKAGE;
   public final static int SYMBOL    = 15;
   public final static int ARRAY     = 16;

   private final static String s_names[] = {
      "void",      "bool", "char", "dcomplex", "double", "fcomplex",
      "float",     "int",  "long", "opaque",   "string", "enum",
      "interface", "class"
   };

   public final static int UNSPECIFIED  = 0;
   public final static int COLUMN_MAJOR = 1;
   public final static int ROW_MAJOR    = 2;

   private final static String s_order[] = {
      "unspecified", "column-major", "row-major"
   };

   private int      d_type;        // integer description of type (from above)
   private SymbolID d_symbol_id;   // symbol identifier if SYMBOL
   private Type     d_array_type;  // type of array if ARRAY
   private int      d_array_dim;   // dimension of array if ARRAY
   private int      d_array_order; // order of array if ARRAY

   /**
    * Create a new primitive type (boolean, char, dcomplex, double, fcomplex,
    * float, int, long, opaque, and string).  Use the other constructors to
    * create a symbol or array type.
    */
   public Type(int type) {
      d_type        = type;
      d_symbol_id   = null;
      d_array_type  = null;
      d_array_dim   = 0;
      d_array_order = UNSPECIFIED;
   }

   /**
    * Create a new type given a user-defined symbol identifier.
    * This symbol identifier will represent an enum, interface,
    * or class.
    */
   public Type(SymbolID id) {
      d_type        = SYMBOL;
      d_symbol_id   = id;
      d_array_type  = null;
      d_array_dim   = 0;
      d_array_order = UNSPECIFIED;
   }
 
   /**
    * Create a new array provided the array type, dimension, and order.
    */
   public Type(Type type, int dim, int order) {
      d_type        = ARRAY;
      d_symbol_id   = null;
      d_array_type  = type;
      d_array_dim   = dim;
      d_array_order = order;
   }

   /**
    * Return the integer that identifies the type of this type.
    * If the type is ENUM, INTERFACE, or CLASS, it is returned
    * as the generic SYMBOL. (for historical purposes)
    */
   public int getType() {
     int type = getDetailedType();
     if ((type == ENUM) || (type == INTERFACE) || (type == CLASS) ||
         (type == PACKAGE)) { 
       type = SYMBOL;
     }
     return type;
   }

  /**
   * Returns the integer that identifies the type of this type.
   * If type == SYMBOL, this method will try to further refine
   * the type to one of (CLASS, INTERFACE, ENUM).  This may, 
   * in fact, fail if the type is undefined (via a forward reference)
   * and then SYMBOL is returned.
   */ 
  public int getDetailedType() { 
    if ( d_type == SYMBOL ) { 
      // try to further refine the type information
      try { 
        Symbol s = SymbolTable.getInstance().lookupSymbol(d_symbol_id);
        if (s != null) {
          int symbolType = s.getSymbolType();
          switch (symbolType) { 
          case Symbol.CLASS:
          case Symbol.ENUM:
          case Symbol.INTERFACE:
          case Symbol.PACKAGE:
            d_type = symbolType;
            break;
          }
        }
      } catch ( Exception e ) { 
        // if all else fails, it remains a symbol
      }          
    }
    return d_type;
  }

   /**
    * Check whether the type is one of the standard primitive types,
    * including strings.
    */
   public boolean isPrimitive() {
      return d_type <= STRING;
   }

   /**
    * Check whether the type is a string.
    */
   public boolean isString() {
      return d_type == STRING;
   }

   /**
    * Check whether the type is a symbol.
    */
   public boolean isSymbol() {
      return getType() == SYMBOL;
   }

   /**
    * Check whether the type is an array.
    */
   public boolean isArray() {
      return d_type == ARRAY;
   }

   /**
    * If this type is an array, then return the dimension; otherwise,
    * return zero.
    */
   public int getArrayDimension() {
      return d_array_dim;
   }

   /**
    * If this type is an array, then return the order; otherwise,
    * return zero.
    */
   public int getArrayOrder() {
      return d_array_order;
   }

   /**
    * If this is an array, return the array type; otherwise return null.
    */
   public Type getArrayType() {
      return d_array_type;
   }

  /**
   * Return <code>true</code> if and only if the type is an array with an
   * ordering specification.  For example, array&lt;int,2,column-major&gt;
   * would return <code>true</code>; array&lt;int, 2&gt; would return
   * <code>false</code>.  For non-array types, this always returns
   * <code>false</code>. 
   */
  public boolean hasArrayOrderSpec() {
    return (d_type == ARRAY) && (d_array_order != UNSPECIFIED);
  }

   /**
    * Return the symbol identifier associated with this symbol type.
    * If this is not a symbol type, then return null.
    */
   public SymbolID getSymbolID() {
      return d_symbol_id;
   }

   /**
    * Return a string representation of the type for printing out the
    * types in a method signature.
    */
   public String getTypeString() {
      StringBuffer s    = new StringBuffer();
      int          type = getType();
      if ( type == SYMBOL ) {
         s.append(d_symbol_id.getFullName());
      } else if ( type == ARRAY ) {
         s.append("array<" + d_array_type.getTypeString());
         if (d_array_dim > 1) {
           s.append("," + d_array_dim);
         }
         if ( (UNSPECIFIED < d_array_order) && (d_array_order <= ROW_MAJOR) ) {
            s.append("," + s_order[d_array_order]);
         } 
         s.append(">");
      } else if ((type >= VOID) && (type <= STRING)) {
         s.append(s_names[d_type]);
      }
      return s.toString();
   }

   /**
    * Compare two type objects and return true if they are the same.
    */
   public boolean equals(Object object) {
      boolean eq = false;
      if ((object != null) && (object instanceof Type)) {
         Type t = (Type) object;
         int my_type = getType();
         int your_type = t.getType();
         if (my_type == your_type) {
            if (my_type == SYMBOL) {
               eq = d_symbol_id.equals(t.d_symbol_id);
            } else if (my_type == ARRAY) {
               eq = (  d_array_type.equals(t.d_array_type)
                    && (d_array_dim   == t.d_array_dim  )
                    && (d_array_order == t.d_array_order)  );
            } else {
               eq = true;
            }
         }
      }
      return eq;
   }
}
