/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.target;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.team.core.Team;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.core.sync.IRemoteSyncElement;
import org.eclipse.team.internal.core.NullSubProgressMonitor;
import org.eclipse.team.internal.core.target.IRemoteTargetResource;
import org.eclipse.team.internal.core.target.ITargetRunnable;
import org.eclipse.team.internal.core.target.ResourceState;
import org.eclipse.team.internal.core.target.SynchronizedTargetProvider;
import org.eclipse.team.internal.core.target.UrlUtil;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;
import org.eclipse.team.internal.ftp.client.FTPClient;
import org.eclipse.team.internal.ftp.client.FTPDirectoryEntry;
import org.eclipse.team.internal.ftp.client.FTPProxyLocation;
import org.eclipse.team.internal.ftp.client.FTPServerException;
import org.eclipse.team.internal.ftp.client.FTPServerLocation;
import org.eclipse.team.internal.ftp.client.IFTPClientListener;
import org.eclipse.team.internal.ftp.client.IFTPRunnable;

public final class FTPTargetProvider extends SynchronizedTargetProvider {

	private static Map openClients = new HashMap();
	
	private List resourcesNeedingRemoteIdentifier;
	
	private static final FTPDirectoryEntry IS_URL_ROOT = new FTPDirectoryEntry(null, true, false, 0 , null);

	public FTPTargetProvider(FTPSite site, IPath intrasitePath) throws TeamException {
		super(site, intrasitePath.makeRelative());
		resourcesNeedingRemoteIdentifier = new ArrayList();
	}

	/**
	 * @see TargetProvider#run(ITargetRunnable, IProgressMonitor)
	 */
	public void run(final ITargetRunnable runnable, IProgressMonitor monitor) throws TeamException {
		run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				FTPTargetProvider.super.run(runnable, monitor);
			}
		}, monitor);
	}
	
	/**
	 * Execute the given runnable in a context that has access to an open ftp connection
	 * that is opened in the directory of the provider.
	 */
	public void run(IFTPRunnable runnable, IProgressMonitor monitor) throws TeamException {
		
		// Determine if we are nested or not
		boolean isOuterRun = false;
		FTPClient client = getClient();
		
		try {
			monitor = Policy.monitorFor(monitor);
			monitor.beginTask(null, 110 + (client.isConnected() ? 0 : 20));
			if (!client.isConnected()) {
				// The path always has a leading / to remove
				String path = getURL().getPath();
				if (path.length() > 0 && path.charAt(0) == '/') {
					path = path.substring(1);
				}
				client.open(Policy.subMonitorFor(monitor, 10));
				if (path.length() > 0) {
					client.changeDirectory(path, Policy.subMonitorFor(monitor, 10));
				}
				openClients.put(getURL(), client);
				isOuterRun = true;
			}
			client.run(runnable, Policy.subMonitorFor(monitor, 100));
			
		} finally {
			if (isOuterRun) {
				try {
					client.close(Policy.subMonitorFor(monitor, 10));
				} finally {
					openClients.remove(getURL());
				}
			}
			monitor.done();
		}
	}

	private FTPClient getClient() {
		FTPClient client = (FTPClient)openClients.get(getURL());
		FTPSite ftpSite = (FTPSite)getSite();
		if (client == null) {
			// Authentication.
			String username = ftpSite.getUsername();
			String password = ftpSite.getPassword();
			FTPServerLocation location = new FTPServerLocation(getURL(), username, password, ftpSite.getUsePassive());
	
			// Proxy server.
			FTPProxyLocation proxyLocation = null;
			if (ftpSite.getProxy() != null) {
				proxyLocation = new FTPProxyLocation(ftpSite.getProxy(), username, password);
			}
	
			// Create client
			client = new FTPClient(location, proxyLocation, new IFTPClientListener() {
				public void responseReceived(int responseCode, String responseText) {
					if (Policy.DEBUG_RESPONSES) {
						System.out.println(responseCode + " " + responseText); //$NON-NLS-1$
					}
				}
				public void requestSent(String command, String argument) {
					if (Policy.DEBUG_REQUESTS) {
						System.out.println(command + " " + argument); //$NON-NLS-1$
					}
				}
			}, ftpSite.getTimeout() * 1000 /* convert timeout to milliseconds */);
		}
		return client;
	}
	
			/**
	 * @see SynchronizedTargetProvider#newState(IResource)
	 */
	public ResourceState newState(IResource resource) {
		return new FTPResourceState(this, resource);
	}
	
	/**
	 * @see SynchronizedTargetProvider#newState(IResource, IRemoteTargetResource)
	 */
	public ResourceState newState(IResource resource, IRemoteTargetResource remote) {
		// The provider of the given remote resource must match the receiver.
		if (remote != null) {
			FTPRemoteTargetResource ftpRemote = (FTPRemoteTargetResource)remote;
			FTPTargetProvider provider = ftpRemote.getProvider();
			if (intrasitePath.equals(provider.intrasitePath)) {
				return new FTPResourceState(resource, ftpRemote);
			} else {
				return new FTPResourceState(resource,
					new FTPRemoteTargetResource(this, UrlUtil.getTrailingPath(remote.getURL(), targetURL), resource.getType() != IResource.FILE));
			}
		} else {
			return newState(resource);
		}
	}
	
	/**
	 * @see TargetProvider#getRemoteResource()
	 */
	public IRemoteTargetResource getRemoteResource() {
		return new FTPRemoteTargetResource(this, Path.EMPTY, true /* is container */);
	}

	/**
	 * @see TargetProvider#getRemoteResourceFor(IResource)
	 */
	public IRemoteTargetResource getRemoteResourceFor(IResource resource) {
		return new FTPRemoteTargetResource(this, resource.getProjectRelativePath(), resource.getType() != IResource.FILE);
	}
	
	/**
	 * @see TargetProvider#getRemoteSyncElement(IResource)
	 */
	public IRemoteSyncElement getRemoteSyncElement(IResource resource) {
		return new FTPRemoteSyncElement(this, resource, getRemoteResourceFor(resource));
	}
	
	protected void createDirectory(final IPath providerRelativePath, IProgressMonitor monitor) throws TeamException {
		if (providerRelativePath.isEmpty()) return;
		run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				try {
					getClient().createDirectory(providerRelativePath.toString(), monitor);
				} catch (FTPServerException e) {
					if (e.getStatus().getCode() == FTPException.DOES_NOT_EXIST) {
						// A parent must not exist, try to create it
						createDirectory(providerRelativePath.removeLastSegments(1), monitor);
						getClient().createDirectory(providerRelativePath.toString(), monitor);
					} else {
						throw e;
					}
				}
			}
		}, Policy.monitorFor(monitor));
		
	}
	
	protected void deleteFile(IPath providerRelativePath, IProgressMonitor monitor) throws FTPException {
		getClient().deleteFile(providerRelativePath.toString(), monitor);
	}
	
	protected void deleteDirectory(IPath providerRelativePath, IProgressMonitor monitor) throws FTPException {
		getClient().deleteDirectory(providerRelativePath.toString(), monitor);
	}

	/*
	 * This method probably throws an exception if the parent doesn't exist
	 */
	protected FTPDirectoryEntry fetchEntry(IPath providerRelativePath, IProgressMonitor progress) throws TeamException {
		try {
			FTPDirectoryEntry[] entries = listFiles(providerRelativePath.removeLastSegments(1), progress);
			if (providerRelativePath.isEmpty()) {
				// We're at the root and it must exist since we got entries
				return IS_URL_ROOT;
			} else {
				for (int i = 0; i < entries.length; i++) {
					FTPDirectoryEntry dirEntry = entries[i];
					if (dirEntry.getName().equals(providerRelativePath.lastSegment())) {
						return dirEntry;
					}
				}
			}
			return null;
		} catch (TeamException e) {
			if (e.getStatus().getCode() == FTPServerException.DOES_NOT_EXIST) {
				return null;
			}
			throw e;
		}
	}
	
	/*
	 * This method probably throws an exception if the parent doesn't exist
	 */
	protected FTPDirectoryEntry fetchEntryForFile(IPath providerRelativePath, IProgressMonitor progress) throws TeamException {
		try {
			FTPDirectoryEntry[] entries = listFiles(providerRelativePath, progress);
			if (entries.length == 0) return null;
			if (entries.length > 1) {
				// Wrong number of entries. Is it a folder
				throw new FTPException(Policy.bind("FTPTargetProvider.remoteNotAFile1")); //$NON-NLS-1$
			}
			FTPDirectoryEntry dirEntry = entries[0];
			if (dirEntry.getName().equals(providerRelativePath.lastSegment())) {
				return dirEntry;
			} if (new Path(dirEntry.getName()).equals(providerRelativePath)) {
				return new FTPDirectoryEntry(
					providerRelativePath.lastSegment(), 
					dirEntry.hasDirectorySemantics(),
					dirEntry.hasFileSemantics(),
					dirEntry.getSize(),
					dirEntry.getModTime());
			} else {
				// Wrong entry. The remote must be a folder
				throw new FTPException(Policy.bind("FTPTargetProvider.remoteNotAFile2")); //$NON-NLS-1$
			}
		} catch (TeamException e) {
			if (e.getStatus().getCode() == FTPServerException.DOES_NOT_EXIST) {
				return null;
			}
			throw e;
		}
	}
	
	protected InputStream getContents(IPath providerRelativePath, boolean isBinary,  IProgressMonitor monitor) throws FTPException {
		return getClient().getContents(providerRelativePath.toString(), isBinary, 0L, monitor);
	}
	
	protected void getFile(IFile localFile, IProgressMonitor monitor) throws FTPException {
		getClient().getFile(localFile.getProjectRelativePath().toString(), localFile,
			Team.getType(localFile) != Team.TEXT, false /* resume */, monitor);
	}
	
	protected void putFile(IFile localFile, IProgressMonitor monitor) throws FTPException {
		getClient().putFile(localFile.getProjectRelativePath().toString(), localFile,
			Team.getType(localFile) != Team.TEXT, monitor);
	}
			
	protected FTPDirectoryEntry[] listFiles(final IPath providerRelativePath, IProgressMonitor monitor) throws TeamException {
		final FTPDirectoryEntry[][] entries = new FTPDirectoryEntry[1][0];
		run(new IFTPRunnable() {
			public void run(IProgressMonitor monitor) throws TeamException {
				String directory = null;
				if (!providerRelativePath.isEmpty()) {
					directory = providerRelativePath.toString();
				}
				entries[0] =  getClient().listFiles(directory, monitor);
			}
		}, Policy.monitorFor(monitor));
		return entries[0];
	}

	/**
	 * @see org.eclipse.team.internal.core.target.TargetProvider#put(IResource[], IProgressMonitor)
	 */
	public void put(final IResource[] resources, IProgressMonitor progress) throws TeamException {
		run(new IFTPRunnable() {
			public void run(IProgressMonitor progress) throws TeamException {
				// set up progress monitoring
				progress = Policy.monitorFor(progress);
				if (FTPPlugin.getPlugin().isFetchRemoteTimestampImmediately()) {
					progress.beginTask(null, 80);
				} else {
					progress.beginTask(null, 100);
				}
				// upload the resources
				FTPTargetProvider.super.put(resources, Policy.subMonitorFor(progress, 80));
				// get the remote identifiers if required
				if (!FTPPlugin.getPlugin().isFetchRemoteTimestampImmediately()) {
					provideRemoteIdentifiers(Policy.subMonitorFor(progress, 20));
				}
			}
		}, progress);
	}

	/**
	 * Record the state so that it's remote timestamp can be fetched at the end of the upload
	 * @param state
	 */
	public void needsRemoteIdentifier(IResource resource) {
		resourcesNeedingRemoteIdentifier.add(resource);
	}
	
	private void provideRemoteIdentifiers(IProgressMonitor progress) throws TeamException {
		// Get the remote identifiers for any resources that require it
		if ( ! resourcesNeedingRemoteIdentifier.isEmpty()) {
			try {
				// group the resources by parent directory
				final Map resourcesByParent = new HashMap();
				for (Iterator iter = resourcesNeedingRemoteIdentifier.iterator(); iter.hasNext();) {
					IResource resource = (IResource) iter.next();
					List children = (List) resourcesByParent.get(resource.getParent());
					if (children == null) {
						children = new ArrayList();
						resourcesByParent.put(resource.getParent(), children);
					}
					children.add(resource);
				}
				// get the child states through each parent so the timestamps will all be fetched at once
				// XXX Needs proper progress
				IProgressMonitor noProgress = new NullSubProgressMonitor(progress);
				for (Iterator iter = resourcesByParent.keySet().iterator(); iter.hasNext();) {
					IContainer parent = (IContainer) iter.next();
					List children = (List) resourcesByParent.get(parent);
					ResourceState parentState = newState(parent);
					ResourceState[] childStates = parentState.getRemoteChildren(noProgress);
					for (int i = 0; i < childStates.length; i++) {
						FTPResourceState state = (FTPResourceState) childStates[i];
						if (children.contains(state.getLocal())) {
							state.recordReleasedIdentifier(noProgress);
						}
					}
				}
			} finally {
				// remove the processed nodes from the list
				resourcesNeedingRemoteIdentifier.clear();
			}
		}
	}
}
