/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.core;

import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.target.IRemoteTargetResource;
import org.eclipse.team.internal.core.target.ResourceState;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.core.target.SynchronizedTargetProvider;
import org.eclipse.webdav.client.DAVClient;
import org.eclipse.webdav.client.RemoteDAVClient;
import org.eclipse.webdav.client.WebDAVFactory;
import org.eclipse.webdav.http.client.HttpClient;

public class WebDavProvider extends SynchronizedTargetProvider {
	
	// Re-use the http client
	private static HttpClient httpClient = new HttpClient();
	
	private DAVClient davClient;
	
	public WebDavProvider(Site site, IPath intrasitePath) throws TeamException {
		super(site, intrasitePath);
		initializeDavClient();
	}
	
	private void initializeDavClient() throws TeamException {
		WebDavSite site = (WebDavSite)getSite();

		// Authentication.
		String username = site.getUsername();
		String password = site.getPassword();
		if ((username != null) && (password != null))
			httpClient.setAuthenticator(new Authenticator(username, password));

		// Proxy server.
		String proxy = site.getProxy();
		if (proxy != null && proxy.length() > 0) {
			try {
				httpClient.setDefaultProxyServerUrl(new URL(proxy));
			} catch (MalformedURLException exception) {
				// Setting the proxy fails if the URL is invalid.
				throw new TeamException(Policy.bind("Config.error"), exception); //$NON-NLS-1$
			}
		}

		// Timeout converted to miliseconds.
		int timeout = site.getTimeout() * 1000;
		// connection recycler timeout and socket timeout being set to the same
		// value.
		httpClient.setConnectionTimeout(timeout);
		httpClient.setSoTimeout(timeout);

		// Can add other configs here later.
		davClient = new RemoteDAVClient(new WebDAVFactory(), httpClient);
	}
	
	/**
	 * Answer a new resource state object.
	 * 
	 * @param resource the managed resource whose state descriptor is sought.
	 * @return a state description.
	 */
	public ResourceState newState(IResource resource) {
		return new DavState(davClient, getSite(), getURL(), resource);
	}
	
	/**
	 * Answer a new resource state object.
	 * 
	 * @param resource the managed resource whose state descriptor is sought.
	 * @param remote the remote resource to be associated with the resource.
	 * @return a state description.
	 */
	public ResourceState newState(IResource resource, IRemoteTargetResource remote) {
		return new DavState(davClient, resource, (DavRemoteTargetResource)remote);
	}

	/**
	 * @see TargetProvider#getRemoteResourceFor(IResource)
	 */
	public IRemoteTargetResource getRemoteResourceFor(IResource resource) {
		return new DavRemoteTargetResource(davClient, getSite(), getURL(), resource.getProjectRelativePath(), resource.getType() != IResource.FILE);
	}
	
	/**
	 * @see TargetProvider#getRemoteResource()
	 */
	public IRemoteTargetResource getRemoteResource() {
		return new DavRemoteTargetResource(davClient, getSite(), getURL(), Path.EMPTY, true /* is a container handle */);
	}
}
