/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.util.misc;

import java.util.Set;
import java.lang.reflect.Constructor;

import com.sun.appserv.management.util.misc.GSetUtil;



/**
	Maps a Throwable to another one in order to avoid the transfer
	of non-standard (proprietary) Exception types, which could result in
	ClassNotFoundException on remote clients.
	<p>
	Any Throwable which either is, or contains,
	a Throwable which is not in the allowed packages is converted.
 */
public final class ThrowableMapper 
{
	final Throwable	mOriginal;
	
	final Set<String>	mOKPackages;
	
	/**
		By default, any Throwable whose package does not start with one
		of these packages must be mapped to something standard.
	 */
	protected final static Set<String> OK_PACKAGES	=
			GSetUtil.newUnmodifiableStringSet( "java.", "javax." );
	
		public
	ThrowableMapper( final Throwable t )
	{
		mOriginal	= t;
		mOKPackages	= OK_PACKAGES;
	}
	
		private static boolean
	shouldMap( final Throwable t )
	{
		final String tClass	= t.getClass().getName();
		
		boolean	shouldMap	= true;
		
		for( final String prefix : OK_PACKAGES )
		{
			if ( tClass.startsWith( prefix ) )
			{
				shouldMap	= false;
				break;
			}
		}
		
		return( shouldMap );
	}
	
	
		public static Throwable
	map( final Throwable t )
	{
		Throwable		result	= t;
		
		if ( t != null )
		{
			final Throwable tCause			= t.getCause();
			final Throwable tCauseMapped	= map( tCause );
			
			// if either this Exception or its cause needs/was mapped,
			// then we must form a new Exception
			
			if ( shouldMap( t ) )
			{
			    // the Throwable itself needs to be mapped
				final String msg	=t.getMessage();
				
				if ( t instanceof Error )
				{
					result	= new Error( msg, tCauseMapped );
				}
				else if ( t instanceof RuntimeException )
				{
					result	= new RuntimeException( msg, tCauseMapped );
				}
				else if ( t instanceof Exception )
				{
					result	= new Exception( msg, tCauseMapped );
				}
				else
				{
					result	= new Throwable( msg, tCauseMapped );
				}
				
				result.setStackTrace( t.getStackTrace() );
			}
			else if ( tCauseMapped != tCause )
			{
			    // the Throwable doesn't need mapping, but its Cause does
			    // create a Throwable of the same class, and insert its
			    // cause and stack trace.
			    try
			    {
			        final Constructor<? extends Throwable> c =
			            t.getClass().getConstructor( String.class, Throwable.class );
			        result  = c.newInstance( t.getMessage(), tCauseMapped);
			    }
			    catch( final Throwable t1 )
			    {
			        try
			        {
    			        final Constructor<? extends Throwable> c =
    			            t.getClass().getConstructor( String.class );
			            result  = c.newInstance( t.getMessage() );
			            result.initCause( tCauseMapped );
			        }
			        catch( final Throwable t2 )
			        {
			            result  = new Throwable( t.getMessage(), tCauseMapped );
			        }
			    }
			    
			    result.setStackTrace( tCause.getStackTrace() );
			}
			else
			{
				result	= t;
			}
		}

		return( result );
	}
	
	/**
		Map the original Throwable to one that is non-proprietary (standard).
		Possible results include java.lang.Exception, java.lang.RuntimeException,
		java.lang.Error.  The original stack trace and exception chain is
		preserved, each element in that chain being mapped if necessary.
		
		@return a Throwable which uses only standard classes
	 */
		public Throwable
	map()
	{
		return( map( mOriginal ) );
	}
}








