/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 
/*
 * $Header: /cvs/glassfish/admin/mbeans/src/java/com/sun/enterprise/admin/dottedname/DottedNameAliasedQuery.java,v 1.3 2005/12/25 03:42:02 tcfujii Exp $
 * $Revision: 1.3 $
 * $Date: 2005/12/25 03:42:02 $
 */
package com.sun.enterprise.admin.dottedname;

import java.util.Collections;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import javax.management.ObjectName;

/*
	Implements DottedNameQuery over config dotted names by presenting a view
	of user-visible dotted names even though the underlying dotted names are
	in fact not user-visible.  For PE, this means that dotted names start with
	the server name "server", but in actuality they are registered starting
	with the config name.  This class hides that detail.
	
	Additionally, domain.* is also aliased into server
 */
public class DottedNameAliasedQuery implements DottedNameQuery
{
	protected final DottedNameQuery				mSrcQuery;
	protected final DottedNameServerInfo			mServerInfo;
	final DottedNameResolverForAliases	mAliasResolver;
	
		public
	DottedNameAliasedQuery( final DottedNameQuery srcQuery, final DottedNameServerInfo serverInfo )
	{
		mSrcQuery	= srcQuery;
		mServerInfo	= serverInfo;
		mAliasResolver	= new DottedNameResolverForAliases( srcQuery, serverInfo );
	}
	
		public ObjectName
	dottedNameToObjectName( final String	dottedName )
	{
		return( mAliasResolver.resolveDottedName( dottedName ) );
	}
	
		public Set
	allDottedNameStrings(  )
	{
		Set	result	= Collections.EMPTY_SET;
		
		try
		{
			result	= allDottedNameStringsThrow();
		}
		catch( DottedNameServerInfo.UnavailableException e )
		{
			DottedNameLogger.logException( e );
		}
		return( result );
	}
	
	/*
		Return a set of all dotted names *as visible to the user*.
		
		This means that we convert any dotted names beginning with a config name
		to dotted names beginning with its corresponding server name.
		
		Certain domain names are also aliased into the server name.
	 */
		protected java.util.Set
	allDottedNameStringsThrow(  )
		throws DottedNameServerInfo.UnavailableException
	{
		final Set		srcSet	= mSrcQuery.allDottedNameStrings();
		final Iterator	iter	= srcSet.iterator();
		final HashSet	destSet	= new HashSet();
		
		final Set	configNames	= mServerInfo.getConfigNames();
		
		while ( iter.hasNext() )
		{
			final String		dottedName	= (String)iter.next();
			final DottedName	dn	= DottedNameFactory.getInstance().get( dottedName );
			
			final String	scope	= dn.getScope();
			
			if ( DottedNameAliasSupport.scopeIsDomain( scope ) )
			{
				if ( DottedNameAliasSupport.isAliasedDomain( dn ) )
				{
					destSet.add (dottedName);
                    addAllNamesForDomain( dn, destSet );
				}
			}
			else
			{
				if ( configNames.contains( scope ) )
				{
					addAllNamesForConfig( dn, destSet );
				}
				else
				{
					// not a config name.
					destSet.add( dottedName );
				}
			}
			
		}
		
		return( destSet );
	}
	
	/*
		Given a config dotted name, generate and add all corresponding dotted
		names that start with the server name (just one in PE).  (In SE/EE there
		could be more than one server using the same config).
	 */
    protected void
	addAllNamesForDomain( final DottedName domainDN, final Set outSet )
		throws DottedNameServerInfo.UnavailableException
	{
		final Iterator iter	= mServerInfo.getServerNames().iterator();
		
		// there may be none if no servers refer to the config--that's OK
		while ( iter.hasNext() )
		{
			final String	serverName	= (String)iter.next();
			
			final String		dottedNameString	=
					DottedName.toString( domainDN.getDomain(), serverName, domainDN.getParts() );
					
			final DottedName	newName	= DottedNameFactory.getInstance().get( dottedNameString );
			outSet.add( newName.toString() );
		}
	}
	
	/*
		Given a config dotted name, generate and add all corresponding dotted
		names that start with the server name (just one in PE).  (In SE/EE there
		could be more than one server using the same config).
	 */
    protected void
	addAllNamesForConfig( final DottedName configDN, final Set outSet )
		throws DottedNameServerInfo.UnavailableException
	{
		final String []	serverNames	= mServerInfo.getServerNamesForConfig( configDN.getScope() );
		
		// there may be none if no servers refer to the config--that's OK
		for( int i = 0; i < serverNames.length; ++i )
		{
			final String		newName	=
					DottedName.toString( configDN.getDomain(), serverNames[ i ], configDN.getParts() );

			outSet.add( newName );
		}
	}
}


