/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * $Id: DomainMBean.java,v 1.3 2005/12/25 03:42:16 tcfujii Exp $
 */

package com.sun.enterprise.admin.mbeans;

//jdk imports
import java.lang.reflect.Method;
import java.io.File;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

//JMX imports
import javax.management.AttributeList;

// commons imports
import com.sun.enterprise.util.SystemPropertyConstants;
import com.sun.enterprise.util.OS;
import com.sun.enterprise.util.i18n.StringManager;

//config imports
import com.sun.enterprise.admin.config.BaseConfigMBean;
import com.sun.enterprise.admin.config.ConfigMBeanHelper;
import com.sun.enterprise.admin.config.MBeanConfigException;
import com.sun.enterprise.admin.meta.MBeanRegistryEntry;
import com.sun.enterprise.admin.meta.naming.MBeanNamingDescriptor;

//core imports
import com.sun.enterprise.instance.InstanceEnvironment;
import com.sun.enterprise.server.ApplicationServer;

import com.sun.enterprise.admin.mbeanapi.IDomainMBean;
import com.sun.enterprise.config.serverbeans.PropertyResolver;

// Logging
import java.util.logging.Level;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.admin.meta.MBeanRegistryFactory;


public class DomainMBean extends BaseConfigMBean
    implements IDomainMBean
{
    private static final StringManager localStrings =
            StringManager.getManager(DomainMBean.class);
    
    /**
     * Resolve tokens of the form ${..} in value.
     * @param value string value to resolve
     * @param instanceName - instance name for obtaining token values. If null, then DAS is used for resolution 
     *
     * @returns resolved string
     *
     */
    public String resolveTokens(String value, String instanceName) throws ConfigException
    {
        return resolveTokens(value, instanceName, false);
    }
    
    /**
     * Resolve tokens of the form ${..} in value.
     * @param value string value to resolve
     * @param instanceName - instance name for obtaining token values. If null, then DAS is used for resolution 
     * @param bResolvePathesAsWell - if true, resolves pathes as well (currently only for DAS!)
     *
     * @returns resolved string
     *
     */
    public String resolveTokens(String value, String instanceName, boolean bResolvePathesAsWell) throws ConfigException
    {
        //FIXME: bResolvePathesAsWell=true is working correct only for DAS 
        if(instanceName==null)
        {
            instanceName=MBeanRegistryFactory.getAdminContext().getServerName();
        }
        PropertyResolver resolver = new PropertyResolver(getConfigContext(), instanceName);
        return resolver.resolve(value, bResolvePathesAsWell);
    }
    /*
     * This operation allows to obtain third part software properties set such as jdbc-resources, connectors etc.
     * Since this set depend on what kind (brand) of software it is for, attributeList parameter defines attributes
     * for undesrstanding what kind of software used.
     * This operation suggests that correspondent MBeans' classes override static getDefaultCustomProperties(attributeList)
     * "mbeanTypeName" - parameter is typeName from mbeans descriptors file.
     */
    public AttributeList getDefaultCustomProperties(String mbeanTypeName, AttributeList attributeList)
    {
        if(mbeanTypeName==null)
            return null;
        try {
            MBeanRegistryEntry entry = m_registry.findMBeanRegistryEntryByType(mbeanTypeName);
            MBeanNamingDescriptor descr = entry.getNamingDescriptor();
            String className = descr.getMBeanClassName();
            Class cl = Class.forName(className);
            Method method = cl.getDeclaredMethod("getDefaultCustomProperties", new Class[]{Class.forName("javax.management.AttributeList")});
            return (AttributeList)method.invoke(null, new Object[]{attributeList});
        } catch (Exception e)
        {
            _sLogger.fine("getDefaultCustomProperties(): Exception for mbeanTypeName:"+mbeanTypeName);
            return null;
        }
            
    }
    
    /*
     * Returns default values for attributes for mbean. 
     * If Custom MBean class implents public static getDefaultAttributeValues(String[]) then it will be called,
     * otherwise returns DTD defined default values (from ConfigBeans)  
     *
     * if attrNames is null - all default attributes values are returning
     */
    public AttributeList getDefaultAttributeValues(String mbeanTypeName, String attrNames[])
    {
        if(mbeanTypeName==null)
            return null;
        try {
            MBeanRegistryEntry entry = m_registry.findMBeanRegistryEntryByType(mbeanTypeName);
            if(attrNames==null)
            {
                attrNames = entry.getAttributeNames();
                if(attrNames==null || attrNames.length<1)
                    return null;
            }
            MBeanNamingDescriptor descr = entry.getNamingDescriptor();
            String className = descr.getMBeanClassName();
            Class cl = Class.forName(className);
            Method method = null;
            try {
                method = cl.getDeclaredMethod("getDefaultAttributeValues", new Class[]{(new String[0]).getClass()});
                return (AttributeList)method.invoke(null, new Object[]{attrNames});
            } catch (Exception e)
            {
                //no custom implementation - just ignore
            }
            // standard 
            return ConfigMBeanHelper.getDefaultAttributeValues(descr, attrNames); 
        } catch (Exception e)
        {
            _sLogger.fine("getDefaultAttributeValues(): Exception for mbeanTypeName:"+mbeanTypeName);
            return null;
        }
    }

     /**
     * Returns the absolute path of the config directory.
     *
     * @returns the absolute path of the config directory.
     */
    public String getConfigDir() {
        InstanceEnvironment env = 
                ApplicationServer.getServerContext().getInstanceEnvironment();
        return env.getConfigDirPath();
    }
  
    private static final String BUNDLED_DOMAINS_ROOT = "/var/appserver/domains";

    private static final String AUTOSTART_FILENAME = "autostart";

    /**
     * Is autostart feature supported for this domain. Enabling autostart will
     * result in a domain being started up at the time of machine startup
     * (boot). The autostart feature is supported only on domains in default
     * domain directory of Solaris bundled release.
     * @returns true if autostart feature is supported for this domain.
     */
    public boolean isAutoStartSupported() {
        if (OS.isUnix()) {
            if (getConfigDir().startsWith(BUNDLED_DOMAINS_ROOT)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Is auto start enabled for this domain. 
     * @throw MBeanConfigException if autostart is not supported for this domain
     */
    public boolean isAutoStartEnabled() throws MBeanConfigException {
        checkAutoStartSupported();
        File autoStartFile = getAutoStartFile();
        return autoStartFile.exists();
    }

    /**
     * Set autostart enabled state.
     * @param state if true enables autostart, otherwise disables autostart
     * @throw MBeanConfigException if autostart is not supported or if there
     *         is an error in setting autostart state.
     */
    public void setAutoStartEnabled(boolean state) throws MBeanConfigException {
        checkAutoStartSupported();
        boolean success = (state ? enableAutoStart() : disableAutoStart());
        if (!success) {
            String msg = localStrings.getString(
                    "admin.mbeans.domain.set_autostart_failed");
            throw new MBeanConfigException(msg);
        }
    }

    /**
     * Returns the name of this domain.
     *
     * @return  domain name
     */
    public String getName() throws MBeanConfigException {

        String name = null;

        try {
            name = System.getProperty(SystemPropertyConstants.DOMAIN_NAME);
        } catch (Exception e) {
            String msg = localStrings.getString(
                    "admin.mbeans.domain.get_name_failed")
                    + " " + e.getLocalizedMessage();
            throw new MBeanConfigException(msg);
        }

        if (name == null) {
            String msg = localStrings.getString(
                    "admin.mbeans.domain.get_name_failed");
            throw new MBeanConfigException(msg);
        }

        return name;
    }

    private boolean enableAutoStart() {
        File autoStartFile = getAutoStartFile();
        boolean success = false;
        try {
            if (!autoStartFile.exists()) {
                success = autoStartFile.createNewFile();
            } else {
                success = true;
            }
        } catch (IOException ioe) {
            _sLogger.log(Level.FINE, "mbean.autostart_ioexception", ioe);
            _sLogger.log(Level.WARNING, "mbean.autostart_enable_error",
                    new Object[] {autoStartFile, ioe.getMessage()});
        }
        return success;
    }

    private boolean disableAutoStart() {
        File autoStartFile = getAutoStartFile();
        boolean success = true;
        if (autoStartFile.exists()) {
            success = autoStartFile.delete();
        }
        return success;
    }

    private void checkAutoStartSupported() throws MBeanConfigException {
        if (!isAutoStartSupported()) {
            String msg = localStrings.getString(
                    "admin.mbeans.domain.autostart_not_supported");
            throw new MBeanConfigException(msg);
        }
    }

    private File getAutoStartFile() {
        return new File(getConfigDir(), AUTOSTART_FILENAME);
    }
}
