/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.enterprise.admin.mbeans.custom.loading;
import java.util.logging.Logger;
import javax.management.MBeanServer;
import javax.management.InstanceAlreadyExistsException;
import javax.management.MBeanRegistrationException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;

import com.sun.enterprise.admin.mbeans.custom.CMBStrings;
import com.sun.enterprise.admin.mbeans.custom.ObjectNameSelectionAlgorithm;
import com.sun.enterprise.config.serverbeans.Mbean;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.admin.common.constant.AdminConstants;
import com.sun.enterprise.admin.mbeans.custom.CustomMBeanConstants;
import com.sun.enterprise.admin.server.core.CustomMBeanRegistration;
import com.sun.enterprise.util.SystemPropertyConstants;
import java.util.Hashtable;
import java.util.List;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanException;
import javax.management.ReflectionException;
import javax.management.RuntimeOperationsException;

/** Class to register all the MBeans defined in the server's configuration. Registering the MBeans is done 
 * as described in the design document. The self-management rules are to be taken into account for this.
 * As it stands now, (AS 9.0) this class is not designed to be thread-safe.
 * The calling code must ensure serial access if needed.
 * @since SJSAS 9.0
 */
public final class CustomMBeanRegistrationImpl implements CustomMBeanRegistration {
    
    /** Creates a new instance of CustomMBeanRegistrar */
    private final MBeanServer mbs;
    private ClassLoader cl;
    private static final Logger logger = Logger.getLogger(AdminConstants.kLoggerName);

    public CustomMBeanRegistrationImpl(final MBeanServer mbs) throws IllegalArgumentException {
        this.mbs    = mbs;
        this.cl     = new MBeanClassLoader();
    }
    
    
    public void registerMBeans(final List<Mbean> mbeans, final boolean continueReg) throws RuntimeException {
        if (mbeans == null)
            throw new IllegalArgumentException(
                CMBStrings.get("InternalError", "registerMBeans() received a null mbeans argument"));
        for (Mbean mbean : mbeans) {
            try {
                registerMBean(mbean);
            } catch(final Throwable t) {
                if (continueReg) {
                    logger.info(CMBStrings.get("cmb.registerError", mbean.getName()));
                }
                else {
                    throw new RuntimeException(t);
                }
            }
        }
    }
    
    public void registerMBeans(final List<Mbean> mbeans) throws RuntimeException {
        this.registerMBeans(mbeans, true);
    }
    public void setClassLoader(final ClassLoader cl) throws IllegalArgumentException {
        if (cl == null)
            throw new IllegalArgumentException(CMBStrings.get("InternalError", "setClassLoader() received a null argument"));
        this.cl = cl;
    }
    public ObjectName registerMBean(final Mbean mbean) throws RuntimeException {
        /* This is the only place where the "registration of the MBean happens.
         */
        if (mbean == null)
            throw new IllegalArgumentException(CMBStrings.get("InternalError", "registerMBean() received a null argument"));
        
        ObjectName ron = null;
        try {
            logger.fine(CMBStrings.get("cmb.loadingMBean1", mbean.getName()));
            final ObjectName mon    = getCascadingAwareObjectName(mbean);
            logger.fine(CMBStrings.get("cmb.loadingMBean2", mon.toString()));
            final Class mc          = loadIt(mbean.getImplClassName());
            final Object mo         = newIt(mc);
            ron = registerIt(mo, mon);

            
            // if the MBean implements MBeanRegistration -- it can set the ON to 
            // anything at all...
            if(!ObjectNameSelectionAlgorithm.implementsMBeanRegistrationInterface(mbean.getImplClassName()))
            {
                if(!mon.equals(ron))
                    throw new RuntimeException(CMBStrings.get("objNameMismatch", mon, ron));
            }
            initIt(mbean, ron);
   
            // WBN 12-15-2005
            // this is just defensive programming -- the Listener should not be
            // calling us on a disabled mbean.  In the case we are being called
            // as part of pre-reg in order to get an objectname, this will make
            // one extra unneeded call to unregisterIt()
            // I say, safety first, performance second!
            if(!mbean.isEnabled())
                unregisterIt(mbean, ron);
            return ( ron );
        } catch (final ClassNotFoundException cnfe) {
            logger.info(CMBStrings.get("cmb.loadingMBean3", mbean.getImplClassName(), cl.getClass().getName()));
            throw new RuntimeException(cnfe);
        } catch (final NoClassDefFoundError ncdfe) {
            logger.info(CMBStrings.get("cmb.loadingMBean4", mbean.getImplClassName(), cl.getClass().getName()));
            throw new RuntimeException(ncdfe);
        } catch (final InstantiationException ie) {
            logger.info(CMBStrings.get("cmb.loadingMBean5", mbean.getImplClassName()));
            throw new RuntimeException(ie);
        } catch (final IllegalAccessException iae) {
            logger.info(CMBStrings.get("cmb.loadingMBean6", mbean.getImplClassName()));
            throw new RuntimeException(iae);
        } catch (final ExceptionInInitializerError eie) {
            logger.info(CMBStrings.get("cmb.loadingMBean7", mbean.getImplClassName()));
            throw new RuntimeException(eie);
        }
        catch (final Throwable e) {
            //Roll back if registration fails. Failure to initialize is failure to register.
            //e.printStackTrace();
            if (ron != null)
                unregisterIt(mbean, ron);
            throw new RuntimeException(e);
        }
    }
    
    public static ObjectName getCascadingAwareObjectName(final Mbean mbean) throws RuntimeException {
        try {
            final ObjectName configON   = new ObjectName(mbean.getObjectName());
            return (getCascadingAwareObjectName(configON) );
        } catch(final Exception e) {
            throw new RuntimeException(e);
        }
    }
    public static ObjectName getCascadingAwareObjectName(final ObjectName configON) throws RuntimeException {
        try {
            final String serverNameKey  = CustomMBeanConstants.SERVER_KEY;
            final String serverNameVal  = System.getProperty(SystemPropertyConstants.SERVER_NAME);
            final Hashtable properties  = configON.getKeyPropertyList();
            properties.put(serverNameKey, serverNameVal);
            final ObjectName ron = new ObjectName(configON.getDomain(), properties);
            return ( ron );
        } catch(final Exception e) {
            throw new RuntimeException(e);
        }
    }
    public static ObjectName getCascadingUnawareObjectName(final ObjectName cascadeON) throws RuntimeException {
        try {
            if (cascadeON == null) {
                throw new IllegalArgumentException(CMBStrings.get("InternalError", "getCascadingUnawareObjectName() received a null argument"));
            }
            ObjectName ron = cascadeON;
            final String serverNameKey  = CustomMBeanConstants.SERVER_KEY;
            final Hashtable properties  = cascadeON.getKeyPropertyList(); // this may be unmodifiable
            if (properties.containsKey(serverNameKey)) {
                final Hashtable np = new Hashtable(properties);
                np.remove(serverNameKey);
                ron = new ObjectName(cascadeON.getDomain(), np);
            }
            return ( ron );
        } catch(final Exception e) {
            throw new RuntimeException(e);
        }        
    }
    ////////// Private Methods ///////
    private Class loadIt(final String classname) throws ClassNotFoundException, NoClassDefFoundError {
        final Class c = cl.loadClass(classname);
        logger.fine(CMBStrings.get("cmb.loadingMBean8", c.getName(), cl.getClass().getName(), c.getClassLoader().getClass().getName()));
        return ( c );
    }
    private Object newIt(final Class c) throws IllegalAccessException, InstantiationException,
        ExceptionInInitializerError {
        return ( c.newInstance() );
    }
    private ObjectName registerIt(final Object mo, final ObjectName on) throws InstanceAlreadyExistsException,
        MBeanRegistrationException, NotCompliantMBeanException, RuntimeOperationsException {
        if(mo == null)
            throw new RuntimeException(CMBStrings.get("objNameNull"));
        final ObjectInstance oi = mbs.registerMBean(mo, on);
        return ( oi.getObjectName() );
    }
    private void initIt(final Mbean mbc, final ObjectName on) throws InstanceNotFoundException, AttributeNotFoundException,
        InvalidAttributeValueException, MBeanException, ReflectionException {
        final MBeanAttributeSetter mas = new MBeanAttributeSetter(mbs, on);
        final ElementProperty[] ats = mbc.getElementProperty();
        for (ElementProperty p : ats) {
            mas.setIt(p.getName(), p.getValue());
            logger.fine(CMBStrings.get("cmb.initMBean",  p.getName(), mbc.getName()));
        }
    }
    private void unregisterIt(final Mbean m, final ObjectName ron) {
        //attempt to unregister the mbean, not being able to do so is not fatal
        try {
            if (mbs.isRegistered(ron))
                mbs.unregisterMBean(ron);
        } catch (final Throwable e) { 
            logger.warning(CMBStrings.get("cmb.unloadMBeanError", m.getName()));
        }
    }
    ////////// Private Methods ///////
}