/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.enterprise.jbi.serviceengine.install;

import com.sun.enterprise.jbi.serviceengine.ServiceEngineException;
import com.sun.enterprise.server.ApplicationServer;
import com.sun.logging.LogDomains;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URI;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.management.ObjectName;



/**
 * Installs Java EE Service Engine on a CAS installation. In openESB world, an
 * instance could either be a CAS or a non CAS ESB member. Java EE Servijece Engine
 * is installed on a instance only if the instance is CAS. All components
 * installed on CAS are installed on a non CAS ESB member by the CAS.
 * @author Manishaa Umbar
 */
public class InstallerImpl implements Installer {
    
        
    /**
     * addComponent operation of ESB installation MBean adds the component
     * to ESB's registry and repository. By adding a component, component becomes
     * installable.
     */
    private static final String ADD_COMPONENT_OPERATION="addComponent";
    
    /**
     * Install a already added component in ESB environment
     */
    private static final String INSTALL_COMPONENT_OPERATION="installComponent";
    
    /**
     * removeComponent operation of ESB installation MBean removes the component
     * from ESB's registry and repository.
     */
    private static final String REMOVE_COMPONENT_OPERATION="removeComponent";
    
    /**
     * UnInstalls  already added component in ESB environment. By uninstalling a component,
     * component becomes removable from the registry and repository of CAS.
     */
    private static final String UNINSTALL_COMPONENT_OPERATION="uninstallComponent";
    
    /**
     * Starts the component
     */
    private static final String START_COMPONENT_OPERATION="startComponent";
    
    /**
     * Stops the component
     */
    private static final String STOP_COMPONENT_OPERATION="stopComponent";
    
    
    private String componentName = null;
    private String SE_BUNDLE = "lib/addons/jbi/appserv-jbise.jar";
    
    private String JBI_FOLDER = "jbi";
    
    private MBeanHelper mbeanHelper;
    
    private String jbiInstanceName;
    
    private boolean jbiInstalled = false;
    
    /**
     * Internal handle to the logger instance
     */
    private static Logger logger =
            LogDomains.getLogger(LogDomains.SERVER_LOGGER);
    
    /** Creates a new instance of CASInstallerImpl */
    public InstallerImpl(MBeanHelper mbeanHelper) {
        this.mbeanHelper = mbeanHelper;
        String installationRoot =
                ApplicationServer.getServerContext().getInstallRoot();
        String jbiInstallationDir = installationRoot + File.separator + JBI_FOLDER;
        boolean mbeanRegistered = false;
        try {
            mbeanRegistered = mbeanHelper.isMBeanRegistered(MBeanHelper.ESB_INSTALLATION_SERVICE);
        } catch(ServiceEngineException ex) {
            logger.log(Level.SEVERE, ex.getMessage(), ex);
        }
        jbiInstalled = new File(jbiInstallationDir).exists() && mbeanRegistered;
    }
    
    public boolean isJBIInstalled() {
        return jbiInstalled;
    }
    
    public void setComponentName(String componentName) {
        this.componentName = componentName;
    }
    
    
    /**
     * Installs the service engine and starts it
     * @param zipFilePath packaged 208 compliant service engine bundle
     *
     */
    public String install(String zipFilePath) throws ServiceEngineException {
        //Use InstallationServiceMBean to install new component
        String result = null;
        if(zipFilePath == null) {
            zipFilePath = getServiceEngineBundle();
                log(Level.FINE, "Java EE Service Engine Bundle :" , zipFilePath);
                try {
                    
                    ObjectName objName = mbeanHelper.getObjectName(
                            MBeanHelper.ESB_INSTALLATION_SERVICE);
                    
                    log(Level.FINEST, "installation_service_log_name" , objName.toString());
                    
                    result = (String)mbeanHelper.invokeMBeanOperation(objName,
                            ADD_COMPONENT_OPERATION, new Object[]{zipFilePath},
                            new String[] {"java.lang.String"});
                            
                    log(Level.FINEST, " Status of addComponent ", result );
                            
                    result = (String)mbeanHelper.invokeMBeanOperation(objName,
                                    INSTALL_COMPONENT_OPERATION,
                                    new Object[]{componentName},
                                    new String[] {"java.lang.String"});
                                    
                    log(Level.FINEST, " Status of installComponent ", result );
                } catch(Exception e) {
                    log(Level.SEVERE,
                            "Error occurred during installation of Java EE Service Engine",
                            e.getMessage());
                }
        }
        return result;
        
    }
    
    /**
     * Starts the component with provided name
     */
    public void start() throws ServiceEngineException {
            try {
                ObjectName objName = mbeanHelper.getObjectName(
                        MBeanHelper.ESB_LIFECYCLE_SERVICE);
                
                log(Level.FINEST, "lifecycle_service_obj_name" , objName.toString());
                
                String result = (String)mbeanHelper.invokeMBeanOperation(objName,
                        START_COMPONENT_OPERATION, new Object[]{componentName},
                        new String[] {"java.lang.String"});
                        log(Level.FINEST, "Start Component Status", result);
            } catch(Exception e) {
                log(Level.SEVERE,
                        "Error occurred during startup of Java EE Service Engine",
                        e.getMessage());
            }
    }
    
    
    /**
     * Checks if the compoenent specified by componentName is installed or
     * not
     */
    public boolean isComponentInstalled() {
            try {
                String domainDir = ApplicationServer.getServerContext().getInstanceEnvironment().getInstancesRoot();
                String fs = File.separator;
                String javaeeSEDir = domainDir + fs + "jbi" + fs + "engines" + fs + "JavaEEServiceEngine";
                if(!(new File(javaeeSEDir).exists())) {
                    return false;
                }
            } catch(Exception e) {
                log(Level.WARNING, "Exception occurred while getting component by name", e.getMessage());
                return false;
            }
        
        return true;
    }
    
    /**
     * Stops the component with provided name
     */
    public void stop() throws ServiceEngineException {
            try {
                
                ObjectName objName = mbeanHelper.getObjectName(
                        MBeanHelper.ESB_LIFECYCLE_SERVICE);
                
                log(Level.FINEST, "lifecycle_service_obj_name" , objName.toString());
                
                String result = (String)mbeanHelper.invokeMBeanOperation(objName,
                        STOP_COMPONENT_OPERATION, new Object[]{componentName},
                        new String[] {"java.lang.String"});
                        log(Level.FINEST, "Start Component Status", result);
            } catch(Exception e) {
                log(Level.SEVERE,
                        "Error occurred during stopping of Java EE Service Engine",
                        e.getMessage());
            }
        
    }
    
    /**
     * Uninstalls the component with provided name
     */
    public void uninstall() throws ServiceEngineException {
            try {
                
                ObjectName objName = mbeanHelper.getObjectName(
                        MBeanHelper.ESB_INSTALLATION_SERVICE);
                
                log(Level.FINEST, "installation_service_log_name" , objName.toString());
                
                String result = (String)mbeanHelper.invokeMBeanOperation(objName,
                        UNINSTALL_COMPONENT_OPERATION, new Object[]{componentName},
                        new String[] {"java.lang.String"});
                        
                        log(Level.FINEST, " Status of uninstallComponent ", result );
                        
                        result = (String)mbeanHelper.invokeMBeanOperation(objName,
                                REMOVE_COMPONENT_OPERATION,
                                new Object[]{componentName},
                                new String[] {"java.lang.String"});
                                
                                log(Level.FINEST, " Status of removeComponent ", result );
            } catch(Exception e) {
                log(Level.SEVERE,
                        "Error occurred during uninstallation of Java EE Service Engine",
                        e.getMessage());
            }
    }
    
    private String getServiceEngineBundle() {
        
        String seBundle = System.getProperty("com.sun.aas.installRoot") + 
                    File.separator + SE_BUNDLE;
        return seBundle;
    }
    
    
    private void log(Level level, String property, String logString) {
        logger.log(level,property,logString);
    }
    
}
