/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.jbi.serviceengine.util.soap;

import java.util.*;
import javax.wsdl.*;
import javax.wsdl.extensions.*;
import javax.wsdl.extensions.soap.*;
import java.util.logging.*;
import javax.xml.soap.SOAPMessage;
import javax.xml.namespace.QName;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;

import com.sun.logging.LogDomains;

/**
 * Represents wsdl metadata of an endpoint. This contains
 * partially resolved operation names, which will be compared
 * with the soap message for getting the exact operation name.
 *
 * @author Binod PG
 */
public class EndpointMetaData {
    
    /**
     * Internal handle to the logger instance
     */
    protected static final Logger logger =
            LogDomains.getLogger(LogDomains.SERVER_LOGGER);

    private Definition def = null;
    private QName serviceName = null;
    private String epName = null;   

    private OperationMetaData operationMetaData; // saves the meta-data in case of only one operation per port.
    private OperationMetaData[] opmds = null;

    /**
     * Constructor. Saves required information for resolving the operation.
     */
    public EndpointMetaData(Definition def, QName serviceName, String epName) {
        this.def = def;
        this.serviceName = serviceName;
        this.epName = epName;   
    }

    /**
     * Constructor without service name and endpoint name.
     * Such an endpoint metadata is just a holder object 
     * for wsdl definition object.
     */
    public EndpointMetaData(Definition def) {
        this.def = def;
    }

    /**
     * Partially resolve the operation name.
     *
     * 1. If there is only one operation, use that as the operation name.
     * 2. If there is more than one operation name, save the input parameter
     *    for matching with the soap message.
     * 3. Since we want to work with any kind of binding, we need to only 
     *    consider the abstract WSDL.
     */
    public void resolve() {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Resolving the WSDL for : " + serviceName + " : " + epName);
        }
        Binding binding = getBinding(def, serviceName, epName);
        javax.wsdl.Operation[] ops = getOperations(binding);
        if (ops.length == 1) {
            operationMetaData = new OperationMetaData();
            operationMetaData.setOperationName(ops[0].getName());
            operationMetaData.setOneWay(javax.wsdl.OperationType.ONE_WAY.equals(ops[0].getStyle()));
        } else if (ops.length > 1) {
            opmds = new OperationMetaData[ops.length];
            int i = 0 ;
            for (javax.wsdl.Operation op : ops) {
                 Message input = null;
                 if (op.getInput() != null) {
                     input = op.getInput().getMessage();
                 }
                 Set s = getInputPartNames(input);
                 OperationMetaData md = new OperationMetaData();
                 md.setOperationName(op.getName());
                 md.setInputParameters(s);
                 md.setOneWay(javax.wsdl.OperationType.ONE_WAY.equals(op.getStyle()));
                 opmds[i++] = md;
            }
        } else {
            //C'mon we dont have an operation ?
            throw new RuntimeException("WSDL operation not resolved");
        }
    }

    public OperationMetaData getOperationMetaData(SOAPMessage soapMsg) {
        if (this.operationMetaData != null) {
            return this.operationMetaData;
        } 
        
        List nodeNames = new ArrayList();
        try {
            javax.xml.soap.SOAPBody soapBody = soapMsg.getSOAPBody();
            NodeList nl = soapBody.getChildNodes();

            for (int i = 0; i < nl.getLength(); i++) {
                Node n = nl.item(i);

                if (Node.ELEMENT_NODE == n.getNodeType()) {
                    String nodeName = n.getLocalName();
                    if (nodeName != null) {
                        nodeNames.add(nodeName);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        for (OperationMetaData om : opmds) {
            if (logger.isLoggable(Level.FINEST)) {
               logger.finest("Matching for" + om.getOperationName());
            }
            Set inputs = om.getInputParameters();
            if (logger.isLoggable(Level.FINEST)) {
                logger.finest("Inputs" + inputs);
                logger.finest("Nodes" + nodeNames);
            }
            if (inputs.containsAll(nodeNames)) {
                return om;
            }
        }
        return null;
    }

    public Definition getDefinition() {
        return def;
    }

    private Set getInputPartNames(Message msg) {
        Set set = new HashSet();

        if (logger.isLoggable(Level.FINE)) {
            logger.fine("Getting input parameters for: "+ msg);
        }
        if (msg != null) {
            Iterator bodyIterator = null;
            Map msgParts = msg.getParts();
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Message Parts are: "+ msgParts);
            }
            if (msgParts != null) {
                bodyIterator = msgParts.keySet().iterator();
            }
            // construct a set of expected names
            while (bodyIterator != null && bodyIterator.hasNext()) {
                String bodyPart = (String) bodyIterator.next();
                Part part = msg.getPart(bodyPart);
                if (part == null) {
                    throw new IllegalStateException("WSDL error");
                }
                QName typeQName = part.getTypeName();
                QName elemQName = part.getElementName();

                if (typeQName != null) {
                    // it uses type, so the root node name is the part name
                    set.add(part.getName());
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine("Added partName: "+ part.getName());
                    }
                } else if (elemQName != null) {
                    //it uses element, so the root node name is the element name
                    if (logger.isLoggable(Level.FINE)) {
                        logger.fine("Added root node: "+ elemQName.getLocalPart());
                        logger.fine("Part name is : " + part.getName());
                    }
                    set.add(elemQName.getLocalPart());
                } 
            }
        }

        return set;
    }

    public static javax.wsdl.Operation[] getOperations(Binding binding) {
        if (binding != null) {
            PortType pt = binding.getPortType();
            if (pt != null) {
                List l = pt.getOperations();
                if (l != null && l.size() > 0) {
                    return (javax.wsdl.Operation[]) 
                    l.toArray(new javax.wsdl.Operation[0]);
                }
            }
        }
        return null;
    }

    public static Binding getBinding(Definition def, QName serviceName, String endpointName) {
        String location = null;
        Service svc = def.getService(serviceName);
        if (svc == null) {
            return null;
        }
        Port port = svc.getPort(QName.valueOf(endpointName).getLocalPart());
        if (port == null) {
            return null;
        } else {
            return port.getBinding();
        }
    }

    /**
     * Class that holds operation and input parameters.
     */
    class OperationMetaData {
        private String operationName;
        private Set inputParams;
        private boolean oneWay;
        
        void setOperationName(String name) {
            this.operationName = name;
        }

        private void setInputParameters(Set params) {
            this.inputParams = params;
        }

        String getOperationName() {
            return this.operationName;
        }
        
        private Set getInputParameters() {
            return inputParams;
        }
        
        boolean isOneWay() {
            return oneWay;
        }
        
        void setOneWay(boolean oneWay) {
            this.oneWay = oneWay;
        }
    }
}
