/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.util;

import com.sun.enterprise.deployment.annotation.introspection.ClassFile;
import com.sun.enterprise.deployment.annotation.introspection.ConstantPoolInfo;
import com.sun.enterprise.deployment.annotation.introspection.CustomAnnotationScanner;
import com.sun.enterprise.deployment.annotation.introspection.EjbComponentAnnotationScanner;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.deploy.shared.FileArchive;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.IOException;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * Abstract superclass for specific types of annotation detectors.
 *
 * @author Qingqing Ouyang
 * @author tjquinn
 */
public abstract class AnnotationDetector {
    
    private ClassFile classFile = new ClassFile();

    public AnnotationDetector() {
        CustomAnnotationScanner scanner = createAnnotationScanner();
        ConstantPoolInfo poolInfo = new ConstantPoolInfo(scanner);
        classFile.setConstantPoolInfo(poolInfo);
    }

    /**
     * Provides a new annotation scanner, tailored to the particular type of
     * annotation detector the concrete subclass implements.
     * @return an implementation of CustomAnnotationScanner to be used by the detector
     */
    protected abstract CustomAnnotationScanner createAnnotationScanner();
    
    public boolean hasAnnotationInArchive(AbstractArchive archive) throws IOException {
        File file = new File(archive.getArchiveUri());
        if (!file.exists()) {
            throw new FileNotFoundException(archive.getArchiveUri());
        }

        //@@@ note that the two methods could probably be combined into one.
        //The challenge is to find the size of each entry without having to
        //read the entry twice, once for the inputstream, the other for size.
        if (file.isDirectory()) {
            return hasAnnotationInDirectory(archive);
        } else {
            return hasAnnotationInJar(archive);
        }
    }

    /**
     *@return true if the jar file contains any of the appropriate type of annotations
     */
    private boolean hasAnnotationInJar(AbstractArchive archive) throws IOException {
        JarFile jf = null;
        try {
            jf = new JarFile(new File(archive.getArchiveUri()));
            Enumeration<JarEntry> entriesEnum = jf.entries();
            while(entriesEnum.hasMoreElements()) {
                JarEntry je = entriesEnum.nextElement();
                if (je.getName().endsWith(".class")) {
                    if (containsAnnotation(jf, je)) {
                        return true;
                    }
                }
            }
        } finally {
            if (jf != null) {
                jf.close();
            }
        }
        return false;
    }

    /**
     *@return true if the directory contains any of the appropriate type of annotations
     */
    private boolean hasAnnotationInDirectory(AbstractArchive archive) throws IOException {
        Enumeration entriesEnum = archive.entries();
        while(entriesEnum.hasMoreElements()) {
            String entry = (String) entriesEnum.nextElement();
            if (entry.endsWith(".class")) {
                File file = null;
                int ind = entry.lastIndexOf("/");
                if (ind != -1) {
                    String entryName = entry.substring(ind + 1);
                    String parent = archive.getArchiveUri() + File.separatorChar + 
                                    entry.substring(0, ind);
                    file  = new File(parent.replace('/', File.separatorChar), entryName);
                } else {
                    file = new File(archive.getArchiveUri(), entry);
                }
                if (containsAnnotation(file)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    public boolean containsAnnotation(JarFile jarFile, JarEntry entry) throws IOException {
        boolean result = false;
        // check if it contains top level annotations...
        ReadableByteChannel channel = null;
        try {
            channel = Channels.newChannel(jarFile.getInputStream(entry));
            if (channel!=null) {
                result = classFile.containsAnnotation(channel, entry.getSize());
             }
             return result;
        } finally {
            if (channel != null) {
                channel.close();
            }
        }
    }
    
    public boolean containsAnnotation(File file) throws FileNotFoundException, IOException {
        boolean result = false;
        // check if it contains top level annotations...
        InputStream is = null;
        try {
            is = new BufferedInputStream(new FileInputStream(file));
            ReadableByteChannel channel = Channels.newChannel(is);
            if (channel!=null) {
                result = classFile.containsAnnotation(channel, file.length());
            }
            return result;
        } finally {
            if (is != null) {
                is.close();
            }
        }
    }
}
