/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment.util;

import com.sun.enterprise.deployment.annotation.introspection.ClassFile;
import com.sun.enterprise.deployment.annotation.introspection.ConstantPoolInfo;
import com.sun.enterprise.deployment.annotation.introspection.CustomAnnotationScanner;
import com.sun.enterprise.deployment.annotation.introspection.EjbComponentAnnotationScanner;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.deploy.shared.FileArchive;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.IOException;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * This is a utility class for detecting ejb component annotations.  This class
 * can be refactored to support other type of annotation dectection in the
 * future.
 *
 * @author Qingqing Ouyang
 */
public class EjbComponentAnnotationDetector {

    private ClassFile classFile = new ClassFile();

    public EjbComponentAnnotationDetector() {
        CustomAnnotationScanner scanner = new EjbComponentAnnotationScanner();
        ConstantPoolInfo poolInfo = new ConstantPoolInfo(scanner);
        classFile.setConstantPoolInfo(poolInfo);
    }

    public boolean hasAnnotationInArchive(AbstractArchive archive) throws IOException {
        File file = new File(archive.getArchiveUri());
        if (!file.exists()) {
            throw new FileNotFoundException(archive.getArchiveUri());
        }

        //@@@ note that the two methods could probably be combined into one.
        //The challenge is to find the size of each entry without having to
        //read the entry twice, once for the inputstream, the other for size.
        if (file.isDirectory()) {
            return hasAnnotationInDirectory(archive);
        } else {
            return hasAnnotationInJar(archive);
        }
    }

    /**
     *@return true if the jar file contains any ejb component annotations
     */
    private boolean hasAnnotationInJar(AbstractArchive archive) throws IOException {
        JarFile jf = null;
        try {
            jf = new JarFile(new File(archive.getArchiveUri()));
            Enumeration<JarEntry> entriesEnum = jf.entries();
            while(entriesEnum.hasMoreElements()) {
                JarEntry je = entriesEnum.nextElement();
                if (je.getName().endsWith(".class")) {
                    // check if it contains top level annotations...
                    ReadableByteChannel channel = Channels.newChannel(jf.getInputStream(je));
                    if (channel!=null) {
                        if (classFile.containsAnnotation(channel, je.getSize())) {
                            return true;
                        }
                     }
                }
            }
        } finally {
            if (jf != null) {
                jf.close();
            }
        }
        return false;
    }

    /**
     *@return true if the directory contains any ejb component annotations
     */
    private boolean hasAnnotationInDirectory(AbstractArchive archive) throws IOException {
        Enumeration entriesEnum = archive.entries();
        while(entriesEnum.hasMoreElements()) {
            String entry = (String) entriesEnum.nextElement();
            if (entry.endsWith(".class")) {
                File file = null;
                int ind = entry.lastIndexOf("/");
                if (ind != -1) {
                    String entryName = entry.substring(ind + 1);
                    String parent = archive.getArchiveUri() + File.separatorChar + 
                                    entry.substring(0, ind);
                    file  = new File(parent.replace('/', File.separatorChar), entryName);
                } else {
                    file = new File(archive.getArchiveUri(), entry);
                }
                // check if it contains top level annotations...
                InputStream is = null;
                try {
                    is = 
                        new BufferedInputStream(new FileInputStream(file));
                    ReadableByteChannel channel = Channels.newChannel(is);
                    if (channel!=null) {
                        if (classFile.containsAnnotation(channel, file.length())) {
                            return true;
                        }
                     }
                } finally {
                    if (is != null) {
                        is.close();
                    }
                }
            }
        }
        return false;
    }
}
