/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.ejb.containers;

import java.util.Date;
import java.util.Collection;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import java.io.Serializable;

import javax.ejb.EJBLocalObject;
import javax.ejb.TimerService;
import javax.ejb.Timer;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.CreateException;

import com.sun.enterprise.Switch;

/*
 * EJBTimerServiceWrappers is the application-level representation
 * of the EJB timer service. 
 *
 * @author Kenneth Saks
 */
public class EJBTimerServiceWrapper implements TimerService {

    private EJBTimerService timerService_;
    private EJBContextImpl ejbContext_;
    private long containerId_;

    private boolean entity_;

    // Only used for entity beans
    private Object timedObjectPrimaryKey_;

    public EJBTimerServiceWrapper(EJBTimerService timerService,
                                  EJBContextImpl ejbContext) 
    {
        timerService_ = timerService;
        ejbContext_   = ejbContext;
        BaseContainer container = (BaseContainer) ejbContext.getContainer(); 
        containerId_  = container.getEjbDescriptor().getUniqueId();
        entity_       = false;
        timedObjectPrimaryKey_   = null;
    }

    public EJBTimerServiceWrapper(EJBTimerService timerService,
                                  EntityContextImpl entityContext) 
    {
        this(timerService, ((EJBContextImpl)entityContext));
        entity_       = true;
        // Delay access of primary key since this might have been called 
        // from ejbCreate
        timedObjectPrimaryKey_   = null;
    }

    public Timer createTimer(long duration, Serializable info) 
        throws IllegalArgumentException, IllegalStateException, EJBException {

        checkCreateTimerCallPermission();

        if( duration < 0 ) {
            throw new IllegalArgumentException("invalid duration=" + duration);
        } 
                             
        TimerPrimaryKey timerId = null;

        try {
            timerId = timerService_.createTimer
                (containerId_, getTimedObjectPrimaryKey(), duration, 0, info);
        } catch(CreateException ce) {            
            EJBException ejbEx = new EJBException();
            ejbEx.initCause(ce);
            throw ejbEx;            
        }

        return new TimerWrapper(timerId, timerService_);
    }

    public Timer createTimer(long initialDuration, long intervalDuration, 
                             Serializable info) 
        throws IllegalArgumentException, IllegalStateException, EJBException {

        checkCreateTimerCallPermission();

        if( initialDuration < 0 ) {
            throw new IllegalArgumentException("invalid initial duration = " +
                                               initialDuration);
        } else if( intervalDuration < 0 ) {
            throw new IllegalArgumentException("invalid interval duration = " +
                                               intervalDuration);
        }
                             
        TimerPrimaryKey timerId = null;

        try {
            timerId = timerService_.createTimer
                (containerId_, getTimedObjectPrimaryKey(), initialDuration, 
                 intervalDuration, info);
        } catch(CreateException ce) {
            EJBException ejbEx = new EJBException();
            ejbEx.initCause(ce);
            throw ejbEx;                       
        }

        return new TimerWrapper(timerId, timerService_);
    }

    public Timer createTimer(Date expiration, Serializable info) 
        throws IllegalArgumentException, IllegalStateException, EJBException {
                             
        checkCreateTimerCallPermission();

        if( expiration == null ) {
            throw new IllegalArgumentException("null expiration");
        } 

        TimerPrimaryKey timerId = null;

        try {
            timerId = timerService_.createTimer(containerId_, 
                                                getTimedObjectPrimaryKey(),
                                                expiration, 0, info);
        } catch(CreateException ce) {
            EJBException ejbEx = new EJBException();
            ejbEx.initCause(ce);
            throw ejbEx;           
        }

        return new TimerWrapper(timerId, timerService_);
    }

    public Timer createTimer(Date initialExpiration, long intervalDuration,
                             Serializable info) 
        throws IllegalArgumentException, IllegalStateException, EJBException {

        checkCreateTimerCallPermission();

        if( initialExpiration == null ) {
            throw new IllegalArgumentException("null expiration");
        } else if ( intervalDuration < 0 ) {
            throw new IllegalArgumentException("invalid interval duration = " +
                                               intervalDuration);
        }

        TimerPrimaryKey timerId = null;
        try {
            timerId = timerService_.createTimer(containerId_, 
                getTimedObjectPrimaryKey(), initialExpiration, 
                intervalDuration, info);
        } catch(CreateException e) {
            EJBException ejbEx = new EJBException();
            ejbEx.initCause(e);
            throw ejbEx;                       
        }

        return new TimerWrapper(timerId, timerService_);
    }

    public Collection getTimers() throws IllegalStateException, EJBException {
        
        checkCallPermission();
        
        Collection timerIds = new HashSet();

        if( ejbContext_.isTimedObject() ) {        
            try {
                timerIds = timerService_.getTimerIds
                    (containerId_,  getTimedObjectPrimaryKey());
            } catch(FinderException fe) {
                EJBException ejbEx = new EJBException();
                ejbEx.initCause(fe);
                throw ejbEx;                         
            }
        } 
                                                        
        Collection timerWrappers = new HashSet();

        for(Iterator iter = timerIds.iterator(); iter.hasNext();) {
            TimerPrimaryKey next = (TimerPrimaryKey) iter.next();
            timerWrappers.add( new TimerWrapper(next, timerService_) );
        }

        return timerWrappers;
    }

    private Object getTimedObjectPrimaryKey() {
        if( !entity_ ) {
            return null;
        } else {
            synchronized(this) {
                if( timedObjectPrimaryKey_ == null ) {
                    timedObjectPrimaryKey_ = 
                        ((EntityContextImpl) ejbContext_).getPrimaryKey();
                }
            }
        }
        return timedObjectPrimaryKey_;
    }

    private void checkCreateTimerCallPermission() 
        throws IllegalStateException {
        if( ejbContext_.isTimedObject() ) {
            checkCallPermission();
        } else {
            throw new IllegalStateException("EJBTimerService.createTimer can "
                + "only be called from a timed object.  This EJB does not " 
                + "implement javax.ejb.TimedObject");                 
        }
    }

    private void checkCallPermission() 
        throws IllegalStateException {
        ejbContext_.checkTimerServiceMethodAccess();
    }

}
