/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.common.domains.registry;

import java.io.IOException;


import java.io.Serializable;


/**
 * Implementors of this interface represent a store which can be
 * locked and which will store and retrieve a single data object
 *
 * @author <a href="mailto:toby.h.ferguson@sun.com">Toby H Ferguson</a>
 * @version 1.0
 */
public interface LockingStore
{

	  /**
		 Write the given object to the store via serialization.
		 <p>
		 Precondition - store is locked
		 <p>
		 Postcondition - store contains this object, and only this
		 object. 
		 @param o the object to be put into the store. Must implement
		 {@link Serializable}
		 @throws IOException if there was a problem writing to the
		 store. Store will no longer be locked.
		 @throws IllegalStateException if the store is not
		 locked. Store will no longer be locked.
	  */
  void writeObject(Object o) throws IOException, IllegalStateException;
	  /**
		 Read the object from the store.
		 <p>
		 precondition - true
		 <p>
		 postcondition - store has not been modified
		 @return the single object that was in the store (or null if
		 the store is empty)
		 @throws TimeoutException if a lock on the store couldn't be
		 obtained in a reasonable time.
		 @throws IOException if there was a problem reading from the
		 store
		 @throws ClassNotFoundException if the object could not be
		 restored from the store
	  */
  Object readObject() throws IOException, TimeoutException, ClassNotFoundException;
	  /**
		 lock the store.
		 <p>
		 precondition - true
		 <p>
		 postcondition - the store is locked
		 @throws IOException if there was a problem initializing the
		 store
		 @throws TimeoutException if there was a problem obtaining a
		 lock in a reasonable amount of time
	  */
  void lock() throws IOException, TimeoutException;
	  /**
		 unlock the store.
		 <p>
		 precondition - store is not locked, or the store has been
		 locked by the caller.
		 <p>
		 postcondition - store is closed and all resources released
	  */
  void unlock();
	  /**
		 get the last time this store was modified
		 @return the time that this store was last modified, as per
		 {@link java.io.File#lastModified()}
	  */
  long lastModified();
}
