/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * StringValidator.java
 *
 * Created on April 12, 2001, 10:42 PM
 */
package com.sun.enterprise.admin.util;

/**
 * Class for validating String method parameters.  Current implementation's
 * stock object (getInstance()) verifies that the String exists and has length > 0
 * @author bnevins
 * @version $Revision: 1.3 $
 */

public class StringValidator extends BaseValidator 
{ 
    private final           int                 mMininumLength;
    private static final    int                 kDefaultMinimumLength   = 1; 
    private static          StringValidator     sDefaultInstance        = null; 
    private static final    String              badArgMessage           = 
        "You can't call StringValidator.validate() with a non-String argument";

    /** Create a StringValidator
     * @param minimumLength The String is invalid if its length is less than this
     */
    public StringValidator(int minimumLength) 
    { 
        Assert.assertRange(minimumLength, 0, Integer.MAX_VALUE, "minimumLength"); 
        mMininumLength = minimumLength; 
    } 

    /** Get the standard StringValidator
     * @return A class variable with a default minimum length of 1
     */	
    public static Validator getInstance()
    {
        // lazy creation...
        if(sDefaultInstance == null)
        {
            sDefaultInstance = new StringValidator(kDefaultMinimumLength);
        }
        return sDefaultInstance;
    } 

    /** Validate a String
     * @param obj The String to be validated
     * @return ValidatorResult is invalid if the String's length was
     * less than the minimum required length
     */
    public ValidatorResult validate(Object obj) 
    { 
        ValidatorResult result = super.validate(obj);

        if (result.isValid()) 
        { 
            Assert.assertit( (obj instanceof String), badArgMessage);

            final String 	str = (String)obj; 
            final int		len	= str.length();

            if(len < mMininumLength) 
            { 
                result = makeBadResult(len);
            } 
        } 
        return result; 
    } 

    private ValidatorResult makeBadResult(final int len) 
    {
        return new ValidatorResult(false, 
                        "The String argument is invalid.  The minimum required length is " +
                        mMininumLength + " and the String's actual length is " + len);
    }
}