
/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.connectors.util;

import com.sun.enterprise.deployment.*;
import com.sun.enterprise.deployment.archivist.*;
import com.sun.enterprise.deployment.deploy.shared.FileArchive;
import com.sun.enterprise.deployment.deploy.shared.MemoryMappedArchive;
import com.sun.enterprise.config.serverbeans.ElementProperty;
import com.sun.enterprise.connectors.*;
import com.sun.logging.LogDomains;

import java.util.logging.*;
import java.util.*;
import java.io.FileInputStream;
import java.io.IOException;

import org.xml.sax.SAXParseException;


/** This is an util class pertaining to the connector deployment descriptor
 *  i.e ra.xml and sun-ra.xml. This consist of methods to obtain the deployment
 *  descriptor and perform various transformations on that  to 
 *  obtain/constructs classes/objects pertaining to the Connector modules 
 *  like ConnectorDescriptorInfo.
 *  @author Srikanth P
 */

public class ConnectorDDTransformUtils {

    static Logger _logger = LogDomains.getLogger(LogDomains.RSR_LOGGER);

    /** Constructs ConnectorDescriptorInfo object from the 
     *  ConnectionDefDescriptor object (deployment descriptor of connector 
     *  module)
     *  @param connectionDefDescriptor ConnectionDefDescriptor object which
               represents the ra.xml and sun-ra.xml
     *  @return Transformed ConnectorDescriptorInfo object
     */

    public static ConnectorDescriptorInfo getConnectorDescriptorInfo(
                          ConnectionDefDescriptor connectionDefDescriptor) 
    {

        ConnectorDescriptorInfo connectorDescInfo = 
                            new ConnectorDescriptorInfo();
        connectorDescInfo.setConnectionDefinitionName(
                          connectionDefDescriptor.getConnectionFactoryIntf());
        connectorDescInfo.setManagedConnectionFactoryClass(
                  connectionDefDescriptor.getManagedConnectionFactoryImpl());
        connectorDescInfo.setConnectionFactoryClass(
                       connectionDefDescriptor.getConnectionFactoryImpl());
        connectorDescInfo.setConnectionFactoryInterface(
                       connectionDefDescriptor.getConnectionFactoryIntf());
        connectorDescInfo.setConnectionInterface(
                       connectionDefDescriptor.getConnectionIntf());
        connectorDescInfo.setConnectionClass(
                       connectionDefDescriptor.getConnectionImpl());
        connectorDescInfo.setMCFConfigProperties(
                       connectionDefDescriptor.getConfigProperties());
        return connectorDescInfo;
    }

    /** 
     * merges the properties mentioned in first parameter with the Set of
     * properties mentioned in second parameter.
     * Values of first parameter takes precedence over second.
     * First parameter represents properties present in domain.xml
     * Second parameter contains values mentioned in deployment descriptors.
     * @param props Array of  properties that needs to be merged with 
     *        properties mentioned in deployement descriptor. These values
     *        takes precedence over  values present in deployment descriptors.
     * @param Properties present in deployment descriptor.
     * @return Set of merged properties.
     */

    public static Set mergeProps(ElementProperty[] props, Set defaultMCFProps){
        HashSet mergedSet = new HashSet();

        if(defaultMCFProps != null) {
            Object[] defaultProps = defaultMCFProps.toArray();

            for (int i = 0; i < defaultProps.length; i++) {
                mergedSet.add(defaultProps[i]);
            }
        }

        for (int i =0; props!= null && i< props.length; i++) {
             EnvironmentProperty ep = new EnvironmentProperty(
                                props[i].getName(),props[i].getValue(),null);
             if (defaultMCFProps.contains(ep)) {
               //get the environment property in the mergedset
                Iterator iter = defaultMCFProps.iterator();
                while(iter.hasNext()){
                       EnvironmentProperty  envProp = 
                       	       (EnvironmentProperty)iter.next();
                       //and if they are equal, set ep's type to envProp's type
                       
                       //This set is important because envProp has the ra.xml 
                       //specified property-Type. When the ra-bean-class does 
                       //not have any getter method for a property, the property
                       //Type specified in ra.xml should be used.
                       if (envProp.equals(ep)) {
                       	  if (envProp.getType() != null) {
                               ep.setType(envProp.getType());
                          }
                       }
                }

                _logger.log(Level.FINER,
                     "After merging props with defaultMCFProps: envPropName: "
                     + ep.getName() + " envPropValue : " + ep.getValue());
             	mergedSet.remove(ep);
             }
             mergedSet.add(ep);
        }

        return mergedSet;
    }

    /**  
     * Get the ConnectorDescriptor object which represents the ra.xml and 
     * sun-ra.xml from an exploded rar module.
     * @param moduleDir Directory where rar is exploded.
     * @return ConnectorDescriptor object which
     *         represents the ra.xml and sun-ra.xml
     * @throws ConnectorRuntimeException if ra.xml could not be located or
     *         invalid. For 1.0 type rar if sun-ra.xml is not present or 
     *         invalid this exception is thrown. For 1.5 type rar sun-ra.xml 
     *         should not be  present.
     */

    public static ConnectorDescriptor getConnectorDescriptor(String moduleDir)
                               throws ConnectorRuntimeException
    {

        try {

            FileArchive fileArchive = new FileArchive();
            fileArchive.open(moduleDir);  // directory where rar is exploded
            ConnectorArchivist connectorArchivist = new ConnectorArchivist();
            ConnectorDescriptor connectorDescriptor =
                   (ConnectorDescriptor) connectorArchivist.open(fileArchive);
            return connectorDescriptor;
        } catch(IOException ex) {
            ConnectorRuntimeException cre =  new ConnectorRuntimeException(
                        "Failed to read the connector deployment descriptors");
            cre.initCause(ex);
            _logger.log(Level.SEVERE,
                    "rardeployment.connector_descriptor_read_error",moduleDir);
            _logger.log(Level.SEVERE,"",cre);
            throw cre;
        } catch(SAXParseException ex) {
            ConnectorRuntimeException cre =  new ConnectorRuntimeException(
                     "Failed to parse the connector deployment descriptors");
            cre.initCause(ex);
            _logger.log(Level.SEVERE,
                   "rardeployment.connector_descriptor_parse_error",moduleDir);
            _logger.log(Level.SEVERE,"",cre);
            throw cre;
        }
    }

    /**
     *   Obtain all the ConnectionDefDescriptor(abstracts the 
     *   <connector-definition> element in ra.xml) objects that are 
     *   present in given ra.xml file.
     *   @param connectorDesc  ConnectorDescriptor object which
     *         represents the ra.xml and sun-ra.xml
     *   @return Array of ConnectionDefDescriptor objects which represent the
     *           <connection-definition> element.
     */


    public static ConnectionDefDescriptor[] getConnectionDefs(
                        ConnectorDescriptor connectorDesc) 
    {
        ConnectionDefDescriptor[] connectionDefDescs = null;
        OutboundResourceAdapter ora = 
                      connectorDesc.getOutboundResourceAdapter();
        if (ora != null) {
            Set connectionDefs = ora.getConnectionDefs();
            int size = connectionDefs.size();
            if(size == 0) {
                return null;
            }
            Iterator iter = connectionDefs.iterator();
            connectionDefDescs = new ConnectionDefDescriptor[size];
            for (int i=0; i<size; ++i) {
               connectionDefDescs[i] =
                       (ConnectionDefDescriptor)iter.next();
            }
        }
        return connectionDefDescs;
    }

    /**
     *  Returns all the message listeners present in the connectorDescriptor
     *  which abstracts the ra.xml
     *  @param desc connectorDescriptor which abstracts the ra.xml
     *  @return Array of MessageListener objects
     */

     public MessageListener[] getMessageListeners(ConnectorDescriptor desc) {

         InboundResourceAdapter inboundRA = null;
         Set messageListenerSet = null;
         if(desc != null && 
                 (inboundRA = desc.getInboundResourceAdapter()) != null) {
             messageListenerSet = inboundRA.getMessageListeners();
         }

         if(messageListenerSet == null) {
             return null;
         }
         int size = messageListenerSet.size();
         MessageListener[] messageListeners = 
                  new MessageListener[size];
         Iterator iter = messageListenerSet.iterator();
         for(int i=0;i<size;++i){
             messageListeners[i] = (MessageListener)iter.next();
         }
         return messageListeners;
     }

     public static String getResourceAdapterClassName
                             (String rarLocation) {
         //Use the deployment APIs to get the name of the resourceadapter
         //class through the connector descriptor
         try {
             FileInputStream fis = new FileInputStream(rarLocation);
             MemoryMappedArchive mma = new MemoryMappedArchive(fis);
             ConnectorArchivist ca = new ConnectorArchivist();
             ConnectorDescriptor cd = (ConnectorDescriptor)ca.open(mma);
             return cd.getResourceAdapterClass();
         } catch (IOException e) {
             _logger.info(e.getMessage());
             _logger.log(Level.FINE, "Error while trying to read connector" +
                    "descriptor to get resource-adapter properties", e);
         } catch (SAXParseException e) {
             _logger.info(e.getMessage());
             _logger.log(Level.FINE, "Error while trying to read connector" +
                    "descriptor to get resource-adapter properties", e);
         }
         return null;
     }
}

