/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.server.ss.provider;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.net.Socket;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.sun.logging.LogDomains;


/**
 * InputStream implementation for the socket wrappers of Quick startup
 * implementation. Implementation is thread safe while using nio to read
 * and write simultaneusly in the non-blocking mode.
 */

class ASInputStream extends InputStream {

    private static Logger logger = LogDomains.getLogger(LogDomains.CORE_LOGGER);

    private SocketChannel sc = null;
    private Socket sock = null;
    private boolean closed = false;
    private Selector selector = null;

    private ByteBuffer bb = null;
    private byte[] bs = null;		// Invoker's previous array
    private byte[] b1 = null;

    ASInputStream(SocketChannel sc, Socket sock) throws IOException{
        this.sc = sc;
        this.sock = sock;
        this.selector = Selector.open();
        this.sc.register(selector, SelectionKey.OP_READ); 
    }


    public synchronized int read() throws IOException {
	if (b1 == null)
	    b1 = new byte[1];
	int n = this.read(b1);
	if (n == 1)
	    return b1[0] & 0xff;
	return -1;
    }

    public synchronized int read(byte[] bs, int off, int len)
	throws IOException
    {
	if ((off < 0) || (off > bs.length) || (len < 0) ||
	    ((off + len) > bs.length) || ((off + len) < 0)) {
	    throw new IndexOutOfBoundsException();
	} else if (len == 0)
	    return 0;

	ByteBuffer bb = ((this.bs == bs)
			 ? this.bb
			 : ByteBuffer.wrap(bs));
	bb.position(off);
	bb.limit(Math.min(off + len, bb.capacity()));
	this.bb = bb;
	this.bs = bs;
	return read(bb);
    }


    private int read(ByteBuffer bb)
	throws IOException
    {
        checkClosed();
        waitForSelect();
        return sc.read(bb);
    }

    private void waitForSelect() throws IOException {
        
        java.net.Socket sock = sc.socket();
        if (sock.isClosed()) {
            close();
            throw new IOException("Socket Closed");
        }

        int timeout = sock.getSoTimeout();
        Iterator it;
        SelectionKey selKey;
        
        selectorblock:
            while (true) {
                boolean timedout = true;
                try {
                    int n = selector.select(timeout);
                    if (sock.isInputShutdown() || sock.isClosed()) {
                        throw new IOException("Input Shutdown");
                    }
                    if (n > 0) {
                        timedout = false;
                    }
                    it = selector.selectedKeys().iterator();
                    while (it.hasNext()) {
                        timedout = false;
                        selKey = (SelectionKey)it.next();
                        if (selKey.isValid() && selKey.isReadable()) {
                            it.remove();
                            break selectorblock;
                        } 
                    }
                } catch (Exception e) {
                    throw (IOException) (new IOException()).initCause(e);
                }
                if (timedout) {
                     boolean wakenup = ((ASSelector) selector).wakenUp();
                     if ( !wakenup && !Thread.currentThread().isInterrupted()) {
                        throw new java.net.SocketTimeoutException("Read timed out");
                     }
                } 
            }
    }


    public void close() throws IOException {
        if (closed) {
            return;
        }
        closed = true;
        try {
            selector.close();
            selector = null;
            sc = null;
        } catch (Exception ie) {
            if ( logger.isLoggable(Level.FINE) ) {
                logger.log(Level.FINE, "" + ie.getMessage(), ie);
            }
        }
    }

    protected void finalize() throws Throwable {
        try {
            close();
        } catch (Throwable t) {}
    }

    private void checkClosed() throws IOException {
        if (closed) {
            throw new IOException("Stream is closed");
        }

        if (sock.isInputShutdown()) {
            throw new IOException("Input Shutdown");
        }
    }

}

