/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
/*
 *  Copyright 1999-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.net.InetAddress;
import java.net.URLEncoder;
import java.util.Enumeration;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.management.ObjectName;

import com.sun.org.apache.commons.modeler.Registry;
import org.apache.coyote.http11.Http11Protocol;
import org.apache.tomcat.util.net.SSLImplementation;
import org.apache.tomcat.util.net.ServerSocketFactory;

/**
 * Abstract the protocol implementation, including threading, etc.
 * Processor is single threaded and specific to stream-based protocols.
 * This is an adaptation of org.apache.coyote.http11.Http11Processor except here
 * NIO is used.
 * 
 * @author Jean-Francois Arcand
 */
public class GrizzlyHttpProtocol extends Http11Protocol {
 
    private int socketBuffer = 9000;
    
    /**
     * The <code>SelectorThread</code> used by this object.
     */
    private SelectorThread selectorThread;

    /**
     * Compression value.
     */
    private String noCompressionUserAgents = null;
    private String restrictedUserAgents = null;
    private String compressableMimeTypes = "text/html,text/xml,text/plain";
    private int compressionMinSize = 2048;
        
    // ------------------------------------------------------- Constructor --//
        
    /**
     * This method is called by the constructor of the Http11Protocol
     * superclass.
     */
    protected void create() {
        
        // 1 Selector thread per Connector.
        selectorThread = new SelectorThread();
        
        setSoLinger(-1);
        setSoTimeout(Constants.DEFAULT_CONNECTION_TIMEOUT);
        setServerSoTimeout(Constants.DEFAULT_SERVER_SOCKET_TIMEOUT);
        setTcpNoDelay(Constants.DEFAULT_TCP_NO_DELAY);
    }

    
    // ---------------------------------------------------- Public methods --//
    /** 
     * Start the protocol
     */
    public void init() throws Exception {
        try {
            checkSocketFactory();
        } catch( Exception ex ) {
            SelectorThread.logger().log(Level.SEVERE,
                       "grizzlyHttpProtocol.socketfactory.initerror",ex);
            throw ex;
        }

        if( socketFactory!=null ) {
            Enumeration attE=attributes.keys();
            while( attE.hasMoreElements() ) {
                String key=(String)attE.nextElement();
                Object v=attributes.get( key );
                socketFactory.setAttribute( key, v );
            }
        }
        
        try {
             selectorThread.setAdapter(adapter);
             selectorThread.initEndpoint();
        } catch (Exception ex) {
            SelectorThread.logger().log(Level.SEVERE, 
                       "grizzlyHttpProtocol.endpoint.initerror", ex);
            throw ex;
        }
    }
    
    
    public void start() throws Exception {
        try {            
            if ( this.oname != null ) {
                try {
                    Registry.getRegistry().registerComponent
                        (selectorThread, this.domain, "selectorThread",
                         "type=Selector,name=http" + selectorThread.getPort());
                } catch (Exception ex) {
                    SelectorThread.logger().log(Level.SEVERE,
                      "grizzlyHttpProtocol.selectorRegistrationFailed", ex);
                }
            } else {
                SelectorThread.logger().log(Level.INFO,
                           "grizzlyHttpProtocol.selectorRegisterProtocol");
            }

            selectorThread.start();
        } catch (Exception ex) {
            SelectorThread.logger().log(Level.SEVERE, 
                       "grizzlyHttpProtocol.endpoint.starterror", 
                       ex);
            throw ex;
        }
        
        SelectorThread.logger().log(Level.INFO, 
                   "grizzlyHttpProtocol.start", String.valueOf(getPort()));
    }

    public void pause() throws Exception {
        try {
            selectorThread.pauseEndpoint();
        } catch (Exception ex) {
            SelectorThread.logger().log(Level.SEVERE, 
                       "grizzlyHttpProtocol.endpoint.pauseerror", 
                       ex);
            throw ex;
        }
        SelectorThread.logger().log(Level.INFO, 
                   "grizzlyHttpProtocol.pause", String.valueOf(getPort()));
    }

    public void resume() throws Exception {
        try {
            selectorThread.resumeEndpoint();
        } catch (Exception ex) {
            SelectorThread.logger().log(Level.SEVERE, 
                       "grizzlyHttpProtocol.endpoint.resumeerror", 
                       ex);
            throw ex;
        }
        SelectorThread.logger().log(Level.INFO, 
                   "grizzlyHttpProtocol.resume", 
                                String.valueOf(getPort()));
    }

    public void destroy() throws Exception {
        SelectorThread.logger().log(Level.INFO, 
                   "grizzlyHttpProtocol.stop", 
                                String.valueOf(getPort()));
        
        if ( domain != null ){
            Registry.getRegistry().
                    unregisterComponent(new ObjectName(domain,"type", "Selector"));
        }
        selectorThread.stopEndpoint();
    }
    
    
    // -------------------- Pool setup --------------------


    public int getMaxThreads() {
        return selectorThread.getMaxThreads();
    }
    
    public void setMaxThreads( int maxThreads ) {
        selectorThread.setMaxThreads(maxThreads);
        setAttribute("maxThreads", "" + maxThreads);
    }
    

    public int getProcessorThreadsIncrement() {
        return selectorThread.threadsIncrement;
    }

    
    public void setProcessorThreadsIncrement( int threadsIncrement ) {
        selectorThread.threadsIncrement = threadsIncrement;
        setAttribute("threadsIncrement", "" + threadsIncrement);  
    }  
    
    
    public void setProcessorWorkerThreadsTimeout(int timeout){
        selectorThread.threadsTimeout = timeout;
        setAttribute("threadsTimeout", "" + timeout);     
    }
    
    
    public int getProcessorWorkerThreadsTimeout(){
        return selectorThread.threadsTimeout;
    }
    // -------------------- Tcp setup --------------------

    public int getBacklog() {
        return selectorThread.ssBackLog;
    }
    
    public void setBacklog( int i ) {
        ;
    }
    
    public int getPort() {
        return selectorThread.getPort();
    }
    
    public void setPort( int port ) {
        selectorThread.setPort(port);
        setAttribute("port", "" + port);
        //this.port=port;
    }

    public InetAddress getAddress() {
        return selectorThread.getAddress();
    }
    
    public void setAddress(InetAddress ia) {
        selectorThread.setAddress( ia );
        setAttribute("address", "" + ia);
    }
    
    public String getName() {
        String encodedAddr = "";
        if (getAddress() != null) {
            encodedAddr = "" + getAddress();
            if (encodedAddr.startsWith("/"))
                encodedAddr = encodedAddr.substring(1);
            encodedAddr = URLEncoder.encode(encodedAddr) + "-";
        }
        return ("http-" + encodedAddr + selectorThread.getPort());
    }
    
    public boolean getTcpNoDelay() {
        return selectorThread.getTcpNoDelay();
    }
    
    public void setTcpNoDelay( boolean b ) {
        selectorThread.setTcpNoDelay( b );
        setAttribute("tcpNoDelay", "" + b);
    }

    public boolean getDisableUploadTimeout() {
        return disableUploadTimeout;
    }
    
    public void setDisableUploadTimeout(boolean isDisabled) {
        disableUploadTimeout = isDisabled;
    }

    public int getSocketBuffer() {
        return socketBuffer;
    }
    
    public void setSocketBuffer(int valueI) {
        socketBuffer = valueI;
    }

    public void setMaxHttpHeaderSize(int maxHttpHeaderSize) {
        super.setMaxHttpHeaderSize(maxHttpHeaderSize);
        selectorThread.setMaxHttpHeaderSize(maxHttpHeaderSize);
    }

    public String getRestrictedUserAgents() {
        return restrictedUserAgents;
    }
    
    public void setRestrictedUserAgents(String valueS) {
        restrictedUserAgents = valueS;
        setAttribute("restrictedUserAgents", valueS);
    }

    public String getNoCompressionUserAgents() {
        return noCompressionUserAgents;
    }
    
    public void setNoCompressionUserAgents(String valueS) {
        noCompressionUserAgents = valueS;
        setAttribute("noCompressionUserAgents", valueS);
    }

    public String getCompressableMimeType() {
        return compressableMimeTypes;
    }
    
    public void setCompressableMimeType(String valueS) {
        compressableMimeTypes = valueS;
        setAttribute("compressableMimeTypes", valueS);
    }

    public int getCompressionMinSize() {
        return compressionMinSize;
    }
    
    public void setCompressionMinSize(int valueI) {
        compressionMinSize = valueI;
        setAttribute("compressionMinSize", "" + valueI);
    }

    public int getSoLinger() {
        return selectorThread.getSoLinger();
    }
    
    public void setSoLinger( int i ) {
        selectorThread.setSoLinger( i );
        setAttribute("soLinger", "" + i);
    }

    public int getSoTimeout() {
        return selectorThread.getSoTimeout();
    }
    
    public void setSoTimeout( int i ) {
        selectorThread.setSoTimeout(i);
        setAttribute("soTimeout", "" + i);
    }
    
    public int getServerSoTimeout() {
        return selectorThread.getServerSoTimeout();
    }
    
    public void setServerSoTimeout( int i ) {
        selectorThread.setServerSoTimeout(i);
        setAttribute("serverSoTimeout", "" + i);
    }
    
    public void setSecure( boolean b ) {
        super.setSecure(b);
        selectorThread.setSecure(b);
    }

    /** 
     * Set the maximum number of Keep-Alive requests that we will honor.
     */
    public void setMaxKeepAliveRequests(int mkar) {
        selectorThread.setMaxKeepAliveRequests(mkar);
    }   
   

    /** 
     * Sets the number of seconds before a keep-alive connection that has
     * been idle times out and is closed.
     *
     * @param timeout Keep-alive timeout in number of seconds
     */    
    public void setKeepAliveTimeoutInSeconds(int timeout) {
        selectorThread.setKeepAliveTimeoutInSeconds(timeout);
    }


    /** 
     * Sets the number of keep-alive threads.
     *
     * @param threadCount Number of keep-alive threads
     */    
    public void setKeepAliveThreadCount(int threadCount) {
        selectorThread.setKeepAliveThreadCount(threadCount);
    }


    /**
     * The minimun threads created at startup.
     */ 
    public void setMinThreads(int minThreads){
        selectorThread.setMinThreads(minThreads);
    }
    

    /**
     * Set the request input buffer size
     */
    public void setBufferSize(int requestBufferSize){
        super.setBufferSize(requestBufferSize);
        selectorThread.setBufferSize(requestBufferSize);
    }
    
    
    /**
     * Set the <code>Selector</code> times out value.
     */ 
    public void setSelectorTimeout(int selectorTimeout){
        selectorThread.selectorTimeout = selectorTimeout;
    }
    
    
    /**
     * Return the <code>Selector</code> times out value.
     */
    public int getSelectorTimeout(){
        return selectorThread.selectorTimeout;
    }
    
   
    /**
     * Set the <code>reader-thread</code> from domain.xml.
     */
    public void setMaxReadWorkerThreads(int maxReadWorkerThreads){
        selectorThread.maxReadWorkerThreads = maxReadWorkerThreads;
    }
    
    
    /**
     * Return the <code>read-thread</code> used by this <code>Selector</code>
     */   
    public int getMaxReadWorkerThreads(){
        return selectorThread.maxReadWorkerThreads;
    }  

    
    public void setDisplayConfiguration(boolean displayConfiguration){
        selectorThread.displayConfiguration = displayConfiguration;
    }
        
    
    public boolean getDisplayConfiguration(){
        return selectorThread.displayConfiguration;
    }

    
    /**
     * Set the <code>recycle-tasks</code> by this <code>Selector</code>
     */
    public void setRecycleTasks(boolean recycleTasks){
        selectorThread.recycleTasks = recycleTasks;
    }
    
    
    /**
     * Return the <code>recycle-tasks</code> used by this 
     * <code>Selector</code>
     */     
    public boolean getRecycleTasks(){
        return selectorThread.recycleTasks;
    }    
    
     
    public void setUseByteBufferView(boolean useByteBufferView){
        selectorThread.useByteBufferView = useByteBufferView;
    }
    
            
    public boolean getUseByteBufferView(){
        return selectorThread.useByteBufferView ;
    }

    
    /**
     * Set the <code>processor-thread</code> from domain.xml
     */   
    public void setMaxProcessorWorkerThreads(int maxProcessorWorkerThreads){
        selectorThread.maxProcessorWorkerThreads = maxProcessorWorkerThreads;
    }
    
    
    /**
     * Return the <code>processor-thread</code> used by this <code>Selector</code>
     */   
    public int getMaxProcessorWorkerThreads(){
        return selectorThread.maxProcessorWorkerThreads;
    }
 
   
    /**
     * Set the <code>reader-queue-length</code> value 
     * on this <code>Selector</code>
     */
    public void setMinReadQueueLength(int minReadQueueLength){
        selectorThread.minReadQueueLength = minReadQueueLength;
    }

    
    /**
     * Return the <code>reader-queue-length</code> value
     * on this <code>Selector</code>
     */
    public int getMinReadQueueLength(){
        return selectorThread.minReadQueueLength;
    }
 
  
    /**
     * Set the <code>processor-queue-length</code> value 
     * on this <code>Selector</code>
     */
    public void setMinProcessorQueueLength(int minProcessorQueueLength){
        selectorThread.minProcessorQueueLength = minProcessorQueueLength;
    }
 
     
    /**
     * Return the <code>processor-queue-length</code> value
     * on this <code>Selector</code>
     */   
    public int getMinProcessorQueueLength(){
        return selectorThread.minProcessorQueueLength;
    }
    
    
    /**
     * Set the <code>use-nio-non-blocking</code> by this <code>Selector</code>
     */    
    public void setUseDirectByteBuffer(boolean useDirectByteBuffer){
        selectorThread.useDirectByteBuffer = useDirectByteBuffer;
    }
    
    
    /**
     * Return the <code>use-nio-non-blocking</code> used by this 
     * <code>Selector</code>
     */     
    public boolean getUseDirectByteBuffer(){
        return selectorThread.useDirectByteBuffer;
    }
    
    
    /**
     * Set the maximum pending connection this <code>Pipeline</code>
     * can handle.
     */
    public void setQueueSizeInBytes(int maxQueueSizeInBytes){
        selectorThread.maxQueueSizeInBytes = maxQueueSizeInBytes;
    }
    
    
    /**
     * Set the <code>SocketServer</code> backlog.
     */
    public void setSocketServerBacklog(int ssBackLog){
        selectorThread.ssBackLog = ssBackLog;
    }
    
    /**
     * Set the number of <code>SelectorThread</code> Grizzly will uses.
     */
    public void setSelectorReadThreadsCount(int selectorReadThreadsCount){
        selectorThread.selectorReadThreadsCount = selectorReadThreadsCount;
    }

 
    /**
     * Set the default response type used. Specified as a semi-colon
     * delimited string consisting of content-type, encoding,
     * language, charset
     */
    public void setDefaultResponseType(String defaultResponseType){
         selectorThread.defaultResponseType = defaultResponseType;
    }


    /**
     * Return the default response type used
     */
    public String getDefaultResponseType(){
         return  selectorThread.defaultResponseType;
    }
    
    
    /**
     * Set the forced response type used. The response type to be forced 
     * if the content served cannot be matched by any of the MIME mappings
     * for extensions. Specified as a semi-colon delimited string consisting of
     * content-type, encoding, language, charset
     */
    public void setForcedResponseType(String forcedResponseType){
        selectorThread.forcedResponseType = forcedResponseType;
    }  
    
        
    /**
     * Return the default response type used
     */
    public String getForcedResponseType(){
        return  selectorThread.forcedResponseType;
    }   
    
    
    //------------------------------------------------- FileCache config -----/

   
    /**
     * The timeout in seconds before remove a <code>FileCacheEntry</code>
     * from the <code>fileCache</code>
     */
    public void setSecondsMaxAge(int sMaxAges){
        selectorThread.secondsMaxAge = sMaxAges;
    }
    
    
    /**
     * Set the maximum entries this cache can contains.
     */
    public void setMaxCacheEntries(int mEntries){
        selectorThread.maxCacheEntries = mEntries;
    }

    
    /**
     * Return the maximum entries this cache can contains.
     */    
    public int getMaxCacheEntries(){
        return selectorThread.maxCacheEntries;
    }
    
    
    /**
     * Set the maximum size a <code>FileCacheEntry</code> can have.
     */
    public void setMinEntrySize(long mSize){
        selectorThread.minEntrySize = mSize;
    }
    
    
    /**
     * Get the maximum size a <code>FileCacheEntry</code> can have.
     */
    public long getMinEntrySize(){
        return selectorThread.minEntrySize;
    }
     
    
    /**
     * Set the maximum size a <code>FileCacheEntry</code> can have.
     */
    public void setMaxEntrySize(long mEntrySize){
        selectorThread.maxEntrySize = mEntrySize;
    }
    
    
    /**
     * Get the maximum size a <code>FileCacheEntry</code> can have.
     */
    public long getMaxEntrySize(){
        return selectorThread.maxEntrySize;
    }
    
    
    /**
     * Set the maximum cache size
     */ 
    public void setMaxLargeCacheSize(long mCacheSize){
        selectorThread.maxLargeFileCacheSize = mCacheSize;
    }

    
    /**
     * Get the maximum cache size
     */ 
    public long getMaxLargeCacheSize(){
        return selectorThread.maxLargeFileCacheSize;
    }
    
    
    /**
     * Set the maximum cache size
     */ 
    public void setMaxSmallCacheSize(long mCacheSize){
        selectorThread.maxSmallFileCacheSize = mCacheSize;
    }
    
    
    /**
     * Get the maximum cache size
     */ 
    public long getMaxSmallCacheSize(){
        return selectorThread.maxSmallFileCacheSize;
    }    

    
    /**
     * Is the fileCache enabled.
     */
    public boolean isFileCacheEnabled(){
        return selectorThread.isFileCacheEnabled;
    }

    
    /**
     * Is the file caching mechanism enabled.
     */
    public void setFileCacheEnabled(boolean isFileCacheEnabled){
        selectorThread.isFileCacheEnabled = isFileCacheEnabled;
    }
   
    
    /**
     * Is the large file cache support enabled.
     */
    public void setLargeFileCacheEnabled(boolean isLargeEnabled){
        selectorThread.isLargeFileCacheEnabled = isLargeEnabled;
    }
   
    
    /**
     * Is the large file cache support enabled.
     */
    public boolean getLargeFileCacheEnabled(){
        return selectorThread.isLargeFileCacheEnabled;
    }    
   
    
    /**
     * Set the documenr root folder
     */
    public void setWebAppRootPath(String rootFolder){
        selectorThread.rootFolder = rootFolder;
    }
    
    
    /**
     * Return the folder's root where application are deployed.
     */
    public String getWebAppRootPath(){
        return selectorThread.rootFolder;
    }     
    
    
    /**
     * Set the logger
     */
    public void setLogger(Logger logger){
        if ( logger != null )
            selectorThread.logger = logger;
    }
    
    
    /**
     * Return the logger used by the Grizzly classes.
     */
    public Logger getLogger(){
        return selectorThread.logger;
    }  
    
    
    /**
     * Return the instance of SelectorThread used by this class.
     */
    public SelectorThread selectorThread(){
        return selectorThread;
    }
    
    
    // --------------------------------------------------------- Private method

    /** Sanity check and socketFactory setup.
     *  IMHO it is better to stop the show on a broken connector,
     *  then leave Tomcat running and broken.
     *  @exception TomcatException Unable to resolve classes
     */
    private void checkSocketFactory() throws Exception {
        if (secure) {
            // The SSL setup code has been moved into
            // SSLImplementation since SocketFactory doesn't
            // provide a wide enough interface
            sslImplementation =
                SSLImplementation.getInstance(sslImplementationName);
            socketFactory = sslImplementation.getServerSocketFactory();
            selectorThread.setSSLImplementation(sslImplementation);
        } else if (socketFactoryName != null) {
            socketFactory = string2SocketFactory(socketFactoryName);
        }
        if(socketFactory==null)
            socketFactory=ServerSocketFactory.getDefault();
        selectorThread.setServerSocketFactory(socketFactory);
    }
}

