/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.jdo.api.persistence.enhancer.classfile;

import java.io.*;
import java.util.Vector;
import java.util.Enumeration;

/**
 * ExceptionsAttribute represents a method attribute in a class file
 * listing the checked exceptions for the method.
 */

public class ExceptionsAttribute extends ClassAttribute {
    public final static String expectedAttrName = "Exceptions";//NOI18N

  /* The list of checked exceptions */
  private Vector exceptionTable;

  /* public accessors */

  /**
   *  Return an enumeration of the checked exceptions
   */
  public Enumeration exceptions() {
    return exceptionTable.elements();
  }

  /**
   * Constructor
   */
  public ExceptionsAttribute(ConstUtf8 attrName, Vector excTable) {
    super(attrName);
    exceptionTable = excTable;
  }

  /**
   * Convenience Constructor - for single exception
   */
  public ExceptionsAttribute(ConstUtf8 attrName, ConstClass exc) {
    super(attrName);
    exceptionTable = new Vector(1);
    exceptionTable.addElement(exc);
  }

  /* package local methods */

  static ExceptionsAttribute read(ConstUtf8 attrName,
				  DataInputStream data, ConstantPool pool)
    throws IOException {
    int nExcepts = data.readUnsignedShort();
    Vector excTable = new Vector();
    while (nExcepts-- > 0) {
      int excIndex = data.readUnsignedShort();
      ConstClass exc_class = null;
      if (excIndex != 0)
	exc_class = (ConstClass) pool.constantAt(excIndex);
      excTable.addElement(exc_class);
    }
        
    return new ExceptionsAttribute(attrName, excTable);
  }

  void write(DataOutputStream out) throws IOException {
    out.writeShort(attrName().getIndex());
    out.writeInt(2+2*exceptionTable.size());
    out.writeShort(exceptionTable.size());
    for (int i=0; i<exceptionTable.size(); i++)
      out.writeShort(((ConstClass) exceptionTable.elementAt(i)).getIndex());
  }

  void print(PrintStream out, int indent) {
    ClassPrint.spaces(out, indent);
    out.print("Exceptions:");//NOI18N
    for (int i=0; i<exceptionTable.size(); i++)
        out.print(" " + ((ConstClass) exceptionTable.elementAt(i)).asString());//NOI18N
    out.println();
  }
  
}

