/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */



/*
 * ParameterSupport.java
 *
 * Created on December 07, 2001
 */

package com.sun.jdo.spi.persistence.support.ejb.ejbqlc;

import java.lang.reflect.Method;
import java.util.ResourceBundle;

import com.sun.jdo.spi.persistence.utility.I18NHelper;

/** 
 * Helper class to handle EJBQL query parameters. 
 *
 * @author  Michael Bouschen
 * @author  Shing Wai Chan
 */
public class ParameterSupport
{
    /** The types of the parameters of the finder/selector method. */
    private Class[] parameterTypes;
    
    /**
     * The EJB names corresponding to types of parameters of the
     * finder/selector method.
     */
    private String[] parameterEjbNames;
    
    /** I18N support. */
    protected final static ResourceBundle msgs = I18NHelper.loadBundle(
        ParameterSupport.class);

    /** 
     * Constructor. 
     * @param method the Method instance of the finder/selector method.
     */
    public ParameterSupport(Method method)
    {
        this.parameterTypes = 
            (method == null) ? new Class[0] : method.getParameterTypes();
        this.parameterEjbNames = new String[this.parameterTypes.length];
    }

    /** 
     * Returns type of the EJBQL parameter by input parameter declaration 
     * string. The specified string denotes a parameter application in EJBQL. 
     * It has the form "?<number>" where <number> is the parameter number 
     * starting with 1.
     * @return class instance representing the parameter type.
     */
    public Class getParameterType(String ejbqlParamDecl)
    {
        return getParameterType(getParamNumber(ejbqlParamDecl));
    }

    /** 
     * Returns the type of the EJBQL parameter by number. 
     * Note, the numbering of EJBQL parameters starts with 1, 
     * so the method expects 1 as the number of the first parameter.
     * @return class instance representing the parameter type.
     */
    public Class getParameterType(int paramNumber)
    {
        // InputParams are numbered starting at 1, so adjust for
        // array indexing.
        return parameterTypes[paramNumber - 1];
    }

    /** 
     * Get EJB name corresponding to the EJBQL parameter by input
     * parameter declaration string.
     * @param ejbqlParamDecl denotes a parameter application in EJBQL. 
     * It has the form "?<number>" where <number> is the parameter number 
     * starting with 1.
     * @return class instance representing the parameter type.
     */
    public String getParameterEjbName(String ejbqlParamDecl)
    {
        return getParameterEjbName(getParamNumber(ejbqlParamDecl));
    }

    /** 
     * Get EJB name corresponding to the EJBQL parameter number.
     * @param paramNumber numbering of parameters starting with 1
     * @return class instance representing the parameter type.
     */
    public String getParameterEjbName(int paramNumber)
    {
        return parameterEjbNames[paramNumber - 1];
    }

    /** 
     * Set EJB name corresponding to the EJBQL parameter by input
     * parameter declaration string.
     * @param ejbqlParamDecl denotes a parameter application in EJBQL. 
     * It has the form "?<number>" where <number> is the parameter number 
     * starting with 1.
     * @param ejbName
     * @return class instance representing the parameter type.
     */
    public void setParameterEjbName(String ejbqlParamDecl, String ejbName)
    {
        parameterEjbNames[getParamNumber(ejbqlParamDecl) - 1] = ejbName;
    }

    /** 
     * Get all EJB names corresponding to the EJBQL parameters.
     * @return class instance representing the parameter type.
     */
    public String[] getParameterEjbNames()
    {
        return parameterEjbNames;
    }

    /** 
     * Returns the name of the corresponding JDO parameter.
     * The specified string denotes a parameter application in EJBQL. 
     * It has the form "?<number>" where <number> is the parameter number 
     * starting with 1.
     * @return name of JDOQL parameter 
     */
    public String getParameterName(String ejbqlParamDecl)
    {
        return getParameterName(getParamNumber(ejbqlParamDecl));
    }

    /** 
     * Returns the name of the corresponding JDO parameter by parameter number.
     * @return name of JDOQL parameter 
     */
    public String getParameterName(int paramNumber)
    {
        return "_jdoParam" + String.valueOf(paramNumber);
    }

    /** 
     * Returns the number of parameters.
     * @return parameter count.
     */
    public int getParameterCount()
    {
        return parameterTypes.length;
    }

    // Internal methods

    /** 
     * Internal method to extract the number from a parameter application 
     * in EJBQL. 
     */
    private int getParamNumber(String ejbqlParamDecl)
    {
        int paramNum = 0;
        try {
            paramNum = Integer.parseInt(ejbqlParamDecl.substring(1));
        } catch(Exception ex) {
            ErrorMsg.error(I18NHelper.getMessage(
                msgs, "EXC_InvalidParameterIndex", //NOI18N
                ejbqlParamDecl, String.valueOf(parameterTypes.length)));
        }
        if (paramNum < 1 || paramNum > parameterTypes.length) {
            ErrorMsg.error(I18NHelper.getMessage(
                msgs, "EXC_InvalidParameterIndex", //NOI18N
                ejbqlParamDecl, String.valueOf(parameterTypes.length)));
        }
        return paramNum;
    }
}
