/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Enumerated;
import javax.persistence.EnumType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.Temporal;
import javax.persistence.Version;

import oracle.toplink.essentials.descriptors.TimestampLockingPolicy;
import oracle.toplink.essentials.descriptors.VersionLockingPolicy;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.ClassAccessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataAccessibleObject;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataColumn;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataHelper;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.sequencing.MetadataGeneratedValue;

import oracle.toplink.essentials.internal.helper.Helper;
import oracle.toplink.essentials.internal.helper.DatabaseField;

import oracle.toplink.essentials.mappings.converters.EnumTypeConverter;
import oracle.toplink.essentials.mappings.converters.SerializedObjectConverter;
import oracle.toplink.essentials.mappings.converters.TypeConversionConverter;

import oracle.toplink.essentials.mappings.DirectToFieldMapping;

import oracle.toplink.essentials.internal.helper.DatabaseField;

/**
 * An relational accessor.
 * 
 * @author Guy Pelletier
 * @since TopLink EJB 3.0 Reference Implementation
 */
public class BasicAccessor extends NonRelationshipAccessor {    
    /**
     * INTERNAL:
     */
    public BasicAccessor(MetadataAccessibleObject accessibleObject, ClassAccessor classAccessor) {
        super(accessibleObject, classAccessor);
    }
    
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
     * Build a metadata column.
     */
    protected MetadataColumn getColumn() {
        Column column = getAnnotation(Column.class);
        return new MetadataColumn(column, this);
    }
    
    /**
     * INTERNAL:
     * Process column details from an @Column or column element into a 
     * MetadataColumn and return it. This will set correct metadata and log 
     * defaulting messages to the user.
     * 
     * NOTE: This method will be called for on columns loaded from an accessor
     * and those loaded from an attribute override.
     */
    public DatabaseField getDatabaseField() {
         // Check if we have an attribute override first, otherwise process for 
         // a column.
         MetadataColumn column;
         
        if (m_descriptor.hasAttributeOverrideFor(getAttributeName())) {
            column = m_descriptor.getAttributeOverrideFor(getAttributeName());
        } else {
            column = getColumn();
        }
        
        // Get the actual database field and apply any defaults.
        DatabaseField field = column.getDatabaseField();
        
        // Set the correct field name, defaulting and logging when necessary.
        field.setName(getName(field.getName(), column.getUpperCaseAttributeName(), m_logger.COLUMN));
        
        // Make sure this is a table name on the field.
        if (field.getTableName().equals("")) {
            field.setTableName(m_descriptor.getPrimaryTableName());
        }
                    
        return field;
     }
     
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
     */
     public String getEnumeratedType() {
        Enumerated enumerated = getAnnotation(Enumerated.class);
        
        if (enumerated == null) {
            return EnumType.ORDINAL.name();
        } else {
            return enumerated.value().name();
        }
     }
     
    /**
     * INTERNAL:
     * Return the temporal type for this accessor. Assumes there is a @Temporal.
     */
    public String getTemporalType() {
        Temporal temporal = getAnnotation(Temporal.class);
        return temporal.value().name();
    }
    
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
	 * Method to check if an annotated element has an @Enumerated.
     */
	public boolean hasEnumerated() {
		return isAnnotationPresent(Enumerated.class);
    }
    
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
     * Return true if this accessor represents a temporal mapping.
     */
	public boolean hasTemporal() {
        return isAnnotationPresent(Temporal.class);
    }
    
    /**
     * INTERNAL:
     */
     public boolean isBasic() {
     	return true;
     }
     
    /**
     * INTERNAL:
     * Return true if this represents an enum type mapping. Will return true
     * if the accessor's reference class is an enum or if an enumerated 
     * sub-element exists.
     */
     public boolean isEnumerated() {
     	return hasEnumerated() || MetadataHelper.isValidEnumeratedType(getReferenceClass());
     }
     
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
     * Return true if this accessor represents an id field.
     */
	public boolean isId() {
        return isAnnotationPresent(Id.class);
    }
    
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
     * Return true if this accessor represents an BLOB/CLOB mapping.
     */
	public boolean isLob() {
        return isAnnotationPresent(Lob.class);
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents a serialized mapping.
     */
	public boolean isSerialized() {
        return MetadataHelper.isValidSerializedType(getReferenceClass());
    }
    
    /**
     * INTERNAL:
     * Return true if this accessor represents a temporal mapping.
     */
	public boolean isTemporal() {
        return hasTemporal() || MetadataHelper.isValidTemporalType(getReferenceClass());
    }
    
    /**
     * INTERNAL: (Overridden in XMLBasicAccessor)
	 * Return true if this accessor represents an optimistic locking field.
     */
	public boolean isVersion() {
        return isAnnotationPresent(Version.class);
    }
    
    /**
     * INTERNAL:
     * Process a basic accessor.
     */
    public void process() {
        // Process the @Column or column element if there is one.
        DatabaseField field = getDatabaseField();
            
        // Process an @Version or version element if there is one.
        if (isVersion()) {
            if (m_descriptor.usesOptimisticLocking()) {
                // Ignore the version locking if it is already set.
                m_logger.logWarningMessage(m_logger.IGNORE_VERSION_LOCKING, this);
            } else {
                processVersion(field);
            }
        } else if (isId()) {
            // Process an @Id or id element.
            processId(field);
        }
                
        if (m_descriptor.hasMappingForAttributeName(getAttributeName())) {
            // Ignore the mapping if one already exists for it.
            m_logger.logWarningMessage(m_logger.IGNORE_MAPPING, this);
        } else {
            // Process a DirectToFieldMapping, that is a Basic that could
            // be used in conjunction with a Lob, Temporal, Enumerated
            // or inferred to be used with a serialized mapping.
            processDirectToFieldMapping(field);
        }
    }
    
    /**
     * INTERNAL:
     * Process a Serialized or Basic into a DirectToFieldMapping. If neither 
     * is found a DirectToFieldMapping is created regardless.
     */
    protected void processDirectToFieldMapping(DatabaseField field) {
        DirectToFieldMapping mapping = new DirectToFieldMapping();
        mapping.setField(field);
        mapping.setIsReadOnly(field.isReadOnly());
        mapping.setAttributeName(getAttributeName());
		mapping.setIsOptional(isOptional());
        
        if (usesIndirection()) {
            m_logger.logWarningMessage(m_logger.IGNORE_BASIC_FETCH_LAZY, this);
        }
        
        // Will check for PROPERTY access
        setAccessorMethods(mapping);
        
        // Check for an enum first since it will fall into a serializable 
        // mapping otherwise (Enums are serialized)
        if (isEnumerated()) {
            processEnumerated(mapping);
        } else if (isLob()) {
            processLob(mapping);
        } else if (isTemporal()) {
            processTemporal(mapping);
        } else if (isSerialized()) {
            processSerialized(mapping);
        }
        
        // Add the mapping to the descriptor.
        m_descriptor.addMapping(mapping);
    }
    
    /**
     * INTERNAL:
     * Process an @Enumerated. The method may still be called if no @Enumerated
     * has been specified but the accessor's reference class is a valid 
     * enumerated type.
     */
    protected void processEnumerated(DirectToFieldMapping mapping) {
        // If this accessor is tagged as an enumerated type, validate the
        // reference class.
        if (hasEnumerated()) {
            if (! MetadataHelper.isValidEnumeratedType(getReferenceClass())) {
                m_validator.throwInvalidTypeForEnumeratedAttribute(getJavaClass(), mapping.getAttributeName(), getReferenceClass());
            }
        }
        
        // Set a EnumTypeConverter on the mapping.
        mapping.setConverter(new EnumTypeConverter(mapping, getReferenceClassName(), getEnumeratedType().equals(EnumType.ORDINAL.name())));
    }
    
    /**
     * INTERNAL: (Overridden In XMLBasicAccessor)
     * Process a @GeneratedValue.
     */
    protected void processGeneratedValue(DatabaseField field) {
        GeneratedValue generatedValue = getAnnotation(GeneratedValue.class);
        
        if (generatedValue != null) {
            processGeneratedValue(new MetadataGeneratedValue(generatedValue), field);
        }
    }
    
    /**
     * INTERNAL:
     */
    protected void processGeneratedValue(MetadataGeneratedValue generatedValue, DatabaseField sequenceNumberField) {
        // Set the sequence number field on the descriptor.		
        DatabaseField existingSequenceNumberField = m_descriptor.getSequenceNumberField();
        
        if (existingSequenceNumberField == null) {
            m_descriptor.setSequenceNumberField(sequenceNumberField);
            getProject().addGeneratedValue(generatedValue, getJavaClass());
        } else {
            m_validator.throwOnlyOneGeneratedValueIsAllowed(getJavaClass(), existingSequenceNumberField.getQualifiedName(), sequenceNumberField.getQualifiedName());
        }
    }
    
    /**
     * INTERNAL:
     * Process an @Id or id element if there is one.
     */
    protected void processId(DatabaseField field) {
    	if (m_descriptor.ignoreIDs()) {
            // Project XML merging. XML wins, ignore annotations/orm xml.
            m_logger.logWarningMessage(m_logger.IGNORE_PRIMARY_KEY, this);
        } else {
            String attributeName = getAttributeName();
            
            if (m_descriptor.hasEmbeddedIdAttribute()) {
                // We found both an Id and an EmbeddedId, throw an exception.
                m_validator.throwEmbeddedIdAndIdFound(getJavaClass(), m_descriptor.getEmbeddedIdAttributeName(), attributeName);
            }
            
            // If this entity has a pk class, we need to validate our ids. 
            m_descriptor.validatePKClassId(attributeName, getReferenceClass());
        
            // Store the Id attribute name. Used with validation and OrderBy.
            m_descriptor.addIdAttributeName(attributeName);

            // Add the primary key field to the descriptor.            
            m_descriptor.addPrimaryKeyField(field);
	
            // Process the generated value for this id.
            processGeneratedValue(field);
            
            // Process a table generator.
            processTableGenerator();
            
            // Process a sequence generator.
            processSequenceGenerator();
        }
    }
    
    /**
     * INTERNAL:
     * Process a @Lob or lob sub-element. The lob must be specified to process 
     * and create a lob type mapping.
     */
    protected void processLob(DirectToFieldMapping mapping) {
        // Set the field classification type on the mapping based on the
        // referenceClass type.
        if (MetadataHelper.isValidClobType(getReferenceClass())) {
            mapping.setFieldClassification(java.sql.Clob.class);    
        } else if (MetadataHelper.isValidBlobType(getReferenceClass())) {
            mapping.setFieldClassification(java.sql.Blob.class);
        } else {
            // The referenceClass is neither a valide BLOB or CLOB attribute.   
            m_validator.throwInvalidTypeForLOBAttribute(getJavaClass(), mapping.getAttributeName(), getReferenceClass());
        }
        
        // Set a TypeConversionConverter on the mapping.
        mapping.setConverter(new TypeConversionConverter(mapping));
    }
    
    /**
     * INTERNAL:
     * Process a potential serializable attribute. If the class implements 
     * the Serializable interface then set a SerializedObjectConverter on 
     * the mapping.
     */
    protected void processSerialized(DirectToFieldMapping mapping) {
        if (Helper.classImplementsInterface(getReferenceClass(), Serializable.class)) {
            mapping.setConverter(new SerializedObjectConverter(mapping));
        } else {
            m_validator.throwInvalidTypeForSerializedAttribute(getJavaClass(), mapping.getAttributeName(), getReferenceClass());
        }
    }
    
    /**
     * INTERNAL:
     * Process a temporal type accessor.
     */
    protected void processTemporal(DirectToFieldMapping mapping) {
        if (hasTemporal()) {
            if (MetadataHelper.isValidTemporalType(getReferenceClass())) {
                // Set a TypeConversionConverter on the mapping.
                mapping.setConverter(new TypeConversionConverter(mapping));
                mapping.setFieldClassification(MetadataHelper.getFieldClassification(getTemporalType()));
            } else {
                m_validator.throwInvalidTypeForTemporalAttribute(getJavaClass(), getAttributeName(), getReferenceClass());
            }    
        } else {
            m_validator.throwNoTemporalTypeSpecified(getJavaClass(), getAttributeName());
        }
    }
    
    /**
     * INTERNAL:
     */
    protected void processVersion(DatabaseField field) {
        Class lockType = getRawClass();
        field.setType(lockType);
        
        if (MetadataHelper.isValidVersionLockingType(lockType)) {
            m_descriptor.setOptimisticLockingPolicy(new VersionLockingPolicy(field));
        } else if (MetadataHelper.isValidTimstampVersionLockingType(lockType)) {
            m_descriptor.setOptimisticLockingPolicy(new TimestampLockingPolicy(field));
        } else {
            m_validator.throwInvalidTypeForVersionAttribute(getJavaClass(), getAttributeName(), lockType);
        }
    }
}
