/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.parsing.ejbql;

import java.util.Vector;

import oracle.toplink.essentials.internal.parsing.*;
import oracle.toplink.essentials.exceptions.*;

// Third party (ANLTR) stuff
import persistence.antlr.LLkParser;
import persistence.antlr.ANTLRException;
import persistence.antlr.MismatchedTokenException;
import persistence.antlr.MismatchedCharException;
import persistence.antlr.NoViableAltException;
import persistence.antlr.NoViableAltForCharException;
import persistence.antlr.TokenStreamRecognitionException;
import persistence.antlr.RecognitionException;
import persistence.antlr.ParserSharedInputState;
import persistence.antlr.TokenStream;
import persistence.antlr.Token;

/**
 * INTERNAL
 * <p><b>Purpose</b>: The super class of the EJBQLParser
 * <p><b>Responsibilities</b>:<ul>
 * <li> Respond to messages from the parser appropriately
 * <li> Build a parse tree
 * <li> Answer any errors ocurring during parsing
 * <li> Answer any parameters found during parsing
 * <li> Maintain the EJBQL string
 * </ul>
 *    @author Jon Driscoll and Joel Lucuik
 *    @since TopLink 4.0
 */
public class EJBQLParserBase extends LLkParser {
    private boolean verbose = false;

    // Removed by JED as part of the re-factoring for EJB 2.1
    //protected String abstractSchemaName = null;
    //protected String variableName = null;
    private Vector errors;
    private String theEjbql = null;

    /** */
    protected static final int EOF_CHAR = 65535; // = (char) -1 = EOF

    /** The factory to create parse tree nodes. */
    protected NodeFactory factory;

    protected EJBQLParserBase(persistence.antlr.TokenBuffer tokenBuf, int k_) {
        super(tokenBuf, k_);
        initialize();
    }

    public EJBQLParserBase(ParserSharedInputState state, int k_) {
        super(state, k_);
    }

    protected EJBQLParserBase(TokenStream lexer, int k) {
        super(lexer, k);
    }

    /** */
    public void setNodeFactory(NodeFactory factory) {
        this.factory = factory;
    }

    /** */
    public NodeFactory getNodeFactory() {
        return factory;
    }

    /** */
    public ParseTree getParseTree() {
        return (ParseTree)getRootNode();
    }
    
    /** */
    public Object getRootNode() {
        return null;
    }

    /**
     * INTERNAL
     * Add the passed error to the error collection
     */
    public void addError(EJBQLException e) {
        getErrors().add(e);
    }

    public void addError(Exception e) {
        addError(EJBQLException.generalParsingException(getEjbqlString(), e));
    }

    /**
     * INTERNAL
     * Build a parser for the passed ejbql string
     */
    public static EJBQLParser buildParserFor(String ejbqlString) {
        return EJBQLParser.buildParserFor(ejbqlString);
    }

    /**
     * Answer an instance of EJBQLParseTree built from parsing EJBQLString.
     * Throw a DeploymentException if an error occurs
     */
    public static ParseTree buildParseTree(String ejbqlString) throws Exception {
        // parse
        EJBQLParser parser = buildParserFor(ejbqlString);
        try {
            parser.document();
        } catch (Exception e) {
            parser.addError(e);
        }
        if (parser.hasErrors()) {
            throw parser.generateException();
        }

        // return the tree
        return parser.getParseTree();
    }

    /**
     * INTERNAL
     * Generate an exception which encapsulates all the exceptions generated
     * by this parser. Special case where the first exception is an EJBQLException.
     */
    public Exception generateException() {
        //Handle exceptions we expect (such as expressionSotSupported)
        Exception firstException = (Exception)getErrors().elementAt(0);
        if (firstException instanceof EJBQLException) {
            return firstException;
        }

        //Handle general exceptions, such as NPE
        EJBQLException exception = EJBQLException.generalParsingException(getEjbqlString());
        exception.setInternalExceptions(getErrors());
        return exception;
    }

    /**
     * INTERNAL
     * Return the error collection
     */
    private Vector getErrors() {
        return errors;
    }

    /**
     * hasErrors(): Answer true if there were errors during the parsing process
     *
     */
    public boolean hasErrors() {
        return !getErrors().isEmpty();
    }

    public void initialize() {
        setErrors(new Vector());
        setNodeFactory(new NodeFactoryImpl());
    }

    /**
     * Insert the method's description here.
     * Creation date: (1/10/01 7:47:41 AM)
     * @return boolean
     */
    public boolean isVerbose() {
        return verbose;
    }

    /**
     * output: show the output if verbose is on
     */
    public void output(String output) {
        if (isVerbose()) {
            System.out.println(output);
        }
    }

    /*
     * Show the string being parsed, and where the error occurred
     */
    public void reportError(RecognitionException ex) {
        EJBQLException error = handleANTLRException(ex);
        addError(error);
    }

    /** */                                                                                   
    protected EJBQLException handleANTLRException(ANTLRException ex) {
        EJBQLException result = null;
        if (ex instanceof MismatchedCharException) {
            MismatchedCharException mismatched = (MismatchedCharException)ex;
            if (mismatched.mismatchType == MismatchedCharException.CHAR) {
                if (mismatched.foundChar == EOF_CHAR) {
                    result = EJBQLException.unexpectedEOF(getEjbqlString());
                }
                else {
                    result = EJBQLException.expectedCharFound(
                        getEjbqlString(),
                        String.valueOf((char)mismatched.expecting), 
                        String.valueOf((char)mismatched.foundChar));
                }
            }
        }
        else if (ex instanceof MismatchedTokenException) {
            MismatchedTokenException mismatched = (MismatchedTokenException)ex;
            Token token = mismatched.token;
            if ((mismatched.mismatchType == MismatchedTokenException.TOKEN) &&
                (token != null)) {
                if (token.getType() == Token.EOF_TYPE) {
                    result = EJBQLException.unexpectedEOF(getEjbqlString());
                }
                else {
                    result = EJBQLException.syntaxErrorAt(
                        getEjbqlString(), token.getText());
                }
            }
        }
        else if (ex instanceof NoViableAltException) {
            Token token = ((NoViableAltException)ex).token;
            if (token != null) {
                if (token.getType() == Token.EOF_TYPE) {
                    result = EJBQLException.unexpectedEOF(getEjbqlString());
                }
                else {
                    result = EJBQLException.unexpectedToken(
                        getEjbqlString(), token.getText());
                }
            }
        }
        else if (ex instanceof NoViableAltForCharException) {
            NoViableAltForCharException noViableAlt = (NoViableAltForCharException)ex;
            result = EJBQLException.unexpectedChar(
                getEjbqlString(), String.valueOf((char)noViableAlt.foundChar));
        }
        else if (ex instanceof TokenStreamRecognitionException) {
            result = handleANTLRException(((TokenStreamRecognitionException)ex).recog);
        } else {
            // no special handling from aboves matches the exception if this
            // line is reached => make it a syntax error
            result = EJBQLException.syntaxError(getEjbqlString());
        }
        return result;
    }

    /**
     * INTERNAL
     * Set the errors vector
     */
    private void setErrors(Vector newErrors) {
        errors = newErrors;
    }

    /**
     * INTERNAL
     * Set the verbose flag
     */
    public void setVerbose(boolean newVerbose) {
        verbose = newVerbose;
    }

    /**
     * INTERNAL
     * Return the DISTINCT state for the parser
     */
    public short getDistinctState() {
        return getParseTree().getDistinctState();
    }

    /**
     * INTERNAL
     * Return the EJBQL String for the parser
     */
    public String getEjbqlString() {
        return theEjbql;
    }

    /**
     * INTERNAL
     * Set the EJBQL String for the parser to the passed value
     */
    public void setEjbqlString(String theEjbql) {
        this.theEjbql = theEjbql;
    }

    /**
     * All the query mechanism related things are initialized here.
     * This method is called on the *clone* of the query with
     * every execution.
     */
    public static EJBQLParser parseEJBQLString(String ejbqlString) throws QueryException {
        // Parse the ejbql string, generate the where clause and set the query's selection criteria
        EJBQLParser parser = buildParserFor(ejbqlString);
        try {
            parser.document();
        } catch (EJBQLException e) {
            parser.addError(e);
        } catch (ANTLRException e) {
            parser.addError(parser.handleANTLRException(e));
        } catch (Exception e) {
            parser.addError(e);
        }

        // Handle any errors generated by the Parser
        if (parser.hasErrors()) {
            throw (EJBQLException)parser.generateException();
        }
        return parser;
    }
}
