/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / command-line mp4 toolbox
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/m4_author.h>
#include <gpac/m4_nodes.h>

extern u32 swf_flags;
extern Float swf_flatten_angle;

void dump_file_text(char *file, char *inName, Bool xmt_dump)
{
	u32 get_file_type_by_ext(char *inName);
	M4File *mp4;
	M4Err e;
	char *szExt;
	M4SceneManager *ctx;
	FILE *dump;
	FILE *iof;
	Bool close;
	char szBuf[1024];

	e = M4OK;
	if (inName) {
		strcpy(szBuf, inName);
		strcat(szBuf, xmt_dump ? ".xml" : ".bt");
		dump = fopen(szBuf, "wt");
		close = 1;
	} else {
		dump = stdout;
		close = 0;
	}

	ctx = NewSceneManager();

	szExt = strchr(file, '.');
	switch (get_file_type_by_ext(file)) {
	case 1:
		mp4 = M4_MovieOpen(file, M4_OPEN_READ);
		if (!mp4) {
			fprintf(stdout, "Cannot open file: %s\n", M4ErrToString(M4_GetLastError(NULL)));
			M4SM_Delete(ctx);
			return;
		}
		e = M4SM_LoadContextFromMP4(ctx, mp4, NULL, NULL, NULL);
		M4_MovieDelete(mp4);
		break;
	case 3:
		iof = fopen(file, "rt");
		if (!iof) {
			fprintf(stdout, "Cannot open file %s\n", file);
			M4SM_Delete(ctx);
			return;
		}
		e = M4SM_LoadContextFromXMT(ctx, iof, NULL, NULL, NULL);
		fclose(iof);
		break;
	case 2:
		iof = fopen(file, "rt");
		if (!iof) {
			fprintf(stdout, "Cannot open file %s\n", file);
			M4SM_Delete(ctx);
			return;
		}
		e = M4SM_LoadContextFromBT(ctx, iof, NULL, NULL, NULL);
		fclose(iof);
		break;
	case 4:
		iof = fopen(file, "rb");
		if (!iof) {
			fprintf(stdout, "Cannot open file %s\n", file);
			M4SM_Delete(ctx);
			return;
		}
		e = M4SM_LoadContextFromSWF(ctx, iof, NULL, NULL, NULL, swf_flags, swf_flatten_angle);
		fclose(iof);
		break;
	}

	M4SM_DumpToText(ctx, dump, xmt_dump);

	if (close) fclose(dump);
	M4SM_Delete(ctx);
	if (e) fprintf(stdout, "Error loading scene: %s\n", M4ErrToString(e));
}


static void dump_stats(FILE *dump, M4SceneStatistics *stats)
{
	u32 i;
	s32 created, count, draw_created, draw_count, deleted, draw_deleted;
	created = count = draw_created = draw_count = deleted = draw_deleted = 0;

	fprintf(dump, "<NodeStatistics>\n");
	fprintf(dump, "<General NumberOfNodeTypes=\"%d\"/>\n", ChainGetCount(stats->node_stats));
	for (i=0; i<ChainGetCount(stats->node_stats); i++) {
		NodeStats *ptr = ChainGetEntry(stats->node_stats, i);
		fprintf(dump, "<NodeStat NodeName=\"%s\">\n", ptr->name);
		
		switch (ptr->tag) {
#ifdef M4_DEF_Bitmap
		case TAG_Bitmap:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_Background2D
		case TAG_Background2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_Circle
		case TAG_Circle:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_CompositeTexture2D
		case TAG_CompositeTexture2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_Curve2D
		case TAG_Curve2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_IndexedFaceSet2D
		case TAG_IndexedFaceSet2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_IndexedLineSet2D
		case TAG_IndexedLineSet2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_PointSet2D
		case TAG_PointSet2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_Rectangle
		case TAG_Rectangle:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_Text
		case TAG_Text:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_Ellipse
		case TAG_Ellipse:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
#ifdef M4_DEF_XCurve2D
		case TAG_XCurve2D:
			draw_count += ptr->nb_created + ptr->nb_used - ptr->nb_del;
			draw_deleted += ptr->nb_del;
			draw_created += ptr->nb_created;
			break;
#endif
		}
		fprintf(dump, "<Instanciation NbObjects=\"%d\" NbUse=\"%d\" NbDestroy=\"%d\"/>\n", ptr->nb_created, ptr->nb_used, ptr->nb_del);
		count += ptr->nb_created + ptr->nb_used;
		deleted += ptr->nb_del;
		created += ptr->nb_created;
		fprintf(dump, "</NodeStat>\n");
	}
	if (i) {
		fprintf(dump, "<CumulatedStat TotalNumberOfNodes=\"%d\" ReallyAllocatedNodes=\"%d\" DeletedNodes=\"%d\"/>\n", count, created, deleted);
		fprintf(dump, "<DrawableNodesCumulatedStat TotalNumberOfNodes=\"%d\" ReallyAllocatedNodes=\"%d\" DeletedNodes=\"%d\"/>\n", draw_count, draw_created, draw_deleted);
	}
	fprintf(dump, "</NodeStatistics>\n");

	created = count = deleted = 0;
	if (ChainGetCount(stats->proto_stats)) {
		fprintf(dump, "<ProtoStatistics>\n");
		fprintf(dump, "General NumberOfProtoUsed=\"%d\"/>\n", ChainGetCount(stats->proto_stats));
		for (i=0; i<ChainGetCount(stats->proto_stats); i++) {
			NodeStats *ptr = ChainGetEntry(stats->proto_stats, i);
			fprintf(dump, "<ProtoStat ProtoName=\"%s\">\n", ptr->name);
			fprintf(dump, "<Instanciation NbObjects=\"%d\" NbUse=\"%d\" NbDestroy=\"%d\"/>\n", ptr->nb_created, ptr->nb_used, ptr->nb_del);
			count += ptr->nb_created + ptr->nb_used;
			deleted += ptr->nb_del;
			created += ptr->nb_created;
			fprintf(dump, "</ProtoStat>\n");
		}
		if (i) fprintf(dump, "<CumulatedStat TotalNumberOfProtos=\"%d\" ReallyAllocatedProtos=\"%d\" DeletedProtos=\"%d\"/>\n", count, created, deleted);
		fprintf(dump, "</ProtoStatistics>\n");
	}
	fprintf(dump, "<FieldStatistic FieldType=\"MFVec2f\">\n");
	fprintf(dump, "<ParsingInfo NumParsed=\"%d\" NumRemoved=\"%d\"/>", stats->count_2d, stats->rem_2d);
	if (stats->count_2d) {
		fprintf(dump, "<ExtendInfo MinVec2f=\"%f %f\" MaxVec2f=\"%f %f\"/>\n", stats->min_2d.x, stats->min_2d.y, stats->max_2d.x, stats->max_2d.y);
	}
	fprintf(dump, "</FieldStatistic>\n");

	fprintf(dump, "<FieldStatistic FieldType=\"MFVec3f\">\n");
	fprintf(dump, "<ParsingInfo NumParsed=\"%d\" NumRemoved=\"%d\"/>", stats->count_3d, stats->rem_3d);
	if (stats->count_3d) {
		fprintf(dump, "<ExtendInfo MinVec3f=\"%f %f %f\" MaxVec3f=\"%f %f %f\"/>\n", stats->min_3d.x, stats->min_3d.y, stats->min_3d.z, stats->max_3d.x, stats->max_3d.y, stats->max_3d.z);
	}
	fprintf(dump, "</FieldStatistic>\n");

	fprintf(dump, "<FieldStatistic FieldType=\"MF/SFColor\">\n");
	fprintf(dump, "<ParsingInfo NumParsed=\"%d\" NumRemoved=\"%d\"/>", stats->count_color, stats->rem_color);
	fprintf(dump, "</FieldStatistic>\n");

	fprintf(dump, "<FieldStatistic FieldType=\"MF/SFFloat\">\n");
	fprintf(dump, "<ParsingInfo NumParsed=\"%d\" NumRemoved=\"%d\"/>", stats->count_float, stats->rem_float);
	fprintf(dump, "</FieldStatistic>\n");

	fprintf(dump, "<FieldStatistic FieldType=\"SFVec2f\">\n");
	fprintf(dump, "<ParsingInfo NumParsed=\"%d\"/>", stats->count_2f);
	fprintf(dump, "</FieldStatistic>\n");
	fprintf(dump, "<FieldStatistic FieldType=\"SFVec3f\">\n");
	fprintf(dump, "<ParsingInfo NumParsed=\"%d\"/>", stats->count_3f);
	fprintf(dump, "</FieldStatistic>\n");
}


static void ReorderAU(Chain *sample_list, M4AUContext *au)
{
	u32 i;
	for (i=0; i<ChainGetCount(sample_list); i++) {
		M4AUContext *ptr = ChainGetEntry(sample_list, i);
		if (
			/*time ordered*/
			(ptr->timing_sec > au->timing_sec) 
			/*set bifs first*/
			|| ((ptr->timing_sec == au->timing_sec) && (ptr->owner->streamType < au->owner->streamType))
		) {
			ChainInsertEntry(sample_list, au, i);
			return;
		}
	}
	ChainAddEntry(sample_list, au);
}
void dump_scene_stats(M4File *file, char *inName, u32 stat_level)
{
	M4Err e;
	FILE *dump;
	Bool close;
	u32 i, j;
	char szBuf[1024];
	M4SceneManager *ctx;
	LPSTATMAN sm;	
	Chain *sample_list;
	LPSCENEGRAPH sg;

	dump = NULL;
	sm = NULL;
	sample_list = NULL;
	
	close = 0;
	ctx = NewSceneManager();
	fprintf(stdout, "Loading scene... ");
	e = M4SM_LoadContextFromMP4(ctx, file, NULL, NULL, NULL);
	if (e) goto exit;
	fprintf(stdout, "OK\n");

	if (inName) {
		strcpy(szBuf, inName);
		strcat(szBuf, "_stat.xml");
		dump = fopen(szBuf, "wt");
		close = 1;
	} else {
		dump = stdout;
		close = 0;
	}

	fprintf(dump, "<!-- BIFS Scene Statistics Generated by MP4Box - GPAC -->\n");
	fprintf(dump, "<SceneStatistics file=\"%s.mp4\" DumpType=\"%s\">\n", inName, (stat_level==1) ? "full scene" : ((stat_level==2) ? "AccessUnit based" : "SceneGraph after each AU"));

	sm = M4SM_NewStatisitics();

	/*stat level 1: complete scene stat*/
	if (stat_level == 1) {
		e = M4SM_GetStatistics(sm, ctx);
		if (!e) dump_stats(dump, M4SM_GetStatisitics(sm) );
		goto exit;
	}
	/*re_order all BIFS-AUs*/
	sample_list = NewChain();
	/*configure all systems streams we're dumping*/
	for (i=0; i<ChainGetCount(ctx->streams); i++) {
		M4StreamContext *sc = ChainGetEntry(ctx->streams, i);
		if (sc->streamType != M4ST_BIFS) continue;
		for (j=0; j<ChainGetCount(sc->AUs); j++) {
			M4AUContext *au = ChainGetEntry(sc->AUs, j);
			ReorderAU(sample_list, au);
		}
	}
	sg = NULL;
	for (i=0; i<ChainGetCount(sample_list); i++) {
		M4AUContext *au = ChainGetEntry(sample_list, i);
		
		for (j=0; j<ChainGetCount(au->commands); j++) {
			SGCommand *com = ChainGetEntry(au->commands, j);
			if (com->tag == SG_SceneReplace) sg = com->graph;
			/*stat level 2 - get command stats*/
			if (stat_level==2) {
				e = M4SM_GetCommandStatistics(sm, com);
				if (e) goto exit;
			}
			/*stat level 3 - apply command*/
			if (stat_level==3) SG_ApplyCommand(sg, com);
		}
		/*stat level 3: get graph stat*/
		if (stat_level==3) {
			e = M4SM_GetGraphStatistics(sm, sg);
			if (e) goto exit;
		}
		if (stat_level==2) {
			fprintf(dump, "<AUStatistics StreamID=\"%d\" AUTime=\"%d\">\n", au->owner->ESID, au->timing);
		} else {
			fprintf(dump, "<GraphStatistics StreamID=\"%d\" AUTime=\"%d\">\n", au->owner->ESID, au->timing);
		}
		/*dump stats*/
		dump_stats(dump, M4SM_GetStatisitics(sm) );
		/*reset stats*/
		M4SM_ResetStatisitics(sm);
		if (stat_level==2) {
			fprintf(dump, "</AUStatistics>\n");
		} else {
			fprintf(dump, "</GraphStatistics>\n");
		}
	}


exit:	
	if (sample_list) DeleteChain(sample_list);
	if (sm) M4SM_DeleteStatisitics(sm);
	M4SM_Delete(ctx);
	if (e) {
		fprintf(stdout, "%s\n", M4ErrToString(e));
	} else {
		fprintf(dump, "</SceneStatistics>\n");
	}

	if (dump && close) fclose(dump);
}

void PrintFloat(Float val, Bool add_space)
{
	if (add_space) fprintf(stdout, " ");
	if (val==M4_MIN_FLOAT) fprintf(stdout, "-I");
	else if (val==M4_MAX_FLOAT) fprintf(stdout, "+I");
	else fprintf(stdout, "%g", val);
}

void PrintNodeSFField(u32 type, void *far_ptr)
{
	if (!far_ptr) return;
	switch (type) {
	case FT_SFBool:
		fprintf(stdout, "%s", (*(SFBool *)far_ptr) ? "TRUE" : "FALSE");
		break;
	case FT_SFInt32:
		fprintf(stdout, "%d", (*(SFInt32 *)far_ptr));
		break;
	case FT_SFFloat:
		PrintFloat((*(SFFloat *)far_ptr), 0);
		break;
	case FT_SFTime:
		fprintf(stdout, "%g", (*(SFTime *)far_ptr));
		break;
	case FT_SFVec2f:
		PrintFloat(((SFVec2f *)far_ptr)->x, 0);
		PrintFloat(((SFVec2f *)far_ptr)->y, 1);
		break;
	case FT_SFVec3f:
		PrintFloat(((SFVec3f *)far_ptr)->x, 0);
		PrintFloat(((SFVec3f *)far_ptr)->y, 1);
		PrintFloat(((SFVec3f *)far_ptr)->z, 1);
		break;
	case FT_SFRotation:
		PrintFloat(((SFRotation *)far_ptr)->xAxis, 0);
		PrintFloat(((SFRotation *)far_ptr)->yAxis, 1);
		PrintFloat(((SFRotation *)far_ptr)->zAxis, 1);
		PrintFloat(((SFRotation *)far_ptr)->angle, 1);
		break;
	case FT_SFColor:
		fprintf(stdout, "%g %g %g", ((SFColor *)far_ptr)->red, ((SFColor *)far_ptr)->green, ((SFColor *)far_ptr)->blue);
		break;
	case FT_SFString:
		if (((SFString*)far_ptr)->buffer)
			fprintf(stdout, "\"%s\"", ((SFString*)far_ptr)->buffer);
		else
			fprintf(stdout, "NULL");
		break;
	}
}
void PrintNode(const char *name)
{
	const char *nname;
	SFNode *node;
	LPSCENEGRAPH sg;
	u32 tag, nbF, i;
	FieldInfo f;
	u8 qt, at;
	Float bmin, bmax;
	u32 nbBits;

	tag = Node_GetTagByName(name);
	if (!tag) {
		fprintf(stdout, "Unknown BIFS node %s\n", name);
		return;
	}

	sg = NewSceneGraph(NULL, NULL, NULL, NULL);
	node = SG_NewNode(sg, tag);
	Node_Register(node, NULL);
	nname = Node_GetName(node);
	if (!node) {
		fprintf(stdout, "Node %s not supported in current built\n", nname);
		return;
	}
	nbF = Node_GetNumFields(node, FCM_ALL);

	fprintf(stdout, "Node Syntax:\n%s {\n", nname);

	for (i=0; i<nbF; i++) {
		Node_GetField(node, i, &f);
		fprintf(stdout, "\t%s %s %s", GetEventTypeName(f.eventType), GetFieldTypeName(f.fieldType), f.name);
		if (f.fieldType==FT_SFNode) fprintf(stdout, " NULL");
		else if (f.fieldType==FT_MFNode) fprintf(stdout, " []");
		else if (SG_IsSFField(f.fieldType)) {
			fprintf(stdout, " ");
			PrintNodeSFField(f.fieldType, f.far_ptr);
		} else {
			void *ptr;
			u32 i, sftype;
			GenMFField *mffield = (GenMFField *) f.far_ptr;
			fprintf(stdout, " [");
			sftype = SG_GetSFType(f.fieldType);
			for (i=0; i<mffield->count; i++) {
				if (i) fprintf(stdout, " ");
				MFField_GetItem(f.far_ptr, f.fieldType, &ptr, i);
				PrintNodeSFField(sftype, ptr);
			}
			fprintf(stdout, "]");
		}
		if (Node_GetAQInfo(node, i, &qt, &at, &bmin, &bmax, &nbBits)) {
			if (qt) {
				fprintf(stdout, " #QP=%d", qt);
				if (qt==13) fprintf(stdout, " NbBits=%d", nbBits);
				if (bmin && bmax) {
					fprintf(stdout, " Bounds=[");
					PrintFloat(bmin, 0);
					fprintf(stdout, ",");
					PrintFloat(bmax, 0);
					fprintf(stdout, "]");
				}
			}
		}
		fprintf(stdout, "\n");
	}
	fprintf(stdout, "}\n\n");

	Node_Unregister(node, NULL);
	SG_Delete(sg);
}

