/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / command-line mp4 toolbox
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/m4_author.h>
#include <gpac/m4_bifs.h>
#include <gpac/m4_scenegraph.h>

#ifndef M4_READ_ONLY

M4File *import_file(char *inName, char *outName, Bool data_ref, Bool no_frame_drop)
{
	M4Err e;
	M4TrackImporter import;
	Bool do_vid, do_aud;
	char *ext, szName[1000];

	do_vid = 1;
	do_aud = 0;
	strcpy(szName, inName);
	ext = strrchr(inName, '.');
	if (!ext) {
		fprintf(stdout, "Unknown input file type\n");
		return NULL;
	}

	/*select switches for AVI import*/
	if (!strnicmp(ext, ".avi", 4)) {
		do_vid = do_aud = 0;
		ext = strrchr(szName, '#');
		if (ext) {
			if (!strnicmp(ext, "#audio", 6)) do_aud = 1;
			else if (!strnicmp(ext, "#video", 6)) do_vid = 1;
		} else {
			do_vid = do_aud = 1;
		}
	}

	memset(&import, 0, sizeof(M4TrackImporter));
	if (outName) {
		import.dest = M4_MovieOpen(outName, M4_OPEN_WRITE);
	} else {
		import.dest = M4_MovieOpen("temp", M4_WRITE_EDIT);
	}

	/*check duration import for AVI / mp3*/
	ext = strrchr(szName, '%');
	if (ext) {
		import.duration = atoi(ext+1) * 1000;
		ext[0] = 0;
	}
	ext = strrchr(szName, '#');
	if (ext) ext[0] = 0;

	import.in_name = szName;
	import.no_frame_drop = no_frame_drop;
	import.use_data_reference = data_ref;
	e = M4OK;
	if (do_vid) {
		import.trackID = 1;
		e = MP4T_ImportMedia(&import);
	}
	if (!e && do_aud) {
		import.trackID = 2;
		e = MP4T_ImportMedia(&import);
	}
	if (e) {
		fprintf(stdout, "Import failed %s\n", M4ErrToString(e));
		M4_MovieDelete(import.dest);
		return NULL;
	}
	return import.dest;
}

M4File *merge_files(char *inName1, char *inName2, char *outName)
{
	M4TrackImporter import;
	u32 i, tk1_a, tk1_v, tk2_a, tk2_v;
	M4File *in1, *in2;
	M4Err e;
	char szFile[MAX_FILE_PATH];

	in2 = NULL;
	import.dest = NULL;
	tk1_a = tk1_v = tk2_a = tk2_v = 0;
	in1 = M4_MovieOpen(inName1, M4_OPEN_READ);
	if (!in1) {
		fprintf(stdout, "Cannot open file %s: %s\n", inName1, M4ErrToString(M4_GetLastError(NULL)));
		return 0;
	}
	for (i=0; i<M4_GetTrackCount(in1); i++) {
		switch (M4_GetMediaType(in1, i+1)) {
		case M4_VisualMediaType:
			if (!tk1_v) tk1_v = M4_GetTrackID(in1, i+1);
			break;
		case M4_AudioMediaType:
			if (!tk1_a) tk1_a = M4_GetTrackID(in1, i+1);
			break;
		}
	}
	if (!tk1_a && !tk1_v) {
		fprintf(stdout, "Error: No audio or visual tracks found in %s\n", inName1);
		goto exit;
	}

	in2 = M4_MovieOpen(inName2, M4_OPEN_READ);
	if (!in2) {
		fprintf(stdout, "Cannot open file %s: %s\n", inName2, M4ErrToString(M4_GetLastError(NULL)));
		goto exit;
	}
	for (i=0; i<M4_GetTrackCount(in2); i++) {
		switch (M4_GetMediaType(in2, i+1)) {
		case M4_VisualMediaType:
			if (!tk2_v) tk2_v = M4_GetTrackID(in2, i+1);
			break;
		case M4_AudioMediaType:
			if (!tk2_a) tk2_a = M4_GetTrackID(in2, i+1);
			break;
		}
	}
	if (!tk2_a && !tk2_v) {
		fprintf(stdout, "Error: No audio or visual tracks found in %s\n", inName2);
		goto exit;
	}
	if (tk1_v && tk2_v) {
		fprintf(stdout, "Error: Both input files have visual tracks - cannot merge files. Please retry with extracted tracks (-single option)\n");
		goto exit;
	}
	if (tk1_a && tk2_a) {
		fprintf(stdout, "Error: Both input files have audio tracks - cannot merge files. Please retry with extracted tracks (-single option)\n");
		goto exit;
	}

	memset(&import, 0, sizeof(M4TrackImporter));

	/*no final interleaving*/
	if (outName) {
		import.dest = M4_MovieOpen(outName, M4_OPEN_WRITE);
	} else {
		import.dest = M4_MovieOpen("temp", M4_WRITE_EDIT);
	}

	import.orig = in1;
	import.trackID = tk1_v ? tk1_v : tk1_a;
	e = MP4T_ImportMedia(&import);
	if (e) {
		fprintf(stdout, "Error importing %s: %s\n", szFile, M4ErrToString(e));
		M4_MovieDelete(import.dest);
		import.dest = NULL;
		goto exit;
	}
	import.orig = in2;
	import.trackID = tk2_v ? tk2_v : tk2_a;
	e = MP4T_ImportMedia(&import);
	if (e) {
		fprintf(stdout, "Error importing %s: %s\n", szFile, M4ErrToString(e));
		M4_MovieDelete(import.dest);
		import.dest = NULL;
		goto exit;
	}

exit:
	if (in1) M4_MovieDelete(in1);
	if (in2) M4_MovieDelete(in2);
	return import.dest;
}


#endif

