// GPAC.cpp : Defines the class behaviors for the application.
//

#include "stdafx.h"
#include "Osmo4.h"

#include "MainFrm.h"
#include "OpenUrl.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// WinGPAC

BEGIN_MESSAGE_MAP(WinGPAC, CWinApp)
	//{{AFX_MSG_MAP(WinGPAC)
	ON_COMMAND(ID_APP_ABOUT, OnAppAbout)
	ON_COMMAND(ID_OPEN_FILE, OnOpenFile)
	ON_COMMAND(ID_FILE_STEP, OnFileStep)
	ON_COMMAND(ID_OPEN_URL, OnOpenUrl)
	ON_COMMAND(ID_FILE_RELOAD, OnFileReload)
	ON_COMMAND(ID_FILE_PLAY, OnFilePlay)
	ON_UPDATE_COMMAND_UI(ID_FILE_PLAY, OnUpdateFilePlay)
	ON_UPDATE_COMMAND_UI(ID_FILE_STEP, OnUpdateFilePlay)
	ON_UPDATE_COMMAND_UI(ID_FILE_RELOAD, OnUpdateFilePlay)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// WinGPAC construction

WinGPAC::WinGPAC()
{
}

/////////////////////////////////////////////////////////////////////////////
// The one and only WinGPAC object

WinGPAC theApp;


Bool Osmo4_EventProc(void *priv, M4Event *evt)
{
	u32 dur;
	WinGPAC *gpac = (WinGPAC *) priv;
	CMainFrame *pFrame = (CMainFrame *) gpac->m_pMainWnd;
	/*shutdown*/
	if (!pFrame) return 0;

	switch (evt->type) {
	case M4E_DURATION:
		dur = (u32) (1000 * evt->duration.duration);
		if (dur<1100) dur = 0;
		gpac->max_duration = dur;
		if (!dur) {
			gpac->can_seek = 0;
			pFrame->m_progBar.m_Slider.EnableWindow(FALSE);
		} else {
			gpac->can_seek = 1;
			pFrame->m_progBar.m_Slider.EnableWindow(TRUE);
			pFrame->m_progBar.m_Slider.SetRangeMin(0);
			pFrame->m_progBar.m_Slider.SetRangeMax(dur);
			pFrame->m_progBar.m_Play.EnableWindow(TRUE);
			pFrame->m_progBar.m_Stop.EnableWindow(TRUE);
		}
		break;

	case M4E_MESSAGE:
		if (evt->message.error!=M4OK) {
			if (evt->message.error<M4OK || !gpac->m_NoConsole) {
				pFrame->console_err = evt->message.error;
				pFrame->console_message = evt->message.message;
				gpac->m_pMainWnd->PostMessage(WM_CONSOLEMSG, 0, 0);
			}
			return 0;
		}
		if (gpac->m_NoConsole) return 0;

		/*process user message*/
		pFrame->console_err = M4OK;
		pFrame->console_message = evt->message.message;
		gpac->m_pMainWnd->PostMessage(WM_CONSOLEMSG, 0, 0);
		break;

	case M4E_SCENESIZE:
		pFrame->PostMessage(WM_SETSIZE, evt->size.width, evt->size.height);
		break;

	case M4E_CONNECT:
		gpac->m_isopen = evt->connect.is_connected;
		break;

	case M4E_QUIT:
		pFrame->PostMessage(WM_CLOSE, 0L, 0L);
		break;
	case M4E_VKEYDOWN:
		if (gpac->can_seek && evt->key.key_states & M4KM_ALT) {
			s32 res;
			switch (evt->key.m4_vk_code) {
			case M4VK_LEFT:
				res = M4T_GetCurrentTimeInMS(gpac->m_term) - 5*gpac->max_duration/100;
				if (res<0) res=0;
				gpac->PlayFromTime(res);
				break;
			case M4VK_RIGHT:
				res = M4T_GetCurrentTimeInMS(gpac->m_term) + 5*gpac->max_duration/100;
				if ((u32) res>=gpac->max_duration) res = 0;
				gpac->PlayFromTime(res);
				break;
			case M4VK_DOWN:
				res = M4T_GetCurrentTimeInMS(gpac->m_term) - 1000;
				if (res<0) res=0;
				gpac->PlayFromTime(res);
				break;
			case M4VK_UP:
				res = M4T_GetCurrentTimeInMS(gpac->m_term) + 1000;
				if ((u32) res>=gpac->max_duration) res = 0;
				gpac->PlayFromTime(res);
				break;
			}
		}
		break;
	case M4E_NAVIGATE:
		/*store URL since it may be destroyed, and post message*/
		gpac->m_navigate_url = evt->navigate.to_url;
		pFrame->PostMessage(WM_NAVIGATE, NULL, NULL);
		return 1;
	}
	return 0;
}


/////////////////////////////////////////////////////////////////////////////
// WinGPAC initialization

BOOL WinGPAC::InitInstance()
{
	// Standard initialization

#ifdef _AFXDLL
	Enable3dControls();			// Call this when using MFC in a shared DLL
#else
	Enable3dControlsStatic();	// Call this when linking to MFC statically
#endif

	SetRegistryKey(_T("GPAC"));

	CMainFrame* pFrame = new CMainFrame;
	m_pMainWnd = pFrame;
	pFrame->LoadFrame(IDR_MAINFRAME, WS_OVERLAPPEDWINDOW | FWS_ADDTOTITLE, NULL, NULL);

	m_pMainWnd->DragAcceptFiles();

	pFrame->SetSize(300, 200);
	pFrame->ShowWindow(SW_SHOW);
	pFrame->UpdateWindow();


	unsigned char config_path[MAX_PATH];
	strcpy((char *) config_path, AfxGetApp()->m_pszHelpFilePath);
	while (config_path[strlen((char *) config_path)-1] != '\\') config_path[strlen((char *) config_path)-1] = 0;

	/*init config and plugins*/
	m_config = NewIniFile((const char *) config_path, "GPAC.cfg");
	if (!m_config) {
		/*create blank config file in the exe dir*/
		unsigned char config_file[MAX_PATH];
		strcpy((char *) config_file, (const char *) config_path);
		strcat((char *) config_file, "GPAC.cfg");
		FILE *ft = fopen((const char *) config_file, "wt");
		fclose(ft);
		m_config = NewIniFile((const char *) config_path, "GPAC.cfg");
		if (!m_config) {
			MessageBox(NULL, "GPAC Configuration file not found", "Fatal Error", MB_OK);
			m_pMainWnd->PostMessage(WM_CLOSE);
		}
	}

	m_config_dir = config_path;

	char *str = IF_GetKey(m_config, "General", "PluginsDirectory");
	m_plugins = NewPluginManager((const unsigned char *) str, m_config);
	if (!m_plugins) {
		char *sOpt;
		/*inital launch*/
		m_plugins = NewPluginManager((const unsigned char *) config_path, m_config);
		if (m_plugins) {
			IF_SetKey(m_config, "General", "PluginsDirectory", (const char *) config_path);

			sOpt = IF_GetKey(m_config, "Render2D", "GraphicsDriver");
			if (!sOpt) IF_SetKey(m_config, "Render2D", "GraphicsDriver", "gdip_rend");

			sOpt = IF_GetKey(m_config, "Downloader", "CacheDirectory");
			if (!sOpt) {
				unsigned char str_path[MAX_PATH];
				sprintf((char *) str_path, "%scache", config_path);
				IF_SetKey(m_config, "Downloader", "CacheDirectory", (const char *) str_path);
			}
			/*setup UDP traffic autodetect*/
			IF_SetKey(m_config, "Network", "AutoReconfigUDP", "yes");
			IF_SetKey(m_config, "Network", "UDPNotAvailable", "no");
			IF_SetKey(m_config, "Network", "UDPTimeout", "10000");
			IF_SetKey(m_config, "Network", "BufferLength", "3000");
		}

		/*check audio config on windows, force config*/
		sOpt = IF_GetKey(m_config, "Audio", "ForceConfig");
		if (!sOpt) {
			IF_SetKey(m_config, "Audio", "ForceConfig", "yes");
			IF_SetKey(m_config, "Audio", "NumBuffers", "8");
			IF_SetKey(m_config, "Audio", "BuffersPerSecond", "16");
		}
		/*by default use GDIplus, much faster than freetype on font loading*/
		IF_SetKey(m_config, "FontEngine", "DriverName", "gdip_rend");

	}	
	if (! PM_GetPluginsCount(m_plugins) ) {
		MessageBox(NULL, "No plugins available - system cannot work", "Fatal Error", MB_OK);
		m_pMainWnd->PostMessage(WM_CLOSE);
	}

	/*setup font dir*/
	str = IF_GetKey(m_config, "FontEngine", "FontDirectory");
	if (!str) {
		::GetWindowsDirectory((char*)config_path, MAX_PATH);
		if (config_path[strlen((char*)config_path)-1] != '\\') strcat((char*)config_path, "\\");
		strcat((char *)config_path, "Fonts");
		IF_SetKey(m_config, "FontEngine", "FontDirectory", (const char *) config_path);
	}

	/*check video driver, if none or raw_out use dx_hw by default*/
	str = IF_GetKey(m_config, "Video", "DriverName");
	if (!str || !stricmp(str, "raw_out")) {
		IF_SetKey(m_config, "Video", "DriverName", "dx_hw");
	}

	/*setup user*/
	memset(&m_user, 0, sizeof(M4User));
	m_user.config = m_config;
	m_user.plugins = m_plugins;
	m_user.opaque = this;
	m_user.os_window_handler = pFrame->m_pWndView->m_hWnd;
	m_user.EventProc = Osmo4_EventProc;


	m_prev_time = 0;
	m_term = NewMPEG4Term(&m_user);
	if (! m_term) {
		MessageBox(NULL, "Cannot load MPEG-4 Terminal", "Fatal Error", MB_OK);
		m_pMainWnd->PostMessage(WM_CLOSE);
	}

	SetOptions();

	pFrame->SetSize(300, 200);
	pFrame->UpdateWindow();

	CCommandLineInfo cmdInfo;
	ParseCommandLine(cmdInfo);

	m_url_changed = 0;
	if (! cmdInfo.m_strFileName.IsEmpty()) {
		m_filename = cmdInfo.m_strFileName;
		m_url_changed = 1;
		Sleep(1100);
		m_pMainWnd->PostMessage(WM_OPENURL);
	}

	return TRUE;
}


void WinGPAC::ReloadTerminal()
{
	Bool reconnect = m_isopen;
	CMainFrame *pFrame = (CMainFrame *) m_pMainWnd;
	pFrame->console_err = M4OK;
	pFrame->console_message = "Reloading MPEG-4 Terminal";
	m_pMainWnd->PostMessage(WM_CONSOLEMSG, 0, 0);

	M4T_Delete(m_term);
	m_term = NewMPEG4Term(&m_user);
	if (!m_term) {
		MessageBox(NULL, "Fatal Error !!", "Couldn't change renderer", MB_OK);
		m_pMainWnd->PostMessage(WM_DESTROY);
		return;
	}
	if (reconnect) {
		m_url_changed = 1;
		m_pMainWnd->PostMessage(WM_OPENURL);
	}
	pFrame->console_message = "MPEG-4 Terminal reloaded";
	m_pMainWnd->PostMessage(WM_CONSOLEMSG, 0, 0);
}


/////////////////////////////////////////////////////////////////////////////
// WinGPAC message handlers





/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
	CAboutDlg();

// Dialog Data
	//{{AFX_DATA(CAboutDlg)
	enum { IDD = IDD_ABOUTBOX };
	//}}AFX_DATA

	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(CAboutDlg)
	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support
	//}}AFX_VIRTUAL

// Implementation
protected:
	//{{AFX_MSG(CAboutDlg)
	virtual BOOL OnInitDialog();
	afx_msg void OnGogpac();
	afx_msg void OnGoosmo4();
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialog(CAboutDlg::IDD)
{
	//{{AFX_DATA_INIT(CAboutDlg)
	//}}AFX_DATA_INIT
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAboutDlg)
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialog)
	//{{AFX_MSG_MAP(CAboutDlg)
	ON_BN_CLICKED(IDC_GOGPAC, OnGogpac)
	ON_BN_CLICKED(IDC_GOOSMO4, OnGoosmo4)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

// App command to run the dialog
void WinGPAC::OnAppAbout()
{
	CAboutDlg aboutDlg;
	aboutDlg.DoModal();
}

BOOL CAboutDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();
	CString str = "Osmo4 / GPAC - version ";
	str += M4_VERSION;
	SetWindowText(str);
	return TRUE;  
}

void CAboutDlg::OnGogpac() 
{
	ShellExecute(NULL, "open", "http://gpac.sourceforge.net", NULL, NULL, SW_SHOWNORMAL);
}

void CAboutDlg::OnGoosmo4() 
{
	ShellExecute(NULL, "open", "http://www.comelec.enst.fr/osmo4", NULL, NULL, SW_SHOWNORMAL);
}

/////////////////////////////////////////////////////////////////////////////
// WinGPAC message handlers


int WinGPAC::ExitInstance() 
{

	M4T_Delete(m_term);
	PM_Delete(m_plugins);
	IF_Delete(m_config);
	return CWinApp::ExitInstance();
}

void WinGPAC::SetOptions()
{
	char *sOpt = IF_GetKey(m_config, "General", "Loop");
	m_Loop = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
	sOpt = IF_GetKey(m_config, "General", "StopAtEnd");
	m_StopAtEnd = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
	sOpt = IF_GetKey(m_config, "General", "ConsoleOff");
	m_NoConsole = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
	sOpt = IF_GetKey(m_config, "General", "ViewXMT");
	m_ViewXMTA  = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
}


void WinGPAC::OnOpenUrl() 
{
	COpenUrl url;
	if (url.DoModal() != IDOK) return;
	m_url_changed = 1;
	m_pMainWnd->PostMessage(WM_OPENURL);
}

void WinGPAC::OnOpenFile() 
{
	u32 keyCount, i;
	CString sFiles;
	sFiles = "All Files (*.*)|*.*|";
	keyCount = IF_GetKeyCount(m_config, "FileAssociations");
	for (i=0; i<keyCount; i++) {
		const char *sKey;
		sKey = IF_GetKeyName(m_config, "FileAssociations", i);
		if (!sKey) continue;
		CString sOpt = IF_GetKey(m_config, "FileAssociations", sKey);
		while (1) {
			int pos = sOpt.Find(' ');
			CString ext = (pos==-1) ? sOpt : sOpt.Left(pos);
			if (sFiles.Find(ext) < 0) {
				CString n;
				n.Format("%s Files (*.%s)|*.%s|", (LPCTSTR) ext, (LPCTSTR) ext, (LPCTSTR) ext),
				sFiles += n;
			}

			if (sOpt==ext) break;
			CString rem;
			rem.Format("%s ", (LPCTSTR) ext);
			sOpt.Replace((LPCTSTR) rem, "");
		}
	}
	
	CFileDialog fd(TRUE,NULL,NULL,OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT, sFiles);
	if (fd.DoModal()!=IDOK) return;
	m_filename = fd.GetPathName();
	m_url_changed = 1;
	m_pMainWnd->PostMessage(WM_OPENURL);
}


void WinGPAC::Pause()
{
	if (!m_isopen) return;
	m_paused = !m_paused;
	M4T_Pause(m_term, m_paused);
}

void WinGPAC::OnMainPause() 
{
	Pause();	
}

void WinGPAC::OnFileStep() 
{
	M4T_Pause(m_term, 2);
	m_paused = 1;
}

void WinGPAC::PlayFromTime(u32 time)
{
	if (m_paused) Pause();
	M4T_PlayFromTime(m_term, time);
}


void WinGPAC::OnFileReload() 
{
	M4T_CloseURL(m_term);
	m_url_changed = 1;
	m_pMainWnd->PostMessage(WM_OPENURL);
}

void WinGPAC::OnFilePlay() 
{
	Pause();	
}

void WinGPAC::OnUpdateFilePlay(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_isopen);	
	if (pCmdUI->m_nID==ID_FILE_PLAY) {
		pCmdUI->SetText(m_isopen ? (m_paused ? "Resume\tCtrl+P" : "Pause\tCtrl+P") : "Play/Pause\tCtrl+P");
	}
}
