/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004
 *					All rights reserved
 *
 *  This file is part of GPAC / Osmo4 wxWidgets GUI
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 *		
 */

#include "fileprops.h"
#include "wxOsmo4.h"
#include <wx/filename.h>


wxFileProps::wxFileProps(wxWindow *parent)
             : wxDialog(parent, -1, wxString(_T("File Properties")))
{

	m_pApp = (wxOsmo4Frame *)parent;
	SetSize(520, 260);

    m_pTreeView = new wxTreeCtrl(this, ID_TREE_VIEW, wxPoint(4, 2), wxSize(200, 180), wxTR_DEFAULT_STYLE | wxSUNKEN_BORDER);
    m_pODView = new wxTextCtrl(this, -1, "", wxPoint(210, 2), wxSize(300, 180), wxTE_MULTILINE | wxTE_READONLY | wxHSCROLL | wxSUNKEN_BORDER);

	m_pBufferView = new wxTextCtrl(this, -1, "", wxPoint(210, 184), wxSize(300, 50), wxTE_MULTILINE | wxTE_READONLY | wxSUNKEN_BORDER);

#ifdef WIN32
	m_pODView->SetBackgroundColour(wxColour("LIGHT GREY"));
	m_pBufferView->SetBackgroundColour(wxColour("LIGHT GREY"));
#endif

	m_pViewWI = new wxButton(this, ID_VIEW_WI, "View World Info", wxPoint(4, 184), wxSize(200, 20));
	m_pViewSG = new wxButton(this, ID_VIEW_SG, "View Scene Graph", wxPoint(4, 210), wxSize(200, 20));


	wxString str = wxFileName(m_pApp->the_url).GetName();
	str += " Properties";
	SetTitle(str);

	m_pTimer = new wxTimer();
	m_pTimer->SetOwner(this, ID_OD_TIMER);
	m_pTimer->Start(500, 0);
	RewriteODTree();

}

wxFileProps::~wxFileProps()
{
	m_pTimer->Stop();
	delete m_pTimer;
}


BEGIN_EVENT_TABLE(wxFileProps, wxDialog)
	EVT_TREE_ITEM_ACTIVATED(ID_TREE_VIEW, wxFileProps::OnSetSelection)
	EVT_TREE_SEL_CHANGED(ID_TREE_VIEW, wxFileProps::OnSetSelection)
	EVT_TREE_ITEM_EXPANDED(ID_TREE_VIEW, wxFileProps::OnSetSelection)
	EVT_TREE_ITEM_COLLAPSED(ID_TREE_VIEW, wxFileProps::OnSetSelection)
	EVT_TIMER(ID_OD_TIMER, wxFileProps::OnTimer)
	EVT_BUTTON(ID_VIEW_SG, wxFileProps::OnViewSG)
	EVT_BUTTON(ID_VIEW_WI, wxFileProps::OnViewWorld)
END_EVENT_TABLE()

void wxFileProps::RewriteODTree()
{
	ODInfo info;
	
	LPODMANAGER root_odm = M4T_GetRootOD(m_pApp->m_term);
	if (!root_odm) return;

	m_pTreeView->DeleteAllItems();
	ODTreeData *root = new ODTreeData(root_odm);
	m_pTreeView->AddRoot("Root OD", -1, -1, root);
	wxTreeItemId rootId = m_pTreeView->GetRootItem();

	/*browse all remotes*/
	while (1) {
		LPODMANAGER remote = M4T_GetRemoteOD(m_pApp->m_term, root_odm);
		if (!remote) break;
		if (M4T_GetODInfo(m_pApp->m_term, root_odm, &info) != M4OK) break;
		m_pTreeView->SetItemText(rootId, wxString::Format("Remote OD (%s)", info.od->URLString) );

		root = new ODTreeData(remote);
		m_pTreeView->AppendItem(rootId, "Object Descriptor", -1, -1, root);
		m_pTreeView->SetItemText(rootId, "Inline Scene");
		rootId = root->GetId();
		root_odm = remote;
	}

	WriteInlineTree(root);
}

void wxFileProps::WriteInlineTree(ODTreeData *root)
{
	ODInfo info;

	/*browse all ODs*/
	u32 count = M4T_GetODCount(m_pApp->m_term, root->m_pODMan);

	for (u32 i=0; i<count; i++) {
		LPODMANAGER odm = M4T_GetODManager(m_pApp->m_term, root->m_pODMan, i);
		if (!odm) return;
		ODTreeData *odd = new ODTreeData(odm);
		m_pTreeView->AppendItem(root->GetId(), "Object Descriptor", -1, -1, odd);

		/*browse all remotes*/
		while (1) {
			LPODMANAGER remote = M4T_GetRemoteOD(m_pApp->m_term, odm);
			if (!remote) break;
			if (M4T_GetODInfo(m_pApp->m_term, odm, &info) != M4OK) break;
			m_pTreeView->SetItemText(odd->GetId(), wxString::Format("Remote OD (%s)", info.od->URLString) );

			ODTreeData *rem = new ODTreeData(remote);
			m_pTreeView->AppendItem(odd->GetId(), "Object Descriptor", -1, -1, rem);
			odd = rem;
			odm = remote;
		}

		/*if inline propagate*/
		switch (M4T_IsInlineOD(m_pApp->m_term, odm)) {
		case 1:
			m_pTreeView->SetItemText(odd->GetId(), "Inline Scene");
			WriteInlineTree(odd);
			break;
		case 2:
			m_pTreeView->SetItemText(odd->GetId(), "Extern Proto Lib");
			break;
		default:
			break;
		}
	}
}

void wxFileProps::OnSetSelection(wxTreeEvent& event)
{
	ODTreeData *odd = (ODTreeData *) m_pTreeView->GetItemData(event.GetItem());
	SetODInfo(odd->m_pODMan);
}

void wxFileProps::SetODInfo(LPODMANAGER odm)
{
	wxString info;
	ODInfo odi;
	u32 h, m, s;

	info = "";
	if (M4T_GetODInfo(m_pApp->m_term, odm, &odi) != M4OK) return;

	m_current_odm = odm;

//	m_OD_ID = odi.od->objectDescriptorID;

	if (odi.has_profiles) {
		info += wxString::Format("Initial Object Descriptor ID %d - ", odi.od->objectDescriptorID);
		if (odi.duration) {
			h = (u32) (odi.duration / 3600);
			m = (u32) (odi.duration / 60) - h*60;
			s = (u32) (odi.duration) - h*3600 - m*60;
			info += wxString::Format("Duration %02d:%02d:%02d\n", h, m, s);
		} else {
			info += "Unknown duration\n";
		}
		info += wxString::Format("\tOD Profile@Level %d\n", odi.OD_pl);
		info += wxString::Format("\tScene Profile@Level %d\n", odi.scene_pl);
		info += wxString::Format("\tGraphics Profile@Level %d\n", odi.graphics_pl);
		info += wxString::Format("\tAudio Profile@Level %d\n", odi.audio_pl);
		info += wxString::Format("\tVisual Profile@Level %d\n", odi.scene_pl);
		info += wxString::Format("\tInline Content Profiled %s\n", odi.inline_pl ? "yes" : "no");
		info += "\n";
	} else {
		info += wxString::Format("Object Descriptor ID %d - ", odi.od->objectDescriptorID);
		if (odi.duration) {
			h = (u32) (odi.duration / 3600);
			m = (u32) (odi.duration / 60) - h*60;
			s = (u32) (odi.duration) - h*3600 - m*60;
			info += wxString::Format("Duration %02d:%02d:%02d\n", h, m, s);
		} else {
			info += "Unknown duration\n";
		}
	}

	if (odi.owns_service) {
		info += wxString::Format("Service Handler: %s\n", odi.service_handler);
		info += wxString::Format("Service URL: %s\n", odi.service_url);
	}
	
	if (odi.od->URLString) {
		info += wxString::Format("Remote OD - URL: %s\n", odi.od->URLString);
		goto common;
	} 
	SetODTime();

	info += "\n";
	if (odi.codec_name) {
		switch (odi.od_type) {
		case M4ST_VISUAL:
			info += wxString::Format("Video Object: Width %d - Height %d\n", odi.width, odi.height);
			break;
		case M4ST_AUDIO:
			info += wxString::Format("Audio Object: Sample Rate %d - %d channels\n", odi.sample_rate, odi.num_channels);
			break;
		}
		info += wxString::Format("Media Codec %s\n", odi.codec_name);
	}
	info += "\n";

	/*dump ES info*/
	u32 count;
	count = ChainGetCount(odi.od->ESDescriptors);
	u32 i, j;
	info += wxString::Format("%d streams in object\n", count);

	for (i=0; i<count; i++) {
		ESDescriptor *esd = (ESDescriptor *) ChainGetEntry(odi.od->ESDescriptors, i);
		info += wxString::Format("Stream ID %d - Clock ID %d\n", esd->ESID, esd->OCRESID);
		if (esd->dependsOnESID) {
			info += wxString::Format("\tDepends on Stream ID %d for decoding\n", esd->dependsOnESID);
		}
		switch (esd->decoderConfig->streamType) {
		case M4ST_OD:
			info += wxString::Format("\tOD Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		case M4ST_OCR:
			info += "\tOCR Stream\n";
			break;
		case M4ST_BIFS:
			info += wxString::Format("\tBIFS Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		case M4ST_VISUAL:
			info += "\tVisual Stream - media type: ";
			switch (esd->decoderConfig->objectTypeIndication) {
			case 0x20: info += "MPEG-4\n"; break;
			case 0x60: info += "MPEG-2 Simple Profile\n"; break;
			case 0x61: info += "MPEG-2 Main Profile\n"; break;
			case 0x62: info += "MPEG-2 SNR Profile\n"; break;
			case 0x63: info += "MPEG-2 Spatial Profile\n"; break;
			case 0x64: info += "MPEG-2 High Profile\n"; break;
			case 0x65: info += "MPEG-2 422 Profile\n"; break;
			case 0x6A: info += "MPEG-1\n"; break;
			case 0x6C: info += "JPEG\n"; break;
			case 0x6D: info += "PNG\n"; break;
			default: info += "Private Type\n"; break;
			}
			break;

		case M4ST_AUDIO:
			info += "\tAudio Stream - media type: ";
			switch (esd->decoderConfig->objectTypeIndication) {
			case 0x40: info += "MPEG-4\n"; break;
			case 0x66: info += "MPEG-2 AAC Main Profile\n"; break;
			case 0x67: info += "MPEG-2 AAC LowComplexity Profile\n"; break;
			case 0x68: info += "MPEG-2 AAC Scalable Sampling Rate Profile\n"; break;
			case 0x69: info += "MPEG-2 Audio\n"; break;
			case 0x6B: info += "MPEG-1 Audio\n"; break;
			default: info += "Private/Unknown\n"; break;
			}
			break;
		case M4ST_MPEG7:
			info += wxString::Format("\tMPEG-7 Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		case M4ST_IPMP:
			info += wxString::Format("\tIPMP Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		case M4ST_OCI:
			info += wxString::Format("\tOCI Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		case M4ST_MPEGJ:
			info += wxString::Format("\tMPEGJ Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		case M4ST_INTERACT:
			info += wxString::Format("\tUser Interaction Stream - version %d\n", esd->decoderConfig->objectTypeIndication);
			break;
		default:
			info += "Private/Unknown\n";
			break;
		}

		info += wxString::Format("\tBuffer Size %d\n\tAverage Bitrate %d bps\n\tMaximum Bitrate %d bps\n", esd->decoderConfig->bufferSizeDB, esd->decoderConfig->avgBitrate, esd->decoderConfig->maxBitrate);
		info += wxString::Format("\tStream Clock Resolution %d\n", esd->slConfig->timestampResolution);
		if (esd->URLString) info += wxString::Format("\tStream Location: %s\n", esd->URLString);

		/*check language*/
		if (ChainGetCount(esd->langDesc)) {
			LanguageDescriptor *ld = (LanguageDescriptor *)ChainGetEntry(esd->langDesc, 0);
			info += wxString::Format("\tStream Language: %c%c%c\n", (char) ((ld->langCode>>16)&0xFF), (char) ((ld->langCode>>8)&0xFF), (char) (ld->langCode & 0xFF) );
		}

	}
	

common:

	/*check OCI (not everything interests us) - FIXME: support for unicode*/
	for (i=0; i<ChainGetCount(odi.od->OCIDescriptors); i++) {
		Descriptor *desc = (Descriptor *) ChainGetEntry(odi.od->OCIDescriptors, i);
		switch (desc->tag) {
		case SegmentDescriptor_Tag:
		{
			SegmentDescriptor *sd = (SegmentDescriptor *) desc;
			info += wxString::Format("\nSegment Descriptor:\nName: %s - start time %g sec - duration %g sec\n", sd->SegmentName, sd->startTime, sd->Duration);
		}
			break;
		case ContentCreatorName_Tag:
		{
			ContentCreatorNameDescriptor *ccn = (ContentCreatorNameDescriptor *)desc;
			info += "\nContent Creators:\n";
			for (j=0; j<ChainGetCount(ccn->ContentCreators); j++) {
				contentCreatorInfo *ci = (contentCreatorInfo *) ChainGetEntry(ccn->ContentCreators, j);
				if (!ci->isUTF8) continue;
				info += wxString::Format("\t%s\n", ci->contentCreatorName);
			}
		}
			break;

		case ShortTextualDescriptor_Tag:
			{
				ShortTextualDescriptor *std = (ShortTextualDescriptor *)desc;
				info += wxString::Format("\n%s: %s\n", std->eventName, std->eventText);
			}
			break;
		/*todo*/
		case ContentCreationDate_Tag:
			break;
		default:
			break;
		}

	}

	m_pODView->Clear();
	m_pODView->AppendText(info);
}



void wxFileProps::SetODTime()
{
	ODInfo odi;
	wxString info;
	u32 h, m, s;

	if (!m_current_odm) {
		m_pBufferView->SetLabel("");
		return;
	}
	if (M4T_GetODInfo(m_pApp->m_term, m_current_odm, &odi)) return;

	info = "";
	switch (odi.status) {
	case 1:
		h = (u32) (odi.current_time / 3600);
		m = (u32) (odi.current_time / 60) - h*60;
		s = (u32) (odi.current_time) - h*3600 - m*60;
		info += wxString::Format("Playing %02d:%02d:%02d", h, m, s);
		break;
	case 2:
		info += "Paused";
		break;
	default:
		info += "Stoped";
		break;
	}
	/*get buffering*/
	if (odi.buffer>=0) info += wxString::Format(" - Buffer: %d ms", odi.buffer);
	else if (odi.buffer==-1) info += " - Not buffering";
	else info += " - Not Playing";

	/*get clock drift*/
	info += wxString::Format(" - Clock drift: %d ms\n", odi.clock_drift);
	/*get DB occupation*/
	if (odi.buffer>=0) info += wxString::Format("Decoding Buffer: %d Units\n", odi.db_unit_count);
	/*get CB occupation*/
	if (odi.cb_max_count) info += wxString::Format("Composition Memory: %d Units (%d max)", odi.cb_unit_count, odi.cb_max_count);

	m_pBufferView->Clear();
	m_pBufferView->AppendText(info);
}

void wxFileProps::OnTimer(wxTimerEvent& WXUNUSED(event))
{
	SetODTime();
}



void wxFileProps::OnViewWorld(wxCommandEvent &WXUNUSED(event)) 
{
#ifdef M4_DEF_WorldInfo
	wxString wit;
	B_WorldInfo *wi = M4T_GetWorldInfo(m_pApp->m_term, m_current_odm);
	if (!wi) {
		wxMessageDialog(this, "No World Info available", "Sorry!", wxOK).ShowModal();
		return;
	}

	wit = "";
	for (u32 i=0; i<wi->info.count; i++) {
		wit += wi->info.vals[i];
		wit += "\n";
	}
	wxMessageDialog(this, wit, wxString((const char *) wi->title.buffer), wxOK).ShowModal();
#else
	wxMessageDialog(this, "Sorry!", "World Info support not built-in", wxOK).ShowModal();
#endif
}

void wxFileProps::OnViewSG(wxCommandEvent &WXUNUSED(event)) 
{
	char *sOpt;
	Bool dump_xmt;
	wxFileName out_file;

	sOpt = IF_GetKey(m_pApp->m_client.config, "Downloader", "CacheDirectory");
	out_file.AssignDir(sOpt);

	sOpt = IF_GetKey(m_pApp->m_client.config, "General", "ViewXMT");
	if (sOpt && !stricmp(sOpt, "yes")) {
		dump_xmt = 1;
		out_file.SetFullName("scene_dump.xml");
	} else {
		out_file.SetFullName("scene_dump.bt");
		dump_xmt = 0;
	}

	FILE *dump = fopen(out_file.GetFullPath().c_str(), "wt");
	if (!dump) {
		wxMessageDialog dlg(this, wxString::Format("Cannot open temporary dump file %s", out_file.GetFullPath().c_str()), "Error", wxOK);
		dlg.ShowModal();
		return;
	}

	M4Err e = M4T_DumpSceneGraph(m_pApp->m_term, dump, dump_xmt, 0, m_current_odm);
	fclose(dump);
	if (e) {
		wxMessageDialog dlg(this, M4ErrToString(e), "Error while dumping", wxOK);
		dlg.ShowModal();
	} else {
		wxString cmd = get_pref_browser(m_pApp->m_client.config);
		cmd += " ";
		cmd += out_file.GetFullPath();
		wxExecute(cmd);
	}
}
