/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004
 *					All rights reserved
 *
 *  This file is part of GPAC / Osmo4 wxWidgets GUI
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 *		
 */

#include "wxOsmo4.h"
#include <gpac/m4_decoder.h>
#include <gpac/m4_graphics.h>
#include <gpac/m4_avhw.h>


#include "wxGPACControl.h"

/*ISO 639-2 code names (complete set in /doc directory)*/
#define NUM_LANGUAGE	59
static char *Languages[118] = 
{
"Albanian","alb",
"Arabic","ara",
"Armenian","arm",
"Azerbaijani","aze",
"Basque","baq",
"Belarusian","bel",
"Bosnian","bos",
"Breton","bre",
"Bulgarian","bul",
"Catalan","cat",
"Celtic (Other)","cel",
"Chinese","chi",
"Croatian","scr",
"Czech","cze",
"Danish","dan",
"Dutch","dut",
"English","eng",
"Esperanto","epo",
"Estonian","est",
"Fijian","fij",
"Finnish","fin",
"French","fre",
"Georgian","geo",
"German","ger",
"Greek, Modern (1453-)","gre",
"Haitian","hat",
"Hawaiian","haw",
"Hebrew","heb",
"Indonesian","ind",
"Iranian (Other)","ira",
"Irish","gle",
"Italian","ita",
"Japanese","jpn",
"Korean","kor",
"Kurdish","kur",
"Latin","lat",
"Lithuanian","lit",
"Luxembourgish","ltz",
"Macedonian","mac",
"Mongolian","mon",
"Norwegian","nor",
"Occitan (post 1500)","oci",
"Persian","per",
"Philippine (Other)","phi" ,
"Polish","pol",
"Portuguese","por",
"Russian","rus",
"Serbian","srp",
"Slovak","slo",
"Slovenian","slv",
"Somali","som",
"Spanish","spa",
"Swedish","swe",
"Tahitian","tah",
"Thai","tha",
"Tibetan","tib",
"Turkish","tur",
"Undetermined","und",
"Vietnamese","vie",
};

#define NUM_RATES 11
static char *BIFSRates[11] = 
{
	"5.0",
	"7.5",
	"10.0",
	"12.5",
	"15.0",
	"24.0",
	"25.0",
	"30.0",
	"50.0",
	"60.0",
	"100.0"
};

wxGPACControl::wxGPACControl(wxWindow *parent)
             : wxDialog(parent, -1, wxString(_T("GPAC Control Panel")))
{
	const char *sOpt;
	SetSize(320, 240);
	u32 i;
	Centre();

	m_pApp = (wxOsmo4Frame *)parent;

	/*main selection*/
	new wxStaticText(this, 0, _T("Category"), wxPoint(25, 8), wxSize(60, 20));
	m_select = new wxComboBox(this, ID_SELECT, _T(""), wxPoint(85, 5), wxSize(160, 20), 0, NULL, wxCB_READONLY);
	new wxButton(this, ID_APPLY, _T("Apply"), wxPoint(260, 5), wxSize(42, 20));
	new wxStaticBox(this, 0, _T(""), wxPoint(5, 25), wxSize(305, 185), wxSUNKEN_BORDER|wxDOUBLE_BORDER |wxSTATIC_BORDER );
	
	/*general section*/
	m_loop = new wxCheckBox(this, 0, "Loop at End", wxPoint(10, 40), wxSize(140, 20));
	m_stopatend = new wxCheckBox(this, 0, "Stop when over", wxPoint(180, 40), wxSize(140, 20));
	m_noconsole = new wxCheckBox(this, 0, "Disable console messages", wxPoint(10, 80), wxSize(180, 20));
	m_viewxmt = new wxCheckBox(this, 0, "View graph in XMT-A format", wxPoint(10, 120), wxSize(180, 20));
#ifdef WIN32
	m_makedefault = new wxButton(this, ID_MAKE_DEF, "Set GPAC as default MPEG-4 Player", wxPoint(10, 180), wxSize(290, 20));
#endif

	/*MPEG-4 systems*/
	m_langtxt = new wxStaticText(this, 0, _T("Prefered Stream Language"), wxPoint(10, 40), wxSize(160, 30), wxALIGN_LEFT);
	m_lang = new wxComboBox(this, 0, _T(""), wxPoint(180, 40), wxSize(120, 30), 0, NULL, wxCB_READONLY);
	m_threadtxt = new wxStaticText(this, 0, _T("Decoder Threading Mode"), wxPoint(10, 90), wxSize(160, 30), wxALIGN_LEFT);
	m_thread = new wxComboBox(this, 0, _T(""), wxPoint(180, 90), wxSize(120, 30), 0, NULL, wxCB_READONLY);
	m_bifsalwaysdrawn = new wxCheckBox(this, 0, "Always draw late BIFS frames", wxPoint(10, 140), wxSize(180, 20));
	m_singletime = new wxCheckBox(this, 0, "Force Single Timeline", wxPoint(10, 180), wxSize(160, 20));

	/*media decoders*/
	m_decaudiotxt = new wxStaticText(this, 0, _T("Prefered Audio Plugin"), wxPoint(100, 40), wxSize(160, 20), wxALIGN_LEFT);
	m_decaudio = new wxComboBox(this, 0, _T(""), wxPoint(40, 60), wxSize(220, 30), 0, NULL, wxCB_READONLY);
	m_decvideotxt = new wxStaticText(this, 0, _T("Prefered Video Plugin"), wxPoint(100, 100), wxSize(160, 20), wxALIGN_LEFT);
	m_decvideo = new wxComboBox(this, 0, _T(""), wxPoint(40, 120), wxSize(220, 30), 0, NULL, wxCB_READONLY);

	/*Rendering*/
	m_fpstxt = new wxStaticText(this, 0, _T("Target Frame Rate"), wxPoint(10, 42), wxSize(160, 20), wxALIGN_LEFT);
	m_fps = new wxComboBox(this, 0, _T(""), wxPoint(180, 40), wxSize(100, 20), 0, NULL, wxCB_READONLY);
	m_aatxt = new wxStaticText(this, 0, _T("Anti-Aliasing"), wxPoint(10, 75), wxSize(160, 20), wxALIGN_LEFT);
	m_aa = new wxComboBox(this, 0, _T(""), wxPoint(180, 75), wxSize(100, 20), 0, NULL, wxCB_READONLY);
	m_fast = new wxCheckBox(this, 0, "Fast Rendering", wxPoint(10, 110), wxSize(160, 20));
	m_force_size = new wxCheckBox(this, 0, "Force Scene Size", wxPoint(180, 110), wxSize(160, 20));
	m_use3D = new wxCheckBox(this, 0, "Use 3D Renderer", wxPoint(10, 140), wxSize(160, 20));
	m_bWas3D = m_use3D->GetValue();
	m_draw_bounds = new wxCheckBox(this, 0, "Draw Bounds", wxPoint(180, 140), wxSize(160, 20));
	m_backtxt = new wxStaticText(this, 0, _T("Default Background Color"), wxPoint(10, 172), wxSize(160, 30), wxALIGN_LEFT);
	m_back = new wxTextCtrl(this, 0, "", wxPoint(180, 170), wxSize(80, 20));
	
	/*Render 2D*/
	m_graphtxt = new wxStaticText(this, 0, _T("Graphics Driver"), wxPoint(10, 42), wxSize(160, 20), wxALIGN_LEFT);
	m_graph = new wxComboBox(this, 0, _T(""), wxPoint(180, 40), wxSize(120, 20), 0, NULL, wxCB_READONLY);
	m_direct = new wxCheckBox(this, 0, "Direct Rendering", wxPoint(10, 90), wxSize(160, 20));
	m_scalable = new wxCheckBox(this, 0, "Scalable Zoom", wxPoint(10, 120), wxSize(160, 20));
	m_noyuv = new wxCheckBox(this, 0, "Disable YUV hardware", wxPoint(10, 150), wxSize(160, 20));
	m_yuvtxt = new wxStaticText(this, 0, _T("(No YUV used)"), wxPoint(180, 152), wxSize(160, 20), wxALIGN_LEFT);

	/*Render 3D*/
	m_raster_outlines = new wxCheckBox(this, 0, "Use OpenGL Raster outlines", wxPoint(10, 35), wxSize(240, 20));
	m_polyaa = new wxCheckBox(this, 0, "Enable polygon anti-aliasing", wxPoint(10, 65), wxSize(240, 20));
	m_wiretxt = new wxStaticText(this, 0, _T("Wireframe mode"), wxPoint(10, 97), wxSize(160, 20), wxALIGN_LEFT);
	m_wire = new wxComboBox(this, 0, _T(""), wxPoint(180, 95), wxSize(120, 20), 0, NULL, wxCB_READONLY);
	m_emulpow2 = new wxCheckBox(this, 0, "Emulate power-of-two textures for video", wxPoint(10, 120), wxSize(260, 20));
	m_norectext = new wxCheckBox(this, 0, "Disable rectangular texture extensions", wxPoint(10, 150), wxSize(260, 20));
	m_copypixels = new wxCheckBox(this, 0, "Bitmap node uses direct pixel copy", wxPoint(10, 180), wxSize(260, 20));

	/*video*/
	m_videotxt = new wxStaticText(this, 0, _T("Video Driver"), wxPoint(10, 62), wxSize(80, 20), wxALIGN_LEFT);
	m_video = new wxComboBox(this, 0, _T(""), wxPoint(90, 60), wxSize(160, 20), 0, NULL, wxCB_READONLY);
	m_switchres = new wxCheckBox(this, 0, "Change video resolution in fullscreen", wxPoint(10, 120), wxSize(240, 20));

	/*audio*/
	m_forcecfg = new wxCheckBox(this, ID_FORCE_AUDIO, "Force Audio Config", wxPoint(10, 35), wxSize(180, 20));
	m_forcecfg->SetValue(1);
	m_nbbuftxt = new wxStaticText(this, 0, _T("Number of Audio Buffers"), wxPoint(20, 65), wxSize(160, 20), wxALIGN_LEFT);
	m_nbbuf = new wxSpinCtrl(this, -1, "", wxPoint(200, 63), wxSize(80, 20), wxSP_WRAP | wxSP_ARROW_KEYS, 1, 30, 15);
	m_nbbuf->SetValue(8);
	m_buflentxt = new wxStaticText(this, 0, _T("Buffers per Second"), wxPoint(20, 90), wxSize(160, 20), wxALIGN_LEFT);
	m_buflen = new wxSpinCtrl(this, -1, "", wxPoint(200, 88), wxSize(80, 20), wxSP_WRAP | wxSP_ARROW_KEYS, 1, 30, 15);
	m_buflen->SetValue(16);
	m_noresync = new wxCheckBox(this, -1, "Disable Audio Resync", wxPoint(10, 120), wxSize(160, 20));
#ifdef WIN32
	m_notifs = new wxCheckBox(this, -1, "Use DirectSound Notifications", wxPoint(10, 150), wxSize(180, 20));
#endif
	m_audiotxt = new wxStaticText(this, 0, _T("Audio Driver"), wxPoint(10, 177), wxSize(80, 20), wxALIGN_LEFT);
	m_audio = new wxComboBox(this, ID_AUDIO_DRIVER, _T(""), wxPoint(100, 175), wxSize(160, 20), 0, NULL, wxCB_READONLY);
	
	/*font*/
	m_fonttxt = new wxStaticText(this, 0, _T("Font Engine"), wxPoint(10, 40), wxSize(100, 25), wxALIGN_LEFT);
	m_font = new wxComboBox(this, 0, _T(""), wxPoint(120, 40), wxSize(160, 20), 0, NULL, wxCB_READONLY);
	m_fontdirtxt = new wxStaticText(this, 0, _T("System Font Directory"), wxPoint(80, 110), wxSize(200, 20), wxALIGN_LEFT);
	m_fontdir = new wxButton(this, ID_FONT_DIR, "...", wxPoint(10, 130), wxSize(295, 20));

	/*download*/
	m_cachedirtxt = new wxStaticText(this, 0, _T("Temporary Downloaded Files"), wxPoint(80, 40), wxSize(220, 25), wxALIGN_LEFT);
	m_cachedir = new wxButton(this, ID_CACHE_DIR, "...", wxPoint(10, 70), wxSize(295, 20));
	m_cleancache = new wxCheckBox(this, -1, "Remove temp files on exit", wxPoint(10, 120), wxSize(160, 20));
	m_restartcache = new wxCheckBox(this, -1, "Always redownload incomplete cached files", wxPoint(10, 150), wxSize(240, 20));

	/*streaming*/
	m_porttxt = new wxStaticText(this, 0, _T("Default RTSP port"), wxPoint(10, 44), wxSize(120, 25), wxALIGN_LEFT);
	m_port = new wxComboBox(this, ID_RTSP_PORT, _T(""), wxPoint(160, 40), wxSize(140, 20), 0, NULL, wxCB_READONLY);
	m_rtsp = new wxCheckBox(this, ID_RTP_OVER_RTSP, "RTP over RTSP", wxPoint(10, 80), wxSize(140, 20));
	m_reorder = new wxCheckBox(this, -1, "use RTP reordering", wxPoint(160, 80), wxSize(130, 20));
	m_timeout = new wxTextCtrl(this, 0, "", wxPoint(10, 120), wxSize(60, 20));
	m_timeouttxt = new wxStaticText(this, 0, _T("milliseconds before control timeout"), wxPoint(80, 120), wxSize(240, 25), wxALIGN_LEFT);
	m_buffer = new wxTextCtrl(this, 0, "", wxPoint(10, 150), wxSize(60, 20));
	m_buffertxt = new wxStaticText(this, 0, _T("milliseconds of Media Buffering"), wxPoint(80, 150), wxSize(240, 25), wxALIGN_LEFT);
	m_dorebuffer = new wxCheckBox(this, ID_RTSP_REBUFFER, "Rebuffer if below", wxPoint(10, 180), wxSize(160, 20));
	m_rebuffer = new wxTextCtrl(this, 0, "", wxPoint(200, 180), wxSize(60, 20));
	m_rebuffer->Disable();

	/*load options*/
	LPINIFILE cfg = m_pApp->m_client.config;
	/*general*/
	sOpt = IF_GetKey(cfg, "General", "Loop");
	m_loop->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "General", "StopAtEnd");
	m_stopatend->SetValue((sOpt && !stricmp(sOpt, "no")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "General", "ConsoleOff");
	m_noconsole->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "General", "ViewXMT");
	m_viewxmt->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);

	/*systems config*/
	sOpt = IF_GetKey(cfg, "Systems", "Language");
	if (!sOpt) sOpt = "eng";
	u32 select = 0;
	for (i = 0; i<NUM_LANGUAGE; i++) {
		m_lang->Append(Languages[2*i]);
		if (sOpt && !stricmp(sOpt, Languages[2*i + 1])) select = i;
	}
	m_lang->SetSelection(select);
	sOpt = IF_GetKey(cfg, "Systems", "ThreadingPolicy");
	select = 0;
	m_thread->Append("Single Thread");
	m_thread->Append("Mutli Thread");
	if (sOpt && !stricmp(sOpt, "Multi")) select = 1;
	m_thread->Append("Free");
	if (sOpt && !stricmp(sOpt, "Free")) select = 2;
	m_thread->SetSelection(select);
	sOpt = IF_GetKey(cfg, "Systems", "ForceSingleClock");
	m_singletime->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Systems", "AlwaysDrawBIFS");
	m_bifsalwaysdrawn->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);


	/*audio dec enum*/
	sOpt = IF_GetKey(cfg, "Systems", "DefAudioDec");
	u32 count = PM_GetPluginsCount(m_pApp->m_client.plugins);
	DecoderInterface *ifc_d;
	select = 0;
	s32 to_sel = 0;
	for (i=0; i<count; i++) {
		if (!PM_LoadInterface(m_pApp->m_client.plugins, i, M4MEDIADECODERINTERFACE, (void **) &ifc_d)) continue;
		if (ifc_d->CanHandleStream(ifc_d, M4ST_AUDIO, 0, NULL, 0, 0)) {
			if (sOpt && !stricmp(ifc_d->plugin_name, sOpt)) select = to_sel;
			m_decaudio->Append(ifc_d->plugin_name);
			to_sel++;
		}
		PM_ShutdownInterface(ifc_d);
	}
	m_decaudio->SetSelection(select);

	/*video dec enum*/
	sOpt = IF_GetKey(cfg, "Systems", "DefVideoDec");
	select = to_sel = 0;
	for (i=0; i<count; i++) {
		if (!PM_LoadInterface(m_pApp->m_client.plugins, i, M4MEDIADECODERINTERFACE, (void **) &ifc_d)) continue;
		if (ifc_d->CanHandleStream(ifc_d, M4ST_VISUAL, 0, NULL, 0, 0)) {
			if (sOpt && !stricmp(ifc_d->plugin_name, sOpt)) select = to_sel;
			m_decvideo->Append(ifc_d->plugin_name);
			to_sel++;
		}
		PM_ShutdownInterface(ifc_d);
	}
	m_decvideo->SetSelection(select);

	/*rendering*/
	sOpt = IF_GetKey(cfg, "Rendering", "RendererName");
	m_bWas3D = (sOpt && strstr(sOpt, "3D")) ? 1 : 0;
	m_use3D->SetValue(m_bWas3D ? 1 : 0);

	sOpt = IF_GetKey(cfg, "Rendering", "ForceSceneSize");
	m_force_size->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Rendering", "BackColor");
	if (sOpt) {
		s32 col = atoi(sOpt);
		wxString sMsg = wxString::Format("0x%.2x%.2x%.2x", (col>>16) & 0xFF, (col>>8) & 0xFF, (col) & 0xFF);
		m_back->SetValue(sMsg);
	} else {
		m_back->SetValue("0x000000");
	}
	sOpt = IF_GetKey(cfg, "Rendering", "FrameRate");
	if (!sOpt) sOpt = "30.0";
	select = 0;
	for (i = 0; i<NUM_RATES; i++) {
		m_fps->Append(BIFSRates[i]);
		if (sOpt && !stricmp(sOpt, BIFSRates[i]) ) select = i;
	}
	m_fps->SetSelection(select);
	
	sOpt = IF_GetKey(cfg, "Rendering", "FastRender");
	m_fast->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);

	sOpt = IF_GetKey(cfg, "Rendering", "AntiAlias");
	m_aa->Append("None");
	m_aa->Append("Text only");
	m_aa->Append("Complete");
	select = 2;
	if (sOpt && !stricmp(sOpt, "Text")) select = 1;
	else if (sOpt && !stricmp(sOpt, "None")) select = 0;
	m_aa->SetSelection(select);

	sOpt = IF_GetKey(cfg, "Rendering", "BoundingVolume");
	if (sOpt && (!stricmp(sOpt, "Box") || !stricmp(sOpt, "Sphere")) ) m_draw_bounds->SetValue(1);
	else m_draw_bounds->SetValue(0);

	/*render2d*/
	sOpt = IF_GetKey(cfg, "Render2D", "DirectRender");
	m_direct->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Render2D", "ScalableZoom");
	m_scalable->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Render2D", "DisableYUV");
	m_noyuv->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);

	/*graphics driver enum*/
	sOpt = IF_GetKey(cfg, "Render2D", "GraphicsDriver");
	void *ifce;
	select = to_sel = 0;
	for (i=0; i<count; i++) {
		if (!PM_LoadInterface(m_pApp->m_client.plugins, i, M4_GRAPHICS_2D_INTERFACE, &ifce)) continue;
		if (sOpt && !stricmp(((BaseInterface *)ifce)->plugin_name, sOpt)) select = to_sel;
		m_graph->Append(((BaseInterface *)ifce)->plugin_name);
		PM_ShutdownInterface(ifce);
		to_sel++;
	}
	m_graph->SetSelection(select);

	/*render3d*/
	sOpt = IF_GetKey(cfg, "Render3D", "RasterOutlines");
	m_raster_outlines->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Render3D", "EmulatePOW2");
	m_emulpow2->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Render3D", "PolygonAA");
	m_polyaa->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Render3D", "Wireframe");
	sOpt = IF_GetKey(cfg, "Render3D", "BitmapCopyPixels");
	m_copypixels->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Render3D", "DisableRectExt");
	m_norectext->SetValue((sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	m_wire->Append("No Wireframe");
	m_wire->Append("Wireframe Only");
	m_wire->Append("Solid and Wireframe");
	if (sOpt && !stricmp(sOpt, "WireOnly")) m_wire->SetSelection(1);
	else if (sOpt && !stricmp(sOpt, "WireOnSolid")) m_wire->SetSelection(2);
	else m_wire->SetSelection(0);

	/*video*/
	sOpt = IF_GetKey(cfg, "Video", "SwitchResolution");
	m_switchres->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Video", "DriverName");
	select = to_sel = 0;
	for (i=0; i<count; i++) {
		if (!PM_LoadInterface(m_pApp->m_client.plugins, i, M4_VIDEO_OUTPUT_INTERFACE, &ifce)) continue;
		if (sOpt && !stricmp(((BaseInterface *)ifce)->plugin_name, sOpt)) select = to_sel;
		m_video->Append(((BaseInterface *)ifce)->plugin_name);
		PM_ShutdownInterface(ifce);
		to_sel++;
	}
	m_video->SetSelection(select);

	/*audio*/
	sOpt = IF_GetKey(cfg, "Audio", "ForceConfig");
	m_forcecfg->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Audio", "NumBuffers");
	m_nbbuf->SetValue( sOpt ? sOpt : "6");
	sOpt = IF_GetKey(cfg, "Audio", "BuffersPerSecond");
	m_buflen->SetValue( sOpt ? sOpt : "15");
	wxCommandEvent event;
	ForceAudio(event);
	sOpt = IF_GetKey(cfg, "Audio", "NoResync");
	m_noresync->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);

	/*driver enum*/
	sOpt = IF_GetKey(cfg, "Audio", "DriverName");
	select = to_sel = 0;
	for (i=0; i<count; i++) {
		if (!PM_LoadInterface(m_pApp->m_client.plugins, i, M4_AUDIO_OUTPUT_INTERFACE, &ifce)) continue;
		if (sOpt && !stricmp(((BaseInterface *)ifce)->plugin_name, sOpt)) select = to_sel;
		m_audio->Append(((BaseInterface *)ifce)->plugin_name);
		PM_ShutdownInterface(ifce);
		to_sel++;
	}
	m_audio->SetSelection(select);
#ifdef WIN32
	sOpt = IF_GetKey(cfg, "Audio", "UseNotification");
	m_notifs->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	wxCommandEvent audevt;
	OnSetAudioDriver(audevt);
#endif

	/*font*/
	sOpt = IF_GetKey(cfg, "FontEngine", "DriverName");
	to_sel = select = 0;
	for (i=0; i<count; i++) {
		if (!PM_LoadInterface(m_pApp->m_client.plugins, i, M4_FONT_RASTER_INTERFACE, &ifce)) continue;
		if (sOpt && !stricmp(((BaseInterface *)ifce)->plugin_name, sOpt)) select = to_sel;
		m_font->Append(((BaseInterface *)ifce)->plugin_name);
		PM_ShutdownInterface(ifce);
		to_sel++;
	}
	m_font->SetSelection(select);
	sOpt = IF_GetKey(cfg, "FontEngine", "FontDirectory");
	if (sOpt) m_fontdir->SetLabel(sOpt);

	/*downloader*/
	sOpt = IF_GetKey(cfg, "Downloader", "CleanCache");
	m_cleancache->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Downloader", "RestartFiles");
	m_restartcache->SetValue( (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0);
	sOpt = IF_GetKey(cfg, "Downloader", "CacheDirectory");
	if (sOpt) m_cachedir->SetLabel(sOpt);
	
	/*streaming*/	
	m_port->Append("554 (RTSP standard)");
	m_port->Append("7070 (RTSP ext)");
	m_port->Append("80 (RTSP / HTTP tunnel)");
	m_port->Append("8080 (RTSP / HTTP tunnel)");
	sOpt = IF_GetKey(cfg, "Streaming", "DefaultPort");
	u32 port = 554;
	Bool force_rtsp = 0;
	if (sOpt) port = atoi(sOpt);
	switch (port) {
	case 8080:
		m_port->SetSelection(3);
		force_rtsp = 1;
		break;
	case 80:
		m_port->SetSelection(2);
		force_rtsp = 1;
		break;
	case 7070:
		m_port->SetSelection(1);
		break;
	default:
		m_port->SetSelection(0);
		break;
	}

	Bool use_rtsp = 0;
	sOpt = IF_GetKey(cfg, "Streaming", "RTPoverRTSP");
	if (sOpt && !stricmp(sOpt, "yes")) use_rtsp = 1;

	if (force_rtsp) {
		m_rtsp->SetValue(1);
		m_rtsp->Enable(0);
		m_reorder->SetValue(0);
		m_reorder->Enable(0);
	} else {
		m_rtsp->SetValue(use_rtsp ? 1 : 0);
		m_rtsp->Enable(1);
		m_reorder->Enable(1);
		sOpt = IF_GetKey(cfg, "Streaming", "ReorderSize");
		m_reorder->SetValue( (sOpt && !stricmp(sOpt, "0")) ? 1 : 0);
	}
	sOpt = IF_GetKey(cfg, "Streaming", "RTSPTimeout");
	m_timeout->SetValue(sOpt ? sOpt : "30000");
	sOpt = IF_GetKey(cfg, "Network", "BufferLength");
	m_buffer->SetValue(sOpt ? sOpt : "3000");
	sOpt = IF_GetKey(cfg, "Network", "RebufferLength");
	u32 buf_len = 0;
	if (sOpt) buf_len = atoi(sOpt);
	if (buf_len) {
		m_dorebuffer->SetValue(1);
		m_rebuffer->SetValue(sOpt);
		m_rebuffer->Enable(1);
	} else {
		m_dorebuffer->SetValue(0);
		m_rebuffer->SetValue("0");
		m_rebuffer->Enable(0);
	}

	RTPoverRTSP(event);


	m_select->Append("General");
	m_select->Append("MPEG-4 Systems");
	m_select->Append("Media Decoders");
	m_select->Append("Rendering");
	m_select->Append("Renderer 2D");
	m_select->Append("Renderer 3D");
	m_select->Append("Video Output");
	m_select->Append("Audio Output");
	m_select->Append("Text Engine");
	m_select->Append("File Download");
	m_select->Append("Real-Time Streaming");
	m_select->SetSelection(0);
	m_sel = 0;
	DoSelect();
}

BEGIN_EVENT_TABLE(wxGPACControl, wxDialog)
	EVT_BUTTON(ID_APPLY, wxGPACControl::Apply)
	EVT_COMBOBOX(ID_SELECT, wxGPACControl::OnSetSelection)
	EVT_CHECKBOX(ID_FORCE_AUDIO, wxGPACControl::ForceAudio)
	EVT_COMBOBOX(ID_AUDIO_DRIVER, wxGPACControl::OnSetAudioDriver)
	EVT_BUTTON(ID_FONT_DIR, wxGPACControl::FontDir)
	EVT_BUTTON(ID_CACHE_DIR, wxGPACControl::CacheDir)
	EVT_CHECKBOX(ID_RTP_OVER_RTSP, wxGPACControl::RTPoverRTSP)
	EVT_CHECKBOX(ID_RTSP_REBUFFER, wxGPACControl::Rebuffer)
	EVT_COMBOBOX(ID_RTSP_PORT, wxGPACControl::OnSetRTSPPort)
#ifdef WIN32
	EVT_BUTTON(ID_MAKE_DEF, wxGPACControl::OnFileassoc)
#endif
END_EVENT_TABLE()



void wxGPACControl::OnSetSelection(wxCommandEvent &WXUNUSED(event))
{
	m_sel = m_select->GetSelection();
	DoSelect();
}

void wxGPACControl::FontDir(wxCommandEvent &WXUNUSED(event))
{
	wxDirDialog dlg(this);
	dlg.SetPath(m_fontdir->GetLabel());
	if (dlg.ShowModal() == wxID_OK) {
		m_fontdir->SetLabel(dlg.GetPath());
	}
}
void wxGPACControl::CacheDir(wxCommandEvent &WXUNUSED(event))
{
	wxDirDialog dlg(this);
	dlg.SetPath(m_cachedir->GetLabel());
	if (dlg.ShowModal() == wxID_OK) {
		m_cachedir->SetLabel(dlg.GetPath());
	}
}

void wxGPACControl::RTPoverRTSP(wxCommandEvent &WXUNUSED(event))
{
	m_reorder->Enable(m_rtsp->GetValue() ? 0 : 1);
}

void wxGPACControl::Rebuffer(wxCommandEvent &WXUNUSED(event))
{
	if (m_dorebuffer->GetValue()) {
		m_rebuffer->Enable();
	} else {
		m_rebuffer->Disable();
	}
}

void wxGPACControl::OnSetRTSPPort(wxCommandEvent &WXUNUSED(event))
{
	if (m_port->GetSelection() > 1) {
		m_rtsp->Enable(0);
		m_reorder->Enable(0);
	} else {
		m_rtsp->Enable(1);
		m_reorder->Enable(1);
	}
}

void wxGPACControl::ForceAudio(wxCommandEvent &WXUNUSED(event))
{
	if (m_forcecfg->GetValue()) {
		m_nbbuf->Enable();
		m_buflen->Enable();
	} else {
		m_nbbuf->Disable();
		m_buflen->Disable();
	}
}

void wxGPACControl::OnSetAudioDriver(wxCommandEvent &WXUNUSED(event))
{
#ifdef WIN32
	if (strstr(m_audio->GetStringSelection().c_str(), "DirectSound")) {
		m_notifs->Enable(1);
	} else {
		m_notifs->Enable(0);
	}
#endif
}

void wxGPACControl::DoSelect()
{
	/*hide everything*/
	/*general*/
	m_loop->Hide();
	m_stopatend->Hide();
	m_noconsole->Hide();
	m_viewxmt->Hide();
#ifdef WIN32
	m_makedefault->Hide();
#endif
	/*systems*/
	m_langtxt->Hide();
	m_lang->Hide();
	m_threadtxt->Hide();
	m_thread->Hide();
	m_bifsalwaysdrawn->Hide();
	m_singletime->Hide();
	/*decoders*/
	m_decaudiotxt->Hide();
	m_decaudio->Hide();
	m_decvideotxt->Hide();
	m_decvideo->Hide();
	/*rendering*/
	m_aatxt->Hide();
	m_aa->Hide();
	m_fpstxt->Hide();
	m_backtxt->Hide();
	m_fps->Hide();
	m_use3D->Hide();
	m_draw_bounds->Hide();
	m_fast->Hide();
	m_force_size->Hide();
	m_back->Hide();
	/*render2d*/
	m_graphtxt->Hide();
	m_graph->Hide();
	m_direct->Hide();
	m_scalable->Hide();
	m_noyuv->Hide();
	m_yuvtxt->Hide();
	/*render3d*/
	m_raster_outlines->Hide();
	m_polyaa->Hide();
	m_emulpow2->Hide();
	m_norectext->Hide();
	m_copypixels->Hide();
	m_wiretxt->Hide();
	m_wire->Hide();
	/*video*/
	m_videotxt->Hide();
	m_video->Hide();
	m_switchres->Hide();
	/*audio*/
	m_forcecfg->Hide();
	m_nbbuftxt->Hide();
	m_nbbuf->Hide();
	m_buflentxt->Hide();
	m_buflen->Hide();
	m_noresync->Hide();
	m_audiotxt->Hide();
	m_audio->Hide();
#ifdef WIN32
	m_notifs->Hide();
#endif
	/*font*/
	m_fonttxt->Hide();
	m_fontdirtxt->Hide();
	m_font->Hide();
	m_fontdir->Hide();
	/*downloader*/
	m_cachedirtxt->Hide();
	m_cachedir->Hide();
	m_cleancache->Hide();
	m_restartcache->Hide();
	/*RTP streaming*/
	m_porttxt->Hide();
	m_port->Hide();
	m_rtsp->Hide();
	m_reorder->Hide();
	m_timeouttxt->Hide();
	m_timeout->Hide();
	m_buffertxt->Hide();
	m_buffer->Hide();
	m_dorebuffer->Hide();
	m_rebuffer->Hide();

	switch (m_sel) {
	case 0:
		m_loop->Show();
		m_stopatend->Show();
		m_noconsole->Show();
		m_viewxmt->Show();
#ifdef WIN32
		m_makedefault->Show();
#endif
		break;
	case 1:
		m_langtxt->Show();
		m_lang->Show();
		m_threadtxt->Show();
		m_thread->Show();
		m_bifsalwaysdrawn->Show();
		m_singletime->Show();
		break;
	case 2:
		m_decaudiotxt->Show();
		m_decaudio->Show();
		m_decvideotxt->Show();
		m_decvideo->Show();
		break;
	case 3:
		m_fpstxt->Show();
		m_backtxt->Show();
		m_fps->Show();
		m_aatxt->Show();
		m_aa->Show();
		m_use3D->Show();
		m_draw_bounds->Show();
		m_fast->Show();
		m_force_size->Show();
		m_back->Show();
		break;
	case 4:
		m_graphtxt->Show();
		m_graph->Show();
		m_direct->Show();
		m_scalable->Show();
		m_noyuv->Show();
		m_yuvtxt->Show();
		break;
	case 5:
		m_raster_outlines->Show();
		m_polyaa->Show();
		m_emulpow2->Show();
		m_norectext->Show();
		m_copypixels->Show();
		m_wiretxt->Show();
		m_wire->Show();
		break;
	case 6:
		m_videotxt->Show();
		m_video->Show();
		m_switchres->Show();
		break;
	case 7:
		m_forcecfg->Show();
		m_nbbuftxt->Show();
		m_nbbuf->Show();
		m_buflentxt->Show();
		m_buflen->Show();
		m_noresync->Show();
		m_audiotxt->Show();
		m_audio->Show();
#ifdef WIN32
		m_notifs->Show();
#endif
		break;
	case 8:
		m_fonttxt->Show();
		m_fontdirtxt->Show();
		m_font->Show();
		m_fontdir->Show();
		break;
	case 9:
		m_cachedirtxt->Show();
		m_cachedir->Show();
		m_cleancache->Show();
		m_restartcache->Show();
		break;
	case 10:
		m_porttxt->Show();
		m_port->Show();
		m_rtsp->Show();
		m_reorder->Show();
		m_timeouttxt->Show();
		m_timeout->Show();
		m_buffertxt->Show();
		m_buffer->Show();
		m_dorebuffer->Show();
		m_rebuffer->Show();
		break;
	}
}


void wxGPACControl::Apply(wxCommandEvent &WXUNUSED(event))
{
	Bool need_reload = 0;


	/*save options*/
	LPINIFILE cfg = m_pApp->m_client.config;

	m_pApp->m_loop = m_loop->GetValue() ? 1 : 0;
	IF_SetKey(cfg, "General", "Loop", m_loop->GetValue() ? "yes" : "no");
	m_pApp->m_stop_at_end = m_stopatend->GetValue() ? 1 : 0;
	IF_SetKey(cfg, "General", "StopAtEnd",  m_stopatend->GetValue() ? "yes" : "no");
	m_pApp->m_console_off = m_noconsole->GetValue() ? 1 : 0;
	IF_SetKey(cfg, "General", "ConsoleOff", m_noconsole->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "General", "ViewXMT", m_viewxmt->GetValue() ? "yes" : "no");

	s32 sel = m_lang->GetSelection();
	IF_SetKey(cfg, "Systems", "Language", Languages[2*sel + 1]);
	sel = m_thread->GetSelection();
	IF_SetKey(cfg, "Systems", "ThreadingPolicy", (sel==0) ? "Single" : ( (sel==1) ? "Multi" : "Free"));
	IF_SetKey(cfg, "Systems", "ForceSingleClock", m_singletime->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Systems", "AlwaysDrawBIFS", m_bifsalwaysdrawn->GetValue() ? "yes" : "no");

	IF_SetKey(cfg, "Systems", "DefAudioDec", m_decaudio->GetStringSelection().c_str());
	IF_SetKey(cfg, "Systems", "DefVideoDec", m_decvideo->GetStringSelection().c_str());
	

	IF_SetKey(cfg, "Rendering", "FastRender", m_fast->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Rendering", "ForceSceneSize", m_force_size->GetValue() ? "yes" : "no");

	u32 col;
	wxString str = m_back->GetLabel();
	sscanf(str.c_str(), "0x%x", &col);
	str = wxString::Format("%d", col);
	IF_SetKey(cfg, "Rendering", "BackColor", str.c_str());
	IF_SetKey(cfg, "Rendering", "FrameRate", BIFSRates[m_fps->GetSelection()]);
	sel = m_aa->GetSelection();
	IF_SetKey(cfg, "Rendering", "AntiAlias", (sel==0) ? "None" : ( (sel==1) ? "Text" : "All"));
	IF_SetKey(cfg, "Rendering", "BoundingVolume", m_draw_bounds->GetValue() ? "Box" : "None");

	Bool is_3D = m_use3D->GetValue() ? 1 : 0;
	if (m_bWas3D != is_3D) {
		IF_SetKey(cfg, "Rendering", "RendererName", is_3D ? "GPAC 3D Renderer" : "GPAC 2D Renderer");
		need_reload = 1;
	}

	IF_SetKey(cfg, "Render2D", "DirectRender", m_direct->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Render2D", "ScalableZoom", m_scalable->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Render2D", "GraphicsDriver", m_graph->GetStringSelection().c_str());
	IF_SetKey(cfg, "Render2D", "DisableYUV", m_noyuv->GetValue() ? "yes" : "no");

	IF_SetKey(cfg, "Render3D", "RasterOutlines", m_raster_outlines->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Render3D", "EmulatePOW2", m_emulpow2->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Render3D", "PolygonAA", m_polyaa->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Render3D", "DisableRectExt", m_norectext->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Render3D", "BitmapCopyPixels", m_copypixels->GetValue() ? "yes" : "no");

	sel = m_wire->GetSelection();
	IF_SetKey(cfg, "Render3D", "Wireframe", (sel==2) ? "WireOnSolid" : ( (sel==1) ? "WireOnly" : "WireNone" ) );

	IF_SetKey(cfg, "Video", "SwitchResolution", m_switchres->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Video", "DriverName", m_video->GetStringSelection().c_str());


	IF_SetKey(cfg, "Audio", "ForceConfig", m_forcecfg->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Audio", "NoResync", m_noresync->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Audio", "NumBuffers", wxString::Format("%d", m_nbbuf->GetValue()) );
	IF_SetKey(cfg, "Audio", "BuffersPerSecond", wxString::Format("%d", m_buflen->GetValue()));
	IF_SetKey(cfg, "Audio", "DriverName", m_audio->GetStringSelection().c_str());
#ifdef WIN32
	if (m_notifs->IsEnabled()) 
		IF_SetKey(cfg, "Audio", "UseNotification", m_notifs->GetValue() ? "yes" : "no");
#endif
	
	IF_SetKey(cfg, "FontEngine", "DriverName", m_font->GetStringSelection().c_str());
	IF_SetKey(cfg, "FontEngine", "FontDirectory", m_fontdir->GetLabel().c_str());

	IF_SetKey(cfg, "Downloader", "CleanCache", m_cleancache->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Downloader", "RestartFiles", m_restartcache->GetValue() ? "yes" : "no");
	IF_SetKey(cfg, "Downloader", "CacheDirectory", m_cachedir->GetLabel().c_str());


	Bool force_rtsp = 0;
	switch (m_port->GetSelection()) {
	case 3:
		IF_SetKey(cfg, "Streaming", "DefaultPort", "8080");
		force_rtsp = 1;
		break;
	case 2:
		IF_SetKey(cfg, "Streaming", "DefaultPort", "80");
		force_rtsp = 1;
		break;
	case 1:
		IF_SetKey(cfg, "Streaming", "DefaultPort", "7070");
		break;
	default:
		IF_SetKey(cfg, "Streaming", "DefaultPort", "554");
		break;
	}

	if (force_rtsp) {
		IF_SetKey(cfg, "Streaming", "RTPoverRTSP", "yes");
	} else {
		IF_SetKey(cfg, "Streaming", "RTPoverRTSP", m_rtsp->GetValue() ? "yes" : "no");
		if (!m_rtsp->GetValue()) IF_SetKey(cfg, "Streaming", "ReorderSize", m_dorebuffer->GetValue() ? "30" : "0");
	}

	IF_SetKey(cfg, "Streaming", "RTSPTimeout", m_timeout->GetValue().c_str());
	IF_SetKey(cfg, "Network", "BufferLength", m_buffer->GetValue().c_str());
	if (m_dorebuffer->GetValue()) {
		IF_SetKey(cfg, "Network", "RebufferLength", m_rebuffer->GetValue().c_str());
	} else {
		IF_SetKey(cfg, "Network", "RebufferLength", "0");
	}


	if (!need_reload) {
		M4T_SetOption(m_pApp->m_term, M4O_ReloadConfig, 1);
		M4T_RefreshWindow(m_pApp->m_term);
	} else {
		m_pApp->ReloadTerminal();
	}
}

#ifdef WIN32
void wxGPACControl::OnFileassoc(wxCommandEvent &WXUNUSED(event)) 
{
	HKEY hSection;
	unsigned char szDir[MAX_PATH];
	unsigned char cmd[MAX_PATH];
	DWORD ioSize = MAX_PATH;
	DWORD dwDisp;

	RegOpenKeyEx(HKEY_LOCAL_MACHINE, "SOFTWARE\\GPAC", 0, KEY_READ, &hSection);
	strcpy((char *) szDir, m_pApp->m_pExePath.c_str());
	if (!hSection) 
		RegCreateKeyEx(HKEY_LOCAL_MACHINE, "SOFTWARE\\GPAC", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hSection, &dwDisp);

	/*overwrite install dir with current path*/
	RegSetValueEx(hSection, "Install_Dir", 0, REG_SZ, szDir, strlen((const char *) szDir)+1);
	RegCloseKey(hSection);

	/*overwrite .mp4 file associations */
	RegCreateKeyEx(HKEY_CLASSES_ROOT, "mp4file\\DefaultIcon", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hSection, &dwDisp);
	strcpy((char *) cmd, (const char *)szDir);
	if (cmd[strlen((const char *) cmd) - 1] != '\\') strcat((char *) cmd, "\\");
	strcat((char *) cmd, "gpac.ico");
	RegSetValueEx(hSection, "", 0, REG_SZ, cmd, strlen((const char *) cmd)+1);
	RegCloseKey(hSection);

	RegCreateKeyEx(HKEY_CLASSES_ROOT, "mp4file\\Shell\\open\\command", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hSection, &dwDisp);
	strcpy((char *) cmd, (const char *)szDir);
	if (cmd[strlen((const char *) cmd) - 1] != '\\') strcat((char *) cmd, "\\");
	strcat((char *) cmd, "gpac.exe \"%L\"");
	RegSetValueEx(hSection, "", 0, REG_SZ, cmd, strlen((const char *) cmd)+1);
	RegCloseKey(hSection);

	RegCreateKeyEx(HKEY_CLASSES_ROOT, ".mp4", 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hSection, &dwDisp);
	RegSetValueEx(hSection, "", 0, REG_SZ, (const unsigned char *) "mp4file", strlen("mp4file")+1);
	RegCloseKey(hSection);
}
#endif
