/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Stream Management sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "SysCodecs.h"

Double IS_GetSceneTime(void *is);
void IS_LoadExternProto(void *is, MFURL *url);


M4Err BIFS_SetProperties(DecoderInterface *plug, InlineScene *scene, M4Client *app, Bool is_anim)
{
	BIFSPriv *priv = plug->privateStack;
	if (priv->codec) return M4BadParam;
	priv->pScene = scene;
	priv->app = app;
	
	priv->codec = BIFS_NewDecoder(scene->graph);
	/*attach the clock*/
	BIFS_SetClock(priv->codec, IS_GetSceneTime, scene);
	/*attach the proto loader*/
	BIFS_SetProtoLoader(priv->codec, IS_LoadExternProto);
	/*ignore all size info on anim streams*/
	if (is_anim) BIFS_IgnoreSizeInfo(priv->codec);
	return M4OK;
}

static M4Err BIFS_AttachStream(DecoderInterface *plug, 
									 u16 ES_ID, 
									 unsigned char *decSpecInfo, 
									 u32 decSpecInfoSize, 
									 u16 DependsOnES_ID,
									 u32 objectTypeIndication, 
									 Bool Upstream)
{
	BIFSPriv *priv = plug->privateStack;
	if (Upstream) return M4NotSupported;
	return BIFS_ConfigureStream(priv->codec, ES_ID, decSpecInfo, decSpecInfoSize, objectTypeIndication);
}

static M4Err BIFS_DetachStream(DecoderInterface *plug, u16 ES_ID)
{
	BIFSPriv *priv = plug->privateStack;
	return BIFS_RemoveStream(priv->codec, ES_ID);
}

static M4Err BIFS_GetCapabilities(DecoderInterface *plug,
										CapObject *capability)
{
	switch (capability->CapCode) {
	default:
		capability->cap.valueINT = 0;
		break;
	}
	return M4OK;
}


static M4Err BIFS_SetCapabilities(DecoderInterface *plug, 
										const CapObject capability)
{
	return M4OK;
}

static M4Err BIFS_Process(DecoderInterface *plug, 
								unsigned char *inBuffer, 
								u32 inBufferLength, 
								u16 ES_ID,
								unsigned char *outBuffer, 
								u32 *outBufferLength, 
								Bool isRAP,
								u8 PaddingBits,
								u32 mmlevel)
{
	M4Err e = M4OK;
	BIFSPriv *priv = plug->privateStack;

	e = BIFS_DecodeAU(priv->codec, ES_ID, inBuffer, inBufferLength);

	/*if scene not attached do it*/
	if (!priv->pScene->graph_attached) {
		IS_AttachGraphToRender(priv->pScene);
	}
	return e;
}

void BIFS_DeleteDec(DecoderInterface *plug)
{
	BIFSPriv *priv = plug->privateStack;
	BIFS_DeleteDecoder(priv->codec);
	free(priv);
	free(plug);
}

DecoderInterface *NewBIFSCodec(u32 PL)
{
	BIFSPriv *priv;
	DecoderInterface *tmp;
	
	tmp = (DecoderInterface *) malloc(sizeof(DecoderInterface));
	if (!tmp) return NULL;
	memset(tmp, 0, sizeof(DecoderInterface));

	priv = (BIFSPriv *) malloc(sizeof(BIFSPriv));
	priv->codec = NULL;
	priv->PL = PL;
	tmp->privateStack = priv;
	tmp->Codec_AttachStream = BIFS_AttachStream;
	tmp->Codec_DetachStream = BIFS_DetachStream;
	tmp->Codec_GetCapabilities = BIFS_GetCapabilities;
	tmp->Codec_SetCapabilities = BIFS_SetCapabilities;
	tmp->Codec_Process = BIFS_Process;
	M4_REG_PLUG(tmp, M4MEDIADECODERINTERFACE, "GPAC BIFS Decoder", "gpac distribution", 0)
	return tmp;
}



M4Err ODS_SetProperties(DecoderInterface *plug, InlineScene *scene, M4Client *app)
{
	ODManager *odm;
	M4ODPriv *priv = plug->privateStack;
	if (priv->scene || priv->app) return M4BadParam;
	priv->scene = scene;
	priv->app = app;

	/*locate service owner*/
	odm = scene->root_od;
	while (odm->remote_OD) odm = odm->remote_OD;
	priv->parentService = odm->net_service;
	assert (priv->parentService);
	return M4OK;
}


static M4Err ODS_ODUpdate(M4ODPriv *priv, ObjectDescriptorUpdate *odU)
{
	ObjectDescriptor *od;
	ODManager *odm;
	u32 count;

	/*extract all our ODs and compare with what we already have...*/
	count = ChainGetCount(odU->objectDescriptors);
	if (count > 255) return M4InvalidDescriptor;

	while (count) {
		od = ChainGetEntry(odU->objectDescriptors, 0);
		odm = IS_FindODM(priv->scene, od->objectDescriptorID);
		/*remove the old OD*/
		if (odm) ODM_RemoveOD(odm);

		odm = NewODManager();
		odm->OD = od;
		odm->term = priv->app;
		odm->parentscene = priv->scene;
		ChainAddEntry(priv->scene->ODlist, odm);

		/*by default, all the OD are setup with the parent service, eg the service delivering this OD stream*/
		ODM_SetupOD(odm, priv->parentService);

		ChainDeleteEntry(odU->objectDescriptors, 0);
		count--;
	}
	return M4OK;
}



static M4Err ODS_RemoveOD(M4ODPriv *priv, ObjectDescriptorRemove *odR)
{
	u32 i;
	for (i=0; i< odR->NbODs; i++) {
		ODManager *odm = IS_FindODM(priv->scene, odR->OD_ID[i]);
		if (odm) ODM_RemoveOD(odm);
	}
	return M4OK;
}

static M4Err ODS_UpdateESD(M4ODPriv *priv, ESDescriptorUpdate *ESDs)
{
	ESDescriptor *esd, *prev;
	ODManager *odm;
	u32 count, i;

	odm = IS_FindODM(priv->scene, ESDs->ODID);
	/*spec: "ignore"*/
	if (!odm) return M4OK;
	/*spec: "ES_DescriptorUpdate shall not be applied on object descriptors that have set URL_Flag to '1' (see 8.6.3)."*/
	if (odm->remote_OD) return M4NonCompliantBitStream;

	count = ChainGetCount(ESDs->ESDescriptors);

	while (count) {
		esd = ChainGetEntry(ESDs->ESDescriptors, 0);
		/*spec: "ES_Descriptors with ES_IDs that have already been received within the same name scope shall be ignored."*/
		prev = NULL;
		for (i=0; i<ChainGetCount(odm->OD->ESDescriptors); i++) {
			prev = ChainGetEntry(odm->OD->ESDescriptors, i);
			if (prev->ESID == esd->ESID) break;
			prev = NULL;
		}

		if (prev) {
			OD_DeleteDescriptor((Descriptor **)&esd);
		} else {
			/*and register new stream*/
			ChainAddEntry(odm->OD->ESDescriptors, esd);
			ODM_SetupStream(odm, esd, odm->net_service);
		}

		/*remove the desc from the AU*/
		ChainDeleteEntry(ESDs->ESDescriptors, 0);
		count--;
	}
	return M4OK;
}


static M4Err ODS_RemoveESD(M4ODPriv *priv, ESDescriptorRemove *ESDs)
{
	ODManager *odm;
	u32 i;
	odm = IS_FindODM(priv->scene, ESDs->ODID);
	/*spec: "ignore"*/
	if (!odm) return M4OK;
	/*spec: "ES_DescriptorRemove shall not be applied on object descriptors that have set URL_Flag to '1' (see 8.6.3)."*/
	if (odm->remote_OD) return M4NonCompliantBitStream;

	for (i=0; i<ESDs->NbESDs; i++) {
		/*blindly call remove*/
		ODM_RemoveStream(odm, ESDs->ES_ID[i]);
	}
	return M4OK;
}

static M4Err ODS_Codec_AttachStream(DecoderInterface *plug, 
									 u16 ES_ID, 
									 unsigned char *decSpecInfo, 
									 u32 decSpecInfoSize, 
									 u16 DependsOnES_ID,
									 u32 objectTypeIndication, 
									 Bool Upstream)
{
	M4ODPriv *priv = plug->privateStack;
	
	if (Upstream) return M4OK;
	//warning, we only support one stream ...
	if (priv->streamCount) return M4NotSupported;
	priv->ESIDs = malloc(sizeof(u16));
	priv->ESIDs[0] = ES_ID;
	priv->streamCount ++;
	//that's it, OD doesn't use decoderSpecificInfo
	return M4OK;
}


static M4Err ODS_Codec_DetachStream(DecoderInterface *plug, u16 ES_ID)
{
	M4ODPriv *priv = plug->privateStack;

	//warning, we only support one stream ...
	if (!priv->streamCount) return M4BadParam;
	free(priv->ESIDs);
	priv->ESIDs = NULL;
	priv->streamCount = 0;

	return M4OK;
}

static M4Err ODS_Codec_GetCapabilities(DecoderInterface *plug,
										CapObject *capability)
{
	//OD codec has no capability
	capability->cap.valueINT = 0;
	return M4OK;
}


static M4Err ODS_Codec_SetCapabilities(DecoderInterface *plug, 
										const CapObject capability)
{
	//OD codec has no capability
	return M4NotSupported;
}

static M4Err ODS_Codec_Process(DecoderInterface *plug, 
								unsigned char *inBuffer, 
								u32 inBufferLength, 
								u16 ES_ID,
								unsigned char *outBuffer, 
								u32 *outBufferLength, 
								Bool isRAP,
								u8 PaddingBits,
								u32 mmlevel)
{
	M4Err e;
	ODCommand *com;
	M4ODPriv *priv = plug->privateStack;
	
	//OD codec directly manipulates the terminal resources. No need for
	//size request

	//1- set up the decoder
	e = OD_SetBuffer(priv->codec, inBuffer, inBufferLength);
	if (e) return e;
	//2- decode the AU
	e = OD_DecodeAU(priv->codec);
	if (e) return e;

	//3- process all the commands in this AU, in order
	while (1) {
		com = OD_GetCommand(priv->codec);
		if (!com) break;

		//ok, we have a command
		switch (com->tag) {
		case ODUpdate_Tag:
			e = ODS_ODUpdate(priv, (ObjectDescriptorUpdate *) com);
			break;
		case ODRemove_Tag:
			e = ODS_RemoveOD(priv, (ObjectDescriptorRemove *) com);
			break;
		case ESDUpdate_Tag:
			e = ODS_UpdateESD(priv, (ESDescriptorUpdate *)com);
			break;
		case ESDRemove_Tag:
			e = ODS_RemoveESD(priv, (ESDescriptorRemove *)com);
			break;
		case IPMPDUpdate_Tag:
			e = M4NotSupported;
			break;
		case IPMPDRemove_Tag:
			e = M4NotSupported;
			break;
		/*should NEVER exist outside the file format*/
		case ESDRemove_Ref_Tag:
			e = M4NonCompliantBitStream;
			break;
		default:
			if (com->tag >= ISO_RESERVED_COMMANDS_BEGIN && com->tag <= ISO_RESERVED_COMMANDS_END) {
				e = M4ISOForbiddenDescriptor;
			} else {
				/*we don't process user commands*/
				e = M4OK;
			}
			break;
		}
		OD_DeleteCommand(&com);
		if (e) return e;
	}
	return M4OK;
}

void ODS_Codec_Delete(DecoderInterface *plug)
{
	M4ODPriv *priv = plug->privateStack;

	if (priv->ESIDs) free(priv->ESIDs);
	OD_DeleteCodec(priv->codec);
	free(priv);
	free(plug);
}

DecoderInterface *NewODSysCodec(u32 PL)
{
	DecoderInterface *tmp;
	M4ODPriv *priv;
	
	tmp = (DecoderInterface *) malloc(sizeof(DecoderInterface));
	if (!tmp) return NULL;
	memset(tmp, 0, sizeof(DecoderInterface));

	priv = (M4ODPriv *) malloc(sizeof(M4ODPriv));
	priv->ESIDs = NULL;
	priv->streamCount = 0;
	//we only work with decoding ...
	priv->codec = OD_NewCodec(OD_READ);
	priv->scene = NULL;
	priv->app = NULL;
	priv->PL = PL;

	tmp->privateStack = priv;
	tmp->Codec_AttachStream = ODS_Codec_AttachStream;
	tmp->Codec_DetachStream = ODS_Codec_DetachStream;
	tmp->Codec_GetCapabilities = ODS_Codec_GetCapabilities;
	tmp->Codec_SetCapabilities = ODS_Codec_SetCapabilities;
	tmp->Codec_Process = ODS_Codec_Process;

	M4_REG_PLUG(tmp, M4MEDIADECODERINTERFACE, "GPAC OD Decoder", "gpac distribution", 0)
	return tmp;
}
