/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Stream Management sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include <gpac/intern/m4_esm_dev.h>
#include "MediaMemory.h"
#include "MediaControl.h"
#include "SysCodecs.h"
#include "InputSensor.h"



/*update config of object*/
void MO_UpdateCaps(MediaObject *mo);


#define TIME_CHECK		3


M4Err Codec_Load(GenericCodec *codec, ESDescriptor *esd, u32 PL);

GenericCodec *NewDecoder(ODManager *odm, ESDescriptor *base_layer, s32 PL, M4Err *e)
{
	GenericCodec *tmp = malloc(sizeof(GenericCodec));
	if (! tmp) {
		*e = M4OutOfMem;
		return NULL;
	}
	memset(tmp, 0, sizeof(GenericCodec));
	tmp->odm = odm;

	if (PL<0) PL = 0xFF;
	*e = Codec_Load(tmp, base_layer, PL);
	
	if (*e) {
		free(tmp);
		return NULL;
	}
	/*remember codec type*/
	tmp->type = base_layer->decoderConfig->streamType;
	tmp->inChannels = NewChain();	
	tmp->Status = CODEC_STOP;
	
	return tmp;
}


M4Err Codec_AddChannel(GenericCodec *codec, Channel *ch)
{
	M4Err e;
	NetworkCommand com;
	Channel *a_ch;
	unsigned char *dsi;
	u32 dsiSize, CUsize, i;
	CapObject cap;
	u32 min, max;


	/*only for valid codecs (eg not OCR)*/
	if (codec->decio) {
		com.dsi = NULL;
		dsi = NULL;
		dsiSize = 0;
		if (ch->esd->decoderConfig->upstream) codec->hasUpstream = 1;
		if (ch->esd->decoderConfig->decoderSpecificInfo) {
			dsi = ch->esd->decoderConfig->decoderSpecificInfo->data;
			dsiSize = ch->esd->decoderConfig->decoderSpecificInfo->dataLength;
		} 
		if (!dsi) {
			com.command_type = CHAN_GET_DSI;
			com.on_channel = ch;
			e = NM_ServiceCommand(ch->service, &com);
			if (!e && com.dsi) {
				dsi = com.dsi;
				dsiSize = com.dsi_len;
			}
		}

		e = codec->decio->Codec_AttachStream(codec->decio, ch->esd->ESID, 
					dsi, dsiSize, ch->esd->dependsOnESID, ch->esd->decoderConfig->objectTypeIndication, ch->esd->decoderConfig->upstream);

		if (com.dsi) free(com.dsi);

		/*some broken ISMA files don't signal the DSI in the IOD, only in sdp...*/
		if (e) return e;

		/*ask codec for desired output capacity - note this may be 0 if stream is not yet configured*/
		cap.CapCode = CAP_OUTPUTSIZE;
		Codec_GetCap(codec, &cap);
		if (codec->CB && (cap.cap.valueINT != codec->CB->UnitSize)) {
			CB_Delete(codec->CB);
			codec->CB = NULL;
		}
		CUsize = cap.cap.valueINT;

		/*get desired amount of units and minimal fullness (used for scheduling)*/
		switch(codec->type) {
		case M4ST_VISUAL:
		case M4ST_AUDIO:
			cap.CapCode = CAP_BUFFER_MIN;
			Codec_GetCap(codec, &cap);
			min = cap.cap.valueINT;
			cap.CapCode = CAP_BUFFER_MAX;
			Codec_GetCap(codec, &cap);
			max = cap.cap.valueINT;
			break;
		default:
			min = max = 0;
		}

		/*setup CB*/
		if (!codec->CB && max) {
			codec->CB = CB_New(CUsize, max);
			codec->CB->Min = min;
			codec->CB->odm = codec->odm;
		}

		/*check re-ordering*/
		cap.CapCode = CAP_VID_REORDER;
		Codec_GetCap(codec, &cap);
		if (cap.cap.valueINT) codec->is_reordering = 1;
	}

	/*assign the first base layer as the codec clock by default, or current channel clock if no clock set
	Also assign codec priority here*/
	if (!ch->esd->dependsOnESID || !codec->ck) {
		codec->ck = ch->clock;
		codec->Priority = ch->esd->streamPriority;
		/*insert base layer first - note we are sure this is a stream of the same type
		as the codec (other streams - OCI, MPEG7, MPEGJ - are not added that way)*/
		return ChainInsertEntry(codec->inChannels, ch, 0);
	}
	else {
		/*make sure all channels are in order*/
		for (i=0; i<ChainGetCount(codec->inChannels); i++) {
			a_ch = ChainGetEntry(codec->inChannels, i);
			if (ch->esd->dependsOnESID == a_ch->esd->ESID) {
				return ChainInsertEntry(codec->inChannels, ch, i+1);
			}
			if (a_ch->esd->dependsOnESID == ch->esd->ESID) {
				return ChainInsertEntry(codec->inChannels, ch, i);
			}
		}
		/*by default append*/
		return ChainAddEntry(codec->inChannels, ch);
	}
}

Bool Codec_RemoveChannel(GenericCodec *codec, struct _es_channel *ch)
{
	s32 i;
	i = ChainFindEntry(codec->inChannels, ch);
	if (i>=0) {
		if (codec->decio) codec->decio->Codec_DetachStream(codec->decio, ch->esd->ESID);
		ChainDeleteEntry(codec->inChannels, (u32) i);
		return 1;
	}
	return 0;
}



/*scalable browsing of input channels: find the AU with the lowest DTS on all input channels*/
void Decoder_GetNextAU(GenericCodec *codec, Channel **activeChannel, LPAUBUFFER *nextAU)
{
	Channel *ch;
	LPAUBUFFER AU;
	u32 count, minDTS, i;
	count = ChainGetCount(codec->inChannels);
	*nextAU = NULL;
	*activeChannel = NULL;

	if (!count) return;

	minDTS = (u32) -1;
	/*reverse browsing to make sure we fill enhancement before base layer*/
	for (i=count;i>0;i--) {
		ch = ChainGetEntry(codec->inChannels, i-1);

		if ((codec->type==M4ST_OCR) && ch->IsClockInit) {
			/*check duration - we assume that scalable OCR streams are just pure nonsense...*/
			if (ch->is_pulling && codec->odm->duration) {
				if (CK_GetTime(codec->ck) > codec->odm->duration) 
					Channel_EndOfStream(ch);
			}
			return;
		}

		AU = Channel_GetAU(ch);
		if (!AU) continue;

		/*we use <= to make sure we first fetch data on base layer if same
		DTS (which is possible in spatial scalability)*/
		if (AU->DTS <= minDTS) {
			minDTS = AU->DTS;
			*activeChannel = ch;
			*nextAU = AU;
		}
	}

	/*FIXME - we're breaking sync (couple of frames delay)*/
	if (*nextAU && codec->is_reordering) (*nextAU)->CTS = (*nextAU)->DTS;
}


M4Err SystemCodec_Process(GenericCodec *codec, u32 TimeAvailable)
{
	LPAUBUFFER AU;
	Channel *ch;
	u32 now, obj_time;
	Bool scene_locked, invalidate_scene;
	Bool check_next_unit;
	M4Err e = M4OK;

	scene_locked = 0;
	invalidate_scene = 0;
	
	/*for resync, if needed - the logic behind this is that there is no composition memory on sytems codecs so
	"frame dropping" is done by preventing the renderer from redrawing after an update and decoding following AU
	so that the renderer is always woken up once all late systems AUs are decoded. This flag is overriden when 
	seeking*/
	check_next_unit = codec->odm->term->bifs_can_resync;
	
check_unit:

#ifdef M4_DEF_MediaControl
	/*muting systems codec means we don't decode until mute is off - likely there will be visible however
	there is no other way to decode system AUs without modifying the content, which is what mute is about on visual...*/
	if (codec->Muted) goto exit;
#endif
	

	/*fetch next AU in DTS order for this codec*/
	Decoder_GetNextAU(codec, &ch, &AU);

	/*no active channel return*/
	if (!AU || !ch) {
		/*if the codec is in EOS state, move to STOP*/
		if (codec->Status == CODEC_EOS) {
			MM_StopCodec(codec);
		}
		goto exit;
	}
	

	/*get the object time*/
	obj_time = CK_GetTime(codec->ck);

	/*clock is not started*/
	if (ch->first_au_fetched && !CK_IsStarted(ch->clock)) goto exit;

	/*check timing based on the input channel and main FPS*/
	now = CK_GetTime(ch->clock);
	if ( (AU->DTS > now + codec->odm->term->half_frame_duration) ) goto exit;

	/*check seeking and update timing - do NOT use the base layer, since BIFS streams may depend on other 
	streams not on the same clock*/
	if (codec->last_unit_cts == AU->CTS ) {
		/*hack for RTSP streaming of systems streams, except InputSensor*/
		if (!ch->is_pulling && (codec->type != M4ST_INTERACT) && (AU->dataLength == codec->prev_au_size)) {
			Channel_DropAU(ch);
			goto check_unit;
		}
		/*seeking for systems is done by not releasing the graph until seek is done*/
		check_next_unit = 1;
	} 
	/*set system stream timing*/
	else {
		codec->last_unit_cts = AU->CTS;
	}

	/*lock scene*/
	if (!scene_locked) {
		Term_LockScene(codec->odm->term, 1);
		scene_locked = 1;
	}

	e = codec->decio->Codec_Process(codec->decio, (unsigned char *) AU->data, AU->dataLength, 
		ch->esd->ESID, NULL, 0, AU->RAP, AU->PaddingBits, MM_LEVEL_NORMAL);

	codec->prev_au_size = AU->dataLength;
	/*current media time for system objects is the clock time, since the media is likely to have random 
	updates in time*/
	codec->odm->current_time = CK_GetTime(codec->ck);

	/*destroy this AU*/
	Channel_DropAU(ch);

	if (e) goto exit;

	/*always force redraw for system codecs*/
	invalidate_scene = 1;

	/*if no release restart*/
	if (check_next_unit) goto check_unit;
	
exit:
	if (scene_locked) Term_LockScene(codec->odm->term, 0);
	if (invalidate_scene) Term_InvalidateScene(codec->odm->term);
	return e;
}

/*Get a pointer to the next CU buffer*/
static M4INLINE M4Err LockCompositionUnit(GenericCodec *dec, u32 CU_TS, char **outBuffer, u32 *availableSize)
{
	LPCUBUFFER cu;
	*outBuffer = NULL;
	*availableSize = 0;

	if (!dec->CB) return M4BadParam;
	
	cu = CB_LockInput(dec->CB, CU_TS);
	if (!cu ) return M4OutOfMem;
	
	cu->TS = CU_TS;
	*outBuffer = cu->data;
	*availableSize = dec->CB->UnitSize;
	return M4OK;
}


static M4INLINE M4Err UnlockCompositionUnit(GenericCodec *dec, u32 CTS, u32 NbBytes)
{
	/*temporal scalability disabling: if we already rendered this, no point getting further*/
	if (CTS < dec->CB->LastRenderedTS) NbBytes = 0;

	/*unlock the CB*/
	CB_UnlockInput(dec->CB, CTS, NbBytes);
	return M4OK;
}


static M4Err ResizeCompositionBuffer(GenericCodec *dec, u32 NewSize)
{
	if (!dec || !dec->CB || !NewSize) return M4BadParam;
	if (NewSize == dec->CB->UnitSize) return M4OK;

	CB_ResizeBuffers(dec->CB, NewSize);
	/*update config*/
	MO_UpdateCaps(dec->odm->mo);
	return M4OK;
}

M4Err MediaCodec_Process(GenericCodec *codec, u32 TimeAvailable)
{
	LPAUBUFFER AU;
	Channel *ch;
	char *cu_buf;
	u32 cu_buf_size, mmlevel, deltaTS;
	u32 first, entryTime, now, obj_time;
	M4Err e = M4OK;

	/*if video codec muted don't decode
	if audio codec muted we dispatch to keep sync in place*/
	if (codec->Muted && (codec->type==M4ST_VISUAL) ) return M4OK;

	entryTime = Term_GetTime(codec->odm->term);

	/*fetch next AU in DTS order for this codec*/
	Decoder_GetNextAU(codec, &ch, &AU);
	/*no active channel return*/
	if (!AU || !ch) {
		/*if the codec is in EOS state, assume we're done*/
		if (codec->Status == CODEC_EOS) {
			/*if codec is reordering, try to flush it*/
			if (codec->is_reordering) {
				if ( LockCompositionUnit(codec, codec->last_unit_cts+1, &cu_buf, &cu_buf_size) == M4OutOfMem) 
					return M4OK;
				e = codec->decio->Codec_Process(codec->decio, NULL, 0, 0, cu_buf, &cu_buf_size, 0, 0, 0);
				if (e==M4OK) e = UnlockCompositionUnit(codec, codec->last_unit_cts+1, cu_buf_size);
			}
			MM_StopCodec(codec);
			if (codec->CB) CB_SetEndOfStream(codec->CB);
		}
		return M4OK;
	}

	/*get the object time*/
	obj_time = CK_GetTime(codec->ck);
	/*Media Time for media codecs is updated in the CB*/

	if (!codec->CB) {
		Channel_DropAU(ch);
		return M4BadParam;
	}
	
	/*try to refill the full buffer*/
	first = 1;
	while (codec->CB->Capacity > codec->CB->UnitCount) {
		/*set media processing level*/
		mmlevel = MM_LEVEL_NORMAL;
		/*SEEK: if the last frame had the same TS, we are seeking. Ask the codec to drop*/
		if (codec->last_unit_cts && (codec->last_unit_cts == AU->CTS) && !ch->esd->dependsOnESID) {
			mmlevel = MM_LEVEL_SEEK;
		}
		/*only if clock is running since otherwise timing is irrelevant*/
		else if (CK_IsStarted(codec->ck)) {
			/*only perform drop in normal playback*/
			if (codec->CB->Status == CB_PLAY) {
				if (AU->CTS < obj_time) {
					/*extremely late, even if we decode the renderer will drop the frame 
					so set the level to drop*/
					mmlevel = MM_LEVEL_DROP;
				}
				/*we are late according to the media manager*/
				else if (codec->PriorityBoost) {
					mmlevel = MM_LEVEL_VERY_LATE;
				}
				/*otherwise we must have an idea of the load in order to set the right level
				use the composition buffer for that, only on the first frame*/
				else if (first) {
					//if the CB is almost empty set to very late
					if (codec->CB->UnitCount <= codec->CB->Min+1) {
						mmlevel = MM_LEVEL_VERY_LATE;
					} else if (codec->CB->UnitCount * 2 <= codec->CB->Capacity) {
						mmlevel = MM_LEVEL_LATE;
					}
					first = 0;
				}
			}
		}

		/*when using temporal scalability make sure we can decode*/
		if (ch->esd->dependsOnESID && (codec->last_unit_dts > AU->DTS)){
//			printf("SCALABLE STREAM DEAD!!\n");
			goto drop;
		}

		if ( LockCompositionUnit(codec, AU->CTS, &cu_buf, &cu_buf_size) == M4OutOfMem) 
			return M4OK;

		e = codec->decio->Codec_Process(codec->decio, AU->data, AU->dataLength, 
			ch->esd->ESID, cu_buf, &cu_buf_size,
			AU->RAP, AU->PaddingBits, mmlevel);

		/*input is too small, resize composition memory*/
		switch (e) {
		case M4BufferTooSmall:
			/*release but no dispatch*/
			UnlockCompositionUnit(codec, AU->CTS, 0);
			if (ResizeCompositionBuffer(codec, cu_buf_size)==M4OK) continue;
			break;
		case M4OK:
			e = UnlockCompositionUnit(codec, AU->CTS, cu_buf_size);
			break;
		/*this happens a lot when using non-MPEG-4 streams (ex: ffmpeg demuxer)*/
		case M4PackedFrames:
			e = UnlockCompositionUnit(codec, AU->CTS, cu_buf_size);
			/*try to update the timing (in ms) for the next CUs*/
			if (codec->bytes_per_sec) {
				deltaTS = cu_buf_size * 1000 / codec->bytes_per_sec;
			} else {
				deltaTS = 30;
			}
			AU->DTS += deltaTS;
			AU->CTS += deltaTS;
			continue;
		default:
			UnlockCompositionUnit(codec, AU->CTS, 0);
			/*error - if the object is in intitial buffering resume it!!*/
			CB_AbortBuffering(codec->CB);
			break;
		}

		codec->last_unit_dts = AU->DTS;
		/*remember base layer timing*/
		if (!ch->esd->dependsOnESID) codec->last_unit_cts = AU->CTS;

drop:
		Channel_DropAU(ch);
		if (e) return e;

		/*escape from decoding loop only if above critical limit - this is to avoid starvation on audio*/
		if (codec->CB->UnitCount > codec->CB->Min) {
			now = Term_GetTime(codec->odm->term);
			if (now - entryTime >= TimeAvailable - TIME_CHECK) {
				return M4OK;
			}
		}

		Decoder_GetNextAU(codec, &ch, &AU);
		if (!ch || !AU) return M4OK;
	}
	return M4OK;
}



M4Err Decoder_ProcessData(GenericCodec *codec, u32 TimeAvailable)
{
	if (codec->Status == CODEC_STOP) return M4OK;

#ifdef M4_DEF_MediaControl
	codec->Muted = (codec->odm->media_ctrl && codec->odm->media_ctrl->control->mute) ? 1 : 0;
#endif

	switch (codec->type) {
	/*audio/visual*/
	case M4ST_VISUAL:
	case M4ST_AUDIO:
		return MediaCodec_Process(codec, TimeAvailable);
	/*systems*/
	case M4ST_BIFS:
	case M4ST_OD:
	case M4ST_INTERACT:
		return SystemCodec_Process(codec, TimeAvailable);

	/*OCR: needed for OCR in pull mode (dummy streams used to sync various sources)*/
	case M4ST_OCR:
	{
		LPAUBUFFER AU;
		Channel *ch;
		/*fetch next AU on OCR (empty AUs)*/
		Decoder_GetNextAU(codec, &ch, &AU);

		/*no active channel return*/
		if (!AU || !ch) {
			/*if the codec is in EOS state, move to STOP*/
			if (codec->Status == CODEC_EOS) {
				MM_StopCodec(codec);
#ifdef M4_DEF_MediaControl
				/*if a mediacontrol is ruling this OCR*/
				if (codec->odm->media_ctrl && codec->odm->media_ctrl->control->loop) MC_Restart(codec->odm); 
#endif
			}
		}
	}
		return M4OK;
	default:
		return M4OK;
	}
}


M4Err Codec_GetCap(GenericCodec *codec, CapObject *cap)
{
	cap->cap.valueINT = 0;
	if (!codec->decio) return M4OK;
	return codec->decio->Codec_GetCapabilities(codec->decio, cap);
}

M4Err Codec_SetCap(GenericCodec *codec, CapObject cap)
{
	if (!codec->decio) return M4OK;
	return codec->decio->Codec_SetCapabilities(codec->decio, cap);
}


void Codec_SetStatus(GenericCodec *codec, u32 Status)
{
	if (!codec) return;

	if (Status == CODEC_PAUSE) codec->Status = CODEC_STOP;
	else if (Status == CODEC_BUFFER) codec->Status = CODEC_PLAY;
	else if (Status == CODEC_PLAY) {
		codec->last_unit_cts = 0;
		codec->prev_au_size = 0;
		codec->Status = Status;
	}
	else codec->Status = Status;

	if (!codec->CB) return;
	
	/*notify CB*/
	switch (Status) {
	case CODEC_PLAY:
		CB_SetStatus(codec->CB, CB_PLAY);
		return;
	case CODEC_PAUSE:
		CB_SetStatus(codec->CB, CB_PAUSE);
		return;
	case CODEC_STOP:
		CB_SetStatus(codec->CB, CB_STOP);
		return;
	case CODEC_EOS:
		/*do NOT notify CB yet, wait till last AU is decoded*/
		return;
	case CODEC_BUFFER:
	default:
		return;
	}
}

M4Err Codec_LoadPlugin(GenericCodec *codec, ESDescriptor *esd, u32 PL)
{
	char szPrefDec[500];
	char *sOpt;
	DecoderInterface *ifce;
	u32 i, plugCount;
	char *cfg;
	u32 cfg_size;
	M4Client *term = codec->odm->term;


	if (esd->decoderConfig->decoderSpecificInfo) {
		cfg = esd->decoderConfig->decoderSpecificInfo->data;
		cfg_size = esd->decoderConfig->decoderSpecificInfo->dataLength;
	} else {
		cfg = NULL;
		cfg_size = 0;
	}

	/*a bit dirty, if FFMPEG is used for demuxer load it for decoder too*/
	if (!stricmp(codec->odm->net_service->ifce->plugin_name, "FFMPEG demuxer")) {
		sOpt = "FFMPEG decoder";
	} else {
		/*use user-defined plugin if any*/
		sOpt = NULL;
		switch (esd->decoderConfig->streamType) {
		case M4ST_VISUAL:
			sOpt = IF_GetKey(term->user->config, "Systems", "DefVideoDec");
			break;
		case M4ST_AUDIO:
			sOpt = IF_GetKey(term->user->config, "Systems", "DefAudioDec");
			break;
		default:
			break;
		}
	}
	
	if (sOpt) {
		if (PM_LoadInterfaceByName(term->user->plugins, sOpt, M4MEDIADECODERINTERFACE, (void **) &ifce)) {
			if (ifce->CanHandleStream(ifce, esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication, cfg, cfg_size, PL) ) {
				codec->decio = ifce;
				return M4OK;
			}
			PM_ShutdownInterface(ifce);		
		}
	}

	/*prefered codec plugin per streamType/objectType from config*/
	sprintf(szPrefDec, "codec_%.2x_%.2x", esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication);
	sOpt = IF_GetKey(term->user->config, "Systems", szPrefDec);
	if (sOpt) {
		if (PM_LoadInterfaceByName(term->user->plugins, sOpt, M4MEDIADECODERINTERFACE, (void **) &ifce)) {
			if (ifce->CanHandleStream(ifce, esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication, cfg, cfg_size, PL) ) {
				codec->decio = ifce;
				return M4OK;
			}
			PM_ShutdownInterface(ifce);		
		}
	}
	/*not found, check all plugins*/
	plugCount = PM_GetPluginsCount(term->user->plugins);
	for (i = 0; i < plugCount ; i++) {
		if (!PM_LoadInterface(term->user->plugins, i, M4MEDIADECODERINTERFACE, (void **) &ifce)) continue;
		if (ifce->CanHandleStream(ifce, esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication, cfg, cfg_size, PL) ) {
			codec->decio = ifce;
			return M4OK;
		}
		PM_ShutdownInterface(ifce);
	}
	return M4CodecNotFound;
}

M4Err Codec_Load(GenericCodec *codec, ESDescriptor *esd, u32 PL)
{
	switch (esd->decoderConfig->streamType) {
	/*media plugins*/
	case M4ST_VISUAL:
	case M4ST_AUDIO:
		return Codec_LoadPlugin(codec, esd, PL);

	/*internal decoders*/
	/*OD codec*/
	case M4ST_OD:
		codec->decio = NewODSysCodec(PL);
		codec->DeleteSystemDecoder = ODS_Codec_Delete;
		return M4OK;

	/*OCR has no codec, just a channel*/
	case M4ST_OCR:
		codec->decio = NULL;
		return M4OK;
	/*BIFS*/
	case M4ST_BIFS:
		codec->decio = NewBIFSCodec(PL);
		codec->DeleteSystemDecoder = BIFS_DeleteDec;
		return M4OK;
	/*InteractionStream*/
	case M4ST_INTERACT:
#ifdef M4_DEF_InputSensor
		codec->decio = NewISCodec(PL);
		codec->DeleteSystemDecoder = IS_DeleteDec;
		return M4OK;
#else
		return M4CodecNotFound;
#endif

	/*TO DO - OCI codec, MPEG-J...*/
	default:
		return M4CodecNotFound;
	}
}


void DeleteCodec(GenericCodec *codec)
{
	if (ChainGetCount(codec->inChannels)) return;

	switch (codec->type) {
	case M4ST_VISUAL:
	case M4ST_AUDIO:
		PM_ShutdownInterface(codec->decio);
		break;
	default:
		if (codec->DeleteSystemDecoder) codec->DeleteSystemDecoder(codec->decio);
		if (codec->type == M4ST_INTERACT) {
			MX_P(codec->odm->term->net_mx);
			ChainDeleteItem(codec->odm->term->input_streams, codec);
			MX_V(codec->odm->term->net_mx);
		}
		break;
	}
	if (codec->CB) CB_Delete(codec->CB);
	DeleteChain(codec->inChannels);
	free(codec);
}


