/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/intern/m4_scenegraph_dev.h>

SGCommand *SG_NewCommand(u32 tag)
{
	SGCommand *ptr;
	SAFEALLOC(ptr, sizeof(SGCommand));
	if (!ptr) return NULL;
	ptr->tag = tag;

	ptr->new_proto_list = NewChain();
	ptr->command_fields = NewChain();
	return ptr;
}

void SG_DeleteCommand(SGCommand *com)
{
	u32 i;
	if (!com) return;

	while (ChainGetCount(com->command_fields)) {
		CommandFieldInfo *inf = ChainGetEntry(com->command_fields, 0);
		ChainDeleteEntry(com->command_fields, 0);

		switch (inf->fieldType) {
		case FT_SFNode:
			if (inf->field_ptr && *(SFNode **) inf->field_ptr) Node_Unregister(*(SFNode **) inf->field_ptr, com->node);
			break;
		case FT_MFNode:
			Node_ForceResetChildren(com->node, *(Chain**) inf->field_ptr);
			DeleteChain(*(Chain**) inf->field_ptr);
			break;
		default:
			SG_DeleteFieldPointer(inf->field_ptr, inf->fieldType);
			break;
		}
		free(inf);
	}
	DeleteChain(com->command_fields);

	for (i=0; i<ChainGetCount(com->new_proto_list); i++) {
		LPPROTO proto = ChainGetEntry(com->new_proto_list, i);
		SG_DeleteProto(proto);
	}
	DeleteChain(com->new_proto_list);

	if (com->node) Node_Unregister(com->node, NULL);

	if (com->del_proto_list) free(com->del_proto_list);
	if (com->def_name) free(com->def_name);
	if (com->graph)	SG_Delete(com->graph);
	free(com);
}

static void SG_CheckFieldChanges(SFNode *node, FieldInfo *field)
{
	/*Notify eventOut in all cases to handle protos*/
	Node_OnEventOut(node, field->allIndex);
	/*and propagate eventIn if any*/
	if (field->on_event_in) {
		field->on_event_in(node);
#ifdef M4_DEF_Script
	} else if ((Node_GetTag(node) == TAG_Script) && (field->eventType==ET_EventIn)) {
		Script_EventIn(node, field);
#endif
	}
}


M4Err SG_ApplyCommand(LPSCENEGRAPH graph, SGCommand *com)
{
	M4Err e;
	CommandFieldInfo *inf;
	FieldInfo field;
	SFNode *def, *node;
	void *slot_ptr;

	if (!com || !graph) return M4BadParam;

	e = M4OK;
	switch (com->tag) {
	case SG_SceneReplace:
		if (graph==com->graph) return M4OK;
		/*reset current graph*/
		SG_Reset(graph);

		/*FIXME - THIS IS BROKEN, WE SHALL NOT DESTROY THE COMMAND GRAPH, BUT DUPLICATE IT*/

		/*destroy all lists and node reg*/
		DeleteChain(graph->Routes);
		graph->Routes = com->graph->Routes;
		DeleteChain(graph->protos);
		graph->protos = com->graph->protos;
		/*these 2 are in case of protos ...*/
		DeleteChain(graph->routes_to_activate);
		graph->routes_to_activate = com->graph->routes_to_activate;
		DeleteChain(graph->routes_to_destroy);
		graph->routes_to_destroy = com->graph->routes_to_destroy;
		free(graph->node_registry);
		graph->node_registry = com->graph->node_registry;
		graph->node_reg_alloc = com->graph->node_reg_alloc;
		graph->node_reg_size = com->graph->node_reg_size;
		graph->pOwningProto = com->graph->pOwningProto;

		graph->RootNode = com->graph->RootNode;
		free(com->graph);
		com->graph = NULL;
		return M4OK;

	case SG_NodeReplace:
		if (!ChainGetCount(com->command_fields)) return M4OK;
		inf = ChainGetEntry(com->command_fields, 0);
		e = Node_ReplaceAllInstances(com->node, inf->new_node, 0);
		if (inf->new_node) Node_Register(inf->new_node, NULL);
		break;
	case SG_FieldReplace:
	{
		u32 i;
		Chain *container, *list;
		if (!ChainGetCount(com->command_fields)) return M4OK;
		inf = ChainGetEntry(com->command_fields, 0);

		e = Node_GetField(com->node, inf->fieldIndex, &field);
		if (e) return e;

		switch (field.fieldType) {
		case FT_SFNode:
		{
			node = *((SFNode **) field.far_ptr);
			e = Node_Unregister(node, com->node);
			*((SFNode **) field.far_ptr) = inf->new_node;
			if (!e) Node_Register(inf->new_node, com->node);
			SG_NodeChanged(com->node, &field);
			break;
		}
		case FT_MFNode:
			container = * ((Chain **) field.far_ptr);
			list = * ((Chain **) inf->field_ptr);
			Node_ResetChildren(com->node, container);

			for (i=0; i<ChainGetCount(list); i++) {
				node = ChainGetEntry(list, i);
				ChainAddEntry(container, node);
				if (!e) Node_Register(node, com->node);
			}
			SG_NodeChanged(com->node, &field);
			break;
		default:
			/*this is a regular field, reset it and clone - we cannot switch pointers since the
			original fields are NOT pointers*/
			if (!SG_IsSFField(field.fieldType)) {
				e = MFField_Reset(field.far_ptr, field.fieldType);
			}
			if (e) return e;
			SG_CopyField(field.far_ptr, inf->field_ptr, field.fieldType);
			SG_NodeChanged(com->node, &field);
			break;
		}
		break;
	}

	case SG_IndexedReplace:
	{
		u32 sftype;
		if (!ChainGetCount(com->command_fields)) return M4OK;
		inf = ChainGetEntry(com->command_fields, 0);

		e = Node_GetField(com->node, inf->fieldIndex, &field);
		if (e) return e;

		/*if MFNode remove the child and set new node*/
		if (field.fieldType == FT_MFNode) {
			/*we must remove the node before in case the new node uses the same ID (not forbidden) and this
			command removes the last instance of the node with the same ID*/
			Node_ReplaceChild(com->node, *((Chain**) field.far_ptr), inf->pos, inf->new_node);
			Node_Register(inf->new_node, com->node);
			SG_NodeChanged(com->node, &field);
		}
		/*erase the field item*/
		else {
			if ((inf->pos < 0) || ((u32) inf->pos >= ((GenMFField *) field.far_ptr)->count) ) {
				inf->pos = ((GenMFField *)field.far_ptr)->count - 1;
			}
			e = MFField_GetItem(field.far_ptr, field.fieldType, & slot_ptr, inf->pos);
			if (e) return e;
			sftype = SG_GetSFType(field.fieldType);
			SG_CopyField(slot_ptr, inf->field_ptr, sftype);
			SG_NodeChanged(com->node, &field);
		}
		break;
	}
	case SG_RouteReplace:
	{
		LPROUTE newR, r;
		r = SG_FindRoute(graph, com->RouteID);

		def = SG_FindNode(graph, com->fromNodeID);
		node = SG_FindNode(graph, com->toNodeID);
		if (!node || !def) return M4InvalidNode;
		newR = SG_NewRoute(graph, def, com->fromFieldIndex, node, com->toFieldIndex);
		/*SG_DeleteRoute(r);*/
		Route_SetID(newR, com->RouteID);

		if (com->def_name) Route_SetName(newR, com->def_name);
		break;
	}
	case SG_NodeDelete:
	{
		if (com->node) Node_ReplaceAllInstances(com->node, NULL, 0);
		break;
	}
	case SG_RouteDelete:
	{
		return SG_DeleteRouteByID(graph, com->RouteID);
	}
	case SG_IndexedDelete:
	{
		if (!ChainGetCount(com->command_fields)) return M4OK;
		inf = ChainGetEntry(com->command_fields, 0);

		e = Node_GetField(com->node, inf->fieldIndex, &field);
		if (e) return e;
		if (SG_IsSFField(field.fieldType)) return M4NonCompliantBitStream;

		/*then we need special handling in case of a node*/
		if (SG_GetSFType(field.fieldType) == FT_SFNode) {
			e = Node_ReplaceChild(com->node, * ((Chain **) field.far_ptr), inf->pos, NULL);
		} else {
			if ((inf->pos < 0) || ((u32) inf->pos >= ((GenMFField *) field.far_ptr)->count) ) {
				inf->pos = ((GenMFField *)field.far_ptr)->count - 1;
			}
			/*this is a regular MFField, just remove the item (realloc)*/
			e = MFField_Remove(field.far_ptr, field.fieldType, inf->pos);
		}
		/*deletion -> node has changed*/
		if (!e) SG_NodeChanged(com->node, &field);
		break;
	}
	case SG_NodeInsert:
	{
		if (!ChainGetCount(com->command_fields)) return M4OK;
		inf = ChainGetEntry(com->command_fields, 0);

		e = Node_InsertChild(com->node, inf->new_node, inf->pos);
		if (!e) Node_Register(inf->new_node, com->node);
		/*notify (children is the 3rd field, so 2 0-based)*/
		if (!e) Node_OnEventOut(com->node, 2);
		break;
	}
	case SG_RouteInsert:
	{
		LPROUTE r;
		def = SG_FindNode(graph, com->fromNodeID);
		node = SG_FindNode(graph, com->toNodeID);
		if (!node || !def) return M4InvalidNode;
		r = SG_NewRoute(graph, def, com->fromFieldIndex, node, com->toFieldIndex);
		if (com->RouteID) Route_SetID(r, com->RouteID);
		if (com->def_name) {
			Route_SetName(r, com->def_name);
			free(com->def_name);
			com->def_name = NULL;
		}
		break;
	}
	case SG_IndexedInsert:
	{
		u32 sftype;
		if (!ChainGetCount(com->command_fields)) return M4OK;
		inf = ChainGetEntry(com->command_fields, 0);
		e = Node_GetField(com->node, inf->fieldIndex, &field);
		if (e) return e;

		/*rescale the MFField and parse the SFField*/
		if (field.fieldType != FT_MFNode) {
			if (inf->pos == -1) {
				e = MFField_Append(field.far_ptr, field.fieldType, & slot_ptr);
			} else {
				e = MFField_Insert(field.far_ptr, field.fieldType, & slot_ptr, inf->pos);
			}
			if (e) return e;
			sftype = SG_GetSFType(field.fieldType);
			SG_CopyField(slot_ptr, inf->field_ptr, sftype);
			SG_NodeChanged(com->node, &field);
		} else {
			if (inf->new_node) {
				e = InsertSFNode(field.far_ptr, inf->new_node, inf->pos);
				if (e) return e;
				Node_Register(inf->new_node, com->node);
			}
		}
		if (!e) SG_CheckFieldChanges(com->node, &field);
		break;
	}
	default:
		return M4NotSupported;
	}
	return e;
}

CommandFieldInfo *SG_NewFieldCommand(SGCommand *com)
{
	CommandFieldInfo *ptr;
	SAFEALLOC(ptr, sizeof(CommandFieldInfo));
	ChainAddEntry(com->command_fields, ptr);
	return ptr;
}


M4Err SG_ApplyCommandList(LPSCENEGRAPH graph, Chain *comList)
{
	M4Err e;
	u32 i;
	for (i=0; i<ChainGetCount(comList); i++) {
		SGCommand *com = ChainGetEntry(comList, i);
		e = SG_ApplyCommand(graph, com);
		if (e) return e;
	}
	return M4OK;
}

