/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_scenegraph_dev.h>
#include <gpac/intern/m4_node_tables.h>

LPSCENEGRAPH NewSceneGraph(
					void (*UserNodeInit)(void *NodeInitCallback, SFNode *newNode), void *NodeInitCallback,
					void (*NodeModified)(void *ModifCallback, SFNode *node), void *ModifCallback				

					)
{
	LPSCENEGRAPH tmp;
	SAFEALLOC(tmp, sizeof(SceneGraph));
	if (!tmp) return NULL;

	tmp->protos = NewChain();
	tmp->unregistered_protos = NewChain();

	tmp->node_registry = malloc(sizeof(SFNode *) * NODEREG_STEP_ALLOC);
	tmp->node_reg_alloc = NODEREG_STEP_ALLOC;

	tmp->Routes = NewChain();
	tmp->routes_to_activate = NewChain();
	tmp->routes_to_destroy = NewChain();

	/*store callbacks*/
	tmp->B_NodeInit = UserNodeInit;
	tmp->NodeInitCallback = NodeInitCallback;
	tmp->NodeModified = NodeModified;
	tmp->ModifCallback = ModifCallback;

	return tmp;
}

LPSCENEGRAPH SG_NewSubScene(LPSCENEGRAPH scene)
{
	LPSCENEGRAPH tmp;
	if (!scene) return NULL;
	tmp = NewSceneGraph(scene->B_NodeInit, scene->NodeInitCallback, scene->NodeModified, scene->ModifCallback);
	if (!tmp) return NULL;
	tmp->parent_scene = scene;
	tmp->userpriv = scene->userpriv;
	/*by default use the same scene time (protos need that) - user overrides it if needed (inlineScene)*/
	tmp->GetSceneTime = scene->GetSceneTime;
	tmp->SceneCallback = scene->SceneCallback;
	tmp->GetExternProtoLib = scene->GetExternProtoLib;
	tmp->js_ifce = scene->js_ifce;
	tmp->Script_Load = scene->Script_Load;
	return tmp;
}


void SG_SetSceneTime(LPSCENEGRAPH sg, Double (*GetSceneTime)(void *scene_callback), void *cbck)
{
	sg->GetSceneTime = GetSceneTime;
	sg->SceneCallback = cbck;
}

Double Node_GetSceneTime(SFNode *node)
{
	return SG_GetSceneTime(node->sgprivate->scenegraph);
}

Double SG_GetSceneTime(LPSCENEGRAPH sg)
{
	if (!sg->GetSceneTime) return 0.0;
	return sg->GetSceneTime(sg->SceneCallback);
}

static void SG_destroy_routes(LPSCENEGRAPH sg)
{
	while (ChainGetCount(sg->routes_to_destroy) ) {
		Route *r = ChainGetEntry(sg->routes_to_destroy, 0);
		ChainDeleteEntry(sg->routes_to_destroy, 0);
		RemoveActivatedRoute(sg, r);
		Route_Destroy(r);
	}
}

void SG_Delete(LPSCENEGRAPH sg)
{	
	if (!sg) return;

	SG_Reset(sg);

	DeleteChain(sg->Routes);
	DeleteChain(sg->protos);
	DeleteChain(sg->unregistered_protos);
	DeleteChain(sg->routes_to_activate);
	DeleteChain(sg->routes_to_destroy);
	free(sg->node_registry);

	free(sg);
}


void SG_Reset(LPSCENEGRAPH sg)
{
	u32 i;
	if (!sg) return;
	if (sg->RootNode) Node_Delete(sg->RootNode, NULL);
	sg->RootNode = NULL;

	while (ChainGetCount(sg->routes_to_activate)) {
		ChainDeleteEntry(sg->routes_to_activate, 0);
	}

	/*destroy all routes*/
	while (ChainGetCount(sg->Routes)) {
		Route *r = ChainGetEntry(sg->Routes, 0);
		/*this will unregister the route from the graph, so don't delete the chain entry*/
		SG_DeleteRoute(r);

	}

	/*remove DEF'ed nodes not referenced in graph (this happens during paring)*/
	for (i=0; i<sg->node_reg_size; i++) {
		assert(sg->node_registry[i]->sgprivate->num_instances==0);
		DestroyNode(sg->node_registry[i]);
	}

	/*destroy all proto*/
	while (ChainGetCount(sg->protos)) {
		PrototypeNode *p = ChainGetEntry(sg->protos, 0);
		/*this will unregister the proto from the graph, so don't delete the chain entry*/
		SG_DeleteProto(p);
	}
	/*destroy all proto*/
	while (ChainGetCount(sg->unregistered_protos)) {
		PrototypeNode *p = ChainGetEntry(sg->unregistered_protos, 0);
		/*this will unregister the proto from the graph, so don't delete the chain entry*/
		SG_DeleteProto(p);
	}

	/*last destroy all routes*/
	SG_destroy_routes(sg);

	sg->simulation_tick = 0;
}


M4INLINE SFNode *SG_SearchForDuplicateNodeID(LPSCENEGRAPH sg, u32 nodeID, SFNode *toExclude)
{
	u32 i;
	for (i=0; i<sg->node_reg_size; i++) {
		if (sg->node_registry[i] == toExclude) continue;
		if (sg->node_registry[i]->sgprivate->NodeID == nodeID) {
			return sg->node_registry[i];
		}
	}
	return NULL;
}

M4INLINE SFNode *SG_SearchForNode(LPSCENEGRAPH sg, SFNode *node)
{
	u32 i;
	for (i=0; i<sg->node_reg_size; i++) {
		if (sg->node_registry[i] == node) {
			return sg->node_registry[i];
		}
	}
	return NULL;
}

static M4INLINE u32 node_search(LPSCENEGRAPH sg, u32 low_pos, u32 high_pos, u32 ID) 
{
	u32 mid_pos;

	assert(low_pos<high_pos);

	mid_pos = (high_pos+low_pos)/2;
	
	if (sg->node_registry[mid_pos]->sgprivate->NodeID == ID) return mid_pos;

	/* greater than middle, search upper half */
	if (sg->node_registry[mid_pos]->sgprivate->NodeID < ID) {
		if (mid_pos+1==sg->node_reg_size) {
			if (sg->node_registry[sg->node_reg_size-1]->sgprivate->NodeID >= ID) return sg->node_reg_size-1;
			return sg->node_reg_size;
		}
		if (sg->node_registry[mid_pos+1]->sgprivate->NodeID >= ID) return mid_pos+1;
		
		return node_search(sg, mid_pos+1, high_pos, ID);
	}

	/* less than middle, search lower half */
	if (mid_pos<=1) {
		if (sg->node_registry[0]->sgprivate->NodeID<ID) return 1;
		return 0;
	}
	if (sg->node_registry[mid_pos-1]->sgprivate->NodeID < ID) return mid_pos;
	return node_search(sg, low_pos, mid_pos-1, ID);
}


M4INLINE SFNode *SG_SearchForNodeByID(LPSCENEGRAPH sg, u32 nodeID)
{
	u32 i;
	if (!sg->node_reg_size) return NULL;

	i = node_search(sg, 0, sg->node_reg_size, nodeID);
	if (i>=sg->node_reg_size ||sg->node_registry[i]->sgprivate->NodeID != nodeID) return NULL;
	return sg->node_registry[i];
}

M4INLINE Bool SG_SearchForNodeIndex(LPSCENEGRAPH sg, SFNode *node, u32 *out_index)
{
	u32 i;
	for (i=0; i<sg->node_reg_size; i++) {
		if (sg->node_registry[i] == node) {
			*out_index = i;
			return 1;
		}
	}
	return 0;
}


void SG_SetPrivate(LPSCENEGRAPH sg, void *ptr)
{
	if (sg) sg->userpriv = ptr;
}

void *SG_GetPrivate(LPSCENEGRAPH sg)
{
	return sg ? sg->userpriv : NULL;
}


void SG_SetSizeInfo(LPSCENEGRAPH sg, u32 width, u32 height, Bool usePixelMetrics)
{
	if (!sg) return;
	if (width && height) {
		sg->width = width;
		sg->height = height;
	} else {
		sg->width = sg->height = 0;
	}
	sg->usePixelMetrics = usePixelMetrics;
}

Bool SG_UsePixelMetrics(LPSCENEGRAPH sg)
{
	return (sg ? sg->usePixelMetrics : 0);
}

Bool SG_GetSizeInfo(LPSCENEGRAPH sg, u32 *width, u32 *height)
{
	if (!sg) return 0;
	*width = sg->width;
	*height = sg->height;
	return 1;
}


SFNode *SG_GetRootNode(LPSCENEGRAPH sg)
{
	return sg ? sg->RootNode : NULL;
}

void SG_SetRootNode(LPSCENEGRAPH sg, SFNode *node)
{
	if (sg) sg->RootNode = node;
}


/*this is not a NodeReplace, thus only the given container is updated - pos is 0-based*/
M4Err Node_ReplaceChild(SFNode *node, Chain *container, s32 pos, SFNode *newNode)
{
	u32 count;
	SFNode *n;
	
	count = ChainGetCount(container);
	if (!count) return M4OK;
	/*last item*/
	if ( (pos == -1) || ((u32) pos >= count) ) pos = count - 1;
	n = ChainGetEntry(container, (u32) pos);

	/*delete node*/

	if (n) Node_Unregister(n, node);

	/*delete entry*/
	ChainDeleteEntry(container, (u32) pos);

	if (newNode) ChainInsertEntry(container, newNode, pos);
	return M4OK;
}


M4Err Node_Unregister(SFNode *pNode, SFNode *parentNode)
{
	u32 node_ind, j;
	SceneGraph *pSG;
	
	pSG = pNode->sgprivate->scenegraph;
	/*if this is a proto its is registered in its parent graph, not the current*/
	if (pNode == (SFNode*)pSG->pOwningProto) pSG = pSG->parent_scene;
	assert(pSG);

	if (parentNode) ChainDeleteItem(pNode->sgprivate->parentNodes, parentNode);

	/*unregister the instance*/
	assert(pNode->sgprivate->num_instances);
	pNode->sgprivate->num_instances -= 1;
	
	/*this is just an instance removed*/
	if (pNode->sgprivate->num_instances) return M4OK;

	assert(ChainGetCount(pNode->sgprivate->parentNodes)==0);

	/*if def, remove from sg def table*/
	if (pNode->sgprivate->NodeID) {
		if (!SG_SearchForNodeIndex(pSG, pNode, &node_ind)) {
			assert(0);
		}
		assert (pNode == pSG->node_registry[node_ind]);
		j = pSG->node_reg_size - node_ind - 1;
		if (j) memmove( & pSG->node_registry[node_ind], & pSG->node_registry[node_ind+1], j * sizeof(SFNode *));
		pSG->node_reg_size -= 1;
	}


	/*check all routes from or to this node and destroy them - cf spec*/
	for (j=0; j<ChainGetCount(pSG->Routes); j++) {
		Route *r = ChainGetEntry(pSG->Routes, j);
		if ( (r->ToNode == pNode) || (r->FromNode == pNode)) {
			SG_DeleteRoute(r);
			j--;
		}
	}

	/*delete the node*/
	DestroyNode(pNode);
	return M4OK;
}

M4Err Node_Register(SFNode *node, SFNode *parentNode)
{
	SceneGraph *pSG; 
	
	pSG = node->sgprivate->scenegraph;
	/*if this is a proto register to the parent graph, not the current*/
	if (node == (SFNode*)pSG->pOwningProto) pSG = pSG->parent_scene;
	assert(pSG);

#if 1
	if (node->sgprivate->NodeID) {
		SFNode *the_node = SG_SearchForNode(pSG, node);
		assert(the_node);
		assert(the_node == node);
	}
#endif

	node->sgprivate->num_instances += 1;
	/*parent may be NULL (top node and proto)*/
	if (parentNode) ChainAddEntry(node->sgprivate->parentNodes, parentNode);
	return M4OK;
}

void Node_ResetChildren(SFNode *node, Chain *list)
{
	while (ChainGetCount(list)) {
		SFNode *p = ChainGetEntry(list, 0);
		ChainDeleteEntry(list, 0);
		Node_Unregister(p, node);
	}
}

/*replace or remove node instance in the given node (eg in all SFNode or MFNode fields)
this doesn't propagate in the scene graph. If updateOrderedGroup and new_node is NULL, the order field of OG
is updated*/
static void ReplaceDEFNode(SFNode *FromNode, u32 NodeID, SFNode *newNode, Bool updateOrderedGroup)
{
	u32 i, j;
	SFNode *p;
	Chain *container;

	FieldInfo field;

	/*browse all fields*/
	for (i=0; i<Node_GetNumFields(FromNode, FCM_ALL); i++) {
		
		Node_GetField(FromNode, i, &field);

		switch (field.fieldType) {
		case FT_SFNode:
			/*set to NULL for SFFields*/
			p = *((SFNode **) field.far_ptr);
			/*this is a USE / DEF*/
			if (p && (Node_GetID(p) == NodeID) ) {
				*((SFNode **) field.far_ptr) = NULL;
				if (newNode) {
					*((SFNode **) field.far_ptr) = newNode;
				}
				goto exit;
			}
			break;
		case FT_MFNode:
			container = *(Chain **) field.far_ptr;
			for (j=0; j<ChainGetCount(container); j++) {
				p = ChainGetEntry(container, j);
				/*replace nodes different from newNode but with same ID*/
				if ((newNode == p) || (Node_GetID(p) != NodeID)) continue;

				ChainDeleteEntry(container, j);
				if (newNode) {
					ChainInsertEntry(container, newNode, j);
				}
#ifdef M4_DEF_OrderedGroup
				else if (updateOrderedGroup && (FromNode->sgprivate->tag==TAG_OrderedGroup)) {
					B_OrderedGroup *og = (B_OrderedGroup *)FromNode;
					MFField_Remove(&og->order, FT_SFInt32, j);
				}
#endif
				goto exit;
			}
			break;
			/*not a node, continue*/
		default:
			continue;
		}
	}
	/*since we don't filter parent nodes this is called once per USE, not per container, so return if found*/
exit:
	SG_NodeChanged(FromNode, &field);
}

/*get all parents of the node and replace, the instance of the node and finally destroy the node*/
M4Err Node_ReplaceAllInstances(SFNode *node, SFNode *new_node, Bool updateOrderedGroup)
{
	u32 i;
	Bool replace_root;
	SFNode *par;
	SceneGraph *pSG = node->sgprivate->scenegraph;

	/*if this is a proto its is registered in its parent graph, not the current*/
	if (node == (SFNode*)pSG->pOwningProto) pSG = pSG->parent_scene;
	if (!SG_SearchForNodeIndex(pSG, node, &i)) return M4BadParam;
	assert(node == pSG->node_registry[i]);

	/*first check if this is the root node*/
	replace_root = (node->sgprivate->scenegraph->RootNode == node) ? 1 : 0;
	/*this shall always be true since we only replace DEF nodes*/
	assert(replace_root || (ChainGetCount(node->sgprivate->parentNodes) == node->sgprivate->num_instances) );

	while ( (i = ChainGetCount(node->sgprivate->parentNodes)) ) {
		par = ChainGetEntry(node->sgprivate->parentNodes, 0);
		ReplaceDEFNode(par, node->sgprivate->NodeID, new_node, updateOrderedGroup);
				
		/*adds the parent to the new node*/
		if (new_node) Node_Register(new_node, par);

		/*unregister node*/
		Node_Unregister(node, par);
		if (i==1) break;
	}

	if (replace_root && new_node) new_node->sgprivate->scenegraph->RootNode = new_node;
	return M4OK;
}

static M4INLINE void insert_node_def(LPSCENEGRAPH sg, SFNode *def)
{
	u32 i, remain;

	if (sg->node_reg_alloc==sg->node_reg_size) {
		sg->node_reg_alloc+=NODEREG_STEP_ALLOC;
		sg->node_registry = realloc(sg->node_registry, sg->node_reg_alloc * sizeof(SFNode *));
	}

	i=0;
	if (sg->node_reg_size) {
		i = node_search(sg, 0, sg->node_reg_size, def->sgprivate->NodeID);
	}
	if (i<sg->node_reg_size) {
		remain = sg->node_reg_size-i;
		memmove(&sg->node_registry[i+1], &sg->node_registry[i], sizeof(SFNode *) * remain);
	}
	sg->node_registry[i] = def;
	sg->node_reg_size++;
}



M4Err Node_SetDEF(SFNode *p, u32 ID, const char *name)
{
	char *new_name;
	u32 i, j;
	SceneGraph *pSG; 
	if (!p || !p->sgprivate->scenegraph) return M4BadParam;

	pSG = p->sgprivate->scenegraph;
	/*if this is a proto register to the parent graph, not the current*/
	if (p == (SFNode*)pSG->pOwningProto) pSG = pSG->parent_scene;

	/*new DEF ID*/
	if (!p->sgprivate->NodeID) {
		p->sgprivate->NodeID = ID;
		if (p->sgprivate->NodeName) free(p->sgprivate->NodeName);
		p->sgprivate->NodeName = NULL;
		if (name) p->sgprivate->NodeName = strdup(name);
		assert(pSG);
		/*just to check things are OK with BIFS creation - note the scene graph supports assigning IDs 
		to existing nodes*/
//		assert(p->sgprivate->num_instances == 0);
		if (ID) insert_node_def(pSG, p);
		return M4OK;
	}

	/*just change DEF name*/
	new_name = name ? strdup(name) : NULL;
	if (p->sgprivate->NodeName) free(p->sgprivate->NodeName);
	p->sgprivate->NodeName = new_name;
	/*same ID, just return*/
	if (p->sgprivate->NodeID == ID) return M4OK;

	/*different ID, remove from node registry and re-insert (needed because node registry is sorted by IDs)*/
	if (!SG_SearchForNodeIndex(pSG, p, &i)) {
		assert(0);
	}
	assert (p == pSG->node_registry[i]);
	j = pSG->node_reg_size - i - 1;
	if (j) memmove( & pSG->node_registry[i], & pSG->node_registry[i+1], j * sizeof(SFNode *));
	pSG->node_reg_size -= 1;
	p->sgprivate->NodeID = ID;
	if (ID) insert_node_def(pSG, p);
	return M4OK;
}

Bool Node_IsDEF(SFNode *node)
{
	if (!node) return 0;
	if (!node->sgprivate->NodeID) return 0;
	if (node->sgprivate->NodeName == NULL) return 0;
	if (!strcmp(node->sgprivate->NodeName, "")) return 0;
	return 1;
}


void Node_Init(SFNode *node)
{
	SceneGraph *pSG = node->sgprivate->scenegraph;
	assert(pSG);

	switch (node->sgprivate->tag) {
#ifdef M4_DEF_ColorInterpolator
	case TAG_ColorInterpolator:
		((B_ColorInterpolator *)node)->on_set_fraction = ColorInt_SetFraction;
		return;
#endif

#ifdef M4_DEF_CoordinateInterpolator 
	case TAG_CoordinateInterpolator:
		((B_CoordinateInterpolator *)node)->on_set_fraction = CoordInt_SetFraction;
		return;
#endif

#ifdef M4_DEF_CoordinateInterpolator2D
	case TAG_CoordinateInterpolator2D:
		((B_CoordinateInterpolator2D *)node)->on_set_fraction = CI2D_SetFraction;
		return;
#endif
	
	/*we need slerp for these two - to do*/
#ifdef M4_DEF_NormalInterpolator
	case TAG_NormalInterpolator:
		((B_NormalInterpolator*)node)->on_set_fraction = NormInt_SetFraction;
		return;
#endif

#ifdef M4_DEF_OrientationInterpolator
	case TAG_OrientationInterpolator:
		((B_OrientationInterpolator*)node)->on_set_fraction = OrientInt_SetFraction;
		return;
#endif

#ifdef M4_DEF_PositionInterpolator
	case TAG_PositionInterpolator:
		((B_PositionInterpolator *)node)->on_set_fraction = PosInt_SetFraction;
		return;
#endif

#ifdef M4_DEF_PositionInterpolator2D
	case TAG_PositionInterpolator2D:
		((B_PositionInterpolator2D *)node)->on_set_fraction = PosInt2D_SetFraction;
		return;
#endif

#ifdef M4_DEF_ScalarInterpolator
	case TAG_ScalarInterpolator:
		((B_ScalarInterpolator *)node)->on_set_fraction = ScalarInt_SetFraction;
		return;
#endif

#ifdef M4_DEF_Valuator
	case TAG_Valuator:
		((B_Valuator *)node)->on_inSFTime = Valuator_SetInSFTime;
		((B_Valuator *)node)->on_inSFBool = Valuator_SetInSFBool;
		((B_Valuator *)node)->on_inSFColor = Valuator_SetInSFColor;
		((B_Valuator *)node)->on_inSFInt32 = Valuator_SetInSFInt32;
		((B_Valuator *)node)->on_inSFFloat = Valuator_SetInSFFloat;
		((B_Valuator *)node)->on_inSFVec2f = Valuator_SetInSFVec2f;
		((B_Valuator *)node)->on_inSFVec3f = Valuator_SetInSFVec3f;
		((B_Valuator *)node)->on_inSFRotation = Valuator_SetInSFRotation;
		((B_Valuator *)node)->on_inSFString = Valuator_SetInSFString;
		((B_Valuator *)node)->on_inMFColor = Valuator_SetInMFColor;
		((B_Valuator *)node)->on_inMFInt32 = Valuator_SetInMFInt32;
		((B_Valuator *)node)->on_inMFFloat = Valuator_SetInMFFloat;
		((B_Valuator *)node)->on_inMFVec2f = Valuator_SetInMFVec2f;
		((B_Valuator *)node)->on_inMFVec3f = Valuator_SetInMFVec3f;
		((B_Valuator *)node)->on_inMFRotation = Valuator_SetInMFRotation;
		((B_Valuator *)node)->on_inMFString = Valuator_SetInMFString;
		return;
#endif

#ifdef M4_DEF_Script
	case TAG_Script:
		return;
#endif

#ifdef M4_DEF_PositionAnimator
#pragma message ("WARNING: PositionAnimator NOT IMPLEMENTED")
	case TAG_PositionAnimator:
		return;
#endif

#ifdef M4_DEF_PositionAnimator2D
#pragma message ("WARNING: PositionAnimator2D NOT IMPLEMENTED")
	case TAG_PositionAnimator2D:
		return;
#endif
	}

	/*user defined init*/
	if (pSG->B_NodeInit) pSG->B_NodeInit(pSG->NodeInitCallback, node);
}


void SG_NodeChanged(SFNode *node, FieldInfo *field)
{
	LPSCENEGRAPH sg;
	if (!node) return;

	sg = node->sgprivate->scenegraph;
	assert(sg);

	/*filter internal ones*/
	switch (node->sgprivate->tag) {
#ifdef M4_DEF_Script
	case TAG_Script: return;
#endif
#ifdef M4_DEF_ColorInterpolator
	case TAG_ColorInterpolator: return;
#endif
#ifdef M4_DEF_CoordinateInterpolator
	case TAG_CoordinateInterpolator: return;
#endif
#ifdef M4_DEF_CoordinateInterpolator2D
	case TAG_CoordinateInterpolator2D: return;
#endif
#ifdef M4_DEF_NormalInterpolator
	case TAG_NormalInterpolator: return;
#endif
#ifdef M4_DEF_OrientationInterpolator
	case TAG_OrientationInterpolator: return;
#endif
#ifdef M4_DEF_PositionInterpolator
	case TAG_PositionInterpolator: return;
#endif
#ifdef M4_DEF_PositionInterpolator2D
	case TAG_PositionInterpolator2D: return;
#endif
#ifdef M4_DEF_ScalarInterpolator
	case TAG_ScalarInterpolator: return;
#endif
#ifdef M4_DEF_Valuator
	case TAG_Valuator: return;
#endif
#ifdef M4_DEF_PositionAnimator
	case TAG_PositionAnimator: return;
#endif
#ifdef M4_DEF_PositionAnimator2D
	case TAG_PositionAnimator2D: return;
#endif
	case TAG_ProtoNode: return;

	}

	/*force child dirty tag*/
	if ((field->fieldType==FT_SFNode) |- (field->fieldType==FT_MFNode)) node->sgprivate->is_dirty |= SG_CHILD_DIRTY;

	if (sg->NodeModified) sg->NodeModified(sg->ModifCallback, node);
}

void QueueRoute(LPSCENEGRAPH sg, Route *r)
{
	u32 now;
	if (!sg) return;

	/*get the top level scene (that's the only reliable one regarding simulatioin tick)*/
	while (sg->parent_scene) sg = sg->parent_scene;
	/*a single route may not be activated more than once in a simulation tick*/
	now = 1 + sg->simulation_tick;
	if (r->lastActivateTime >= now) return;
	r->lastActivateTime = now;
	ChainAddEntry(sg->routes_to_activate, r);
}


/*activate all routes in the order they where triggered*/
void SG_ActivateRoutes(LPSCENEGRAPH sg)
{
	Route *r;
	SFNode *targ;
	if (!sg) return;

	sg->simulation_tick++;

	while (ChainGetCount(sg->routes_to_activate)) {
		r = ChainGetEntry(sg->routes_to_activate, 0);
		ChainDeleteEntry(sg->routes_to_activate, 0);
		if (r) {
			targ = r->ToNode;
			if (ActivateRoute(r))
				if (r->is_setup) SG_NodeChanged(targ, &r->ToField);
		}
	}
	SG_destroy_routes(sg);
}

void RemoveActivatedRoute(LPSCENEGRAPH sg, Route *r)
{
	/*get the top level scene*/
	while (sg->parent_scene) sg = sg->parent_scene;
	/*remove route from queue list*/
	ChainDeleteItem(sg->routes_to_activate, r);
}


/*calls RenderNode on this node*/
void Node_Render(SFNode *node, void *renderStack)
{
	if (!node) return;

	if (node->sgprivate->tag != TAG_ProtoNode) {
		if (node->sgprivate->RenderNode) 
			node->sgprivate->RenderNode(node, renderStack);
		return;
	}

	/*proto only traverses its first child*/
	if (((ProtoInstance *) node)->RenderingNode) {
		node = ((ProtoInstance *) node)->RenderingNode;
	}
	/*if no rendering function is assigned this is a real proto (otherwise this is an hardcoded one)*/
	else if (!node->sgprivate->RenderNode) {
		/*if no rendering node, check if the proto is fully instanciated (externProto)*/
		ProtoInstance *proto_inst = (ProtoInstance *) node;
		if (!proto_inst->proto_interface->ExternProto.count) return;
		/*try to load the code*/
		proto_instanciate(proto_inst->proto_interface, proto_inst);
		if (!proto_inst->RenderingNode) return;
		node = proto_inst->RenderingNode;
		node->sgprivate->scenegraph->NodeModified(node->sgprivate->scenegraph->ModifCallback, node);
	}
	if (node->sgprivate->RenderNode) {
		node->sgprivate->RenderNode(node, renderStack);
	}
	
}

/*blindly calls RenderNode on all nodes in the "children" list*/
void Node_RenderChildren(SFNode *node, void *renderStack)
{
	u32 i;
	SFNode *ptr;
	SFParent *par = (SFParent *)node;
	for (i=0; i<ChainGetCount(par->children); i++) {
		ptr = ChainGetEntry(par->children, i);
		if (ptr) Node_Render(ptr, renderStack);
	}
}

SFNode *SG_NewNode(LPSCENEGRAPH inScene, u32 tag)
{
	SFNode *node;
	if (!inScene) return NULL;
	/*cannot create proto this way*/
	if (tag==TAG_ProtoNode) return NULL;
	else if (tag==TAG_UndefinedNode) {
		node = NewSFNode();
	} else {
		node = CreateNode(tag);
	}
	if (node) node->sgprivate->scenegraph = inScene;

#ifdef M4_DEF_Script
	/*script is inited as soon as created since fields are dynamically added*/
	if (tag==TAG_Script) Script_Init(node);
#endif
	return node;
}


PrototypeNode *SG_FindProto(LPSCENEGRAPH sg, u32 ProtoID, char *name)
{
	PrototypeNode *proto;
	u32 i;

	assert(sg);

	/*browse all top-level */
	for (i=0; i<ChainGetCount(sg->protos); i++) {
		proto = ChainGetEntry(sg->protos, i);
		if (proto->ID == ProtoID) return proto;
		if (name && proto->Name && !stricmp(name, proto->Name)) return proto;
	}
	/*browse all top-level unregistered in reverse order*/
	for (i=ChainGetCount(sg->unregistered_protos); i>0; i--) {
		proto = ChainGetEntry(sg->unregistered_protos, i-1);
		if (proto->ID == ProtoID) return proto;
		if (name && proto->Name && !stricmp(name, proto->Name)) return proto;
	}
	return NULL;
}


u32 Node_GetChildTable(SFNode *Node)
{
	assert(Node);
#ifdef NODE_USE_POINTERS
	return Node->sgprivate->child_ndt;
#else
	return Node_GetChildNDT(Node);
#endif
}

M4Err Node_GetField(SFNode *node, u32 FieldIndex, FieldInfo *info)
{
	assert(node);
	assert(info);
	info->allIndex = FieldIndex;
	info->on_event_in = 0;
#ifdef NODE_USE_POINTERS
	return node->sgprivate->get_field(node, info);
#else
	switch (node->sgprivate->tag) {
#ifdef M4_DEF_Script
	case TAG_Script: return Script_GetField(node, info);
#endif
	case TAG_ProtoNode: return Proto_GetField(NULL, node, info);
	default: return SFNode_GetField(node, info);
	}
#endif

}

u32 Node_GetNumFields(SFNode *Node, u8 IndexMode)
{
	assert(Node);
#ifdef NODE_USE_POINTERS
	return Node->sgprivate->get_field_count(Node, IndexMode);
#else
	if (Node->sgprivate->tag == TAG_ProtoNode) 
		return Proto_GetNumFields(Node, IndexMode);
#ifdef M4_DEF_Script
	else if (Node->sgprivate->tag == TAG_Script) 
		return Script_GetNumFields(Node, IndexMode);
#endif
	else
		return SFNode_GetFieldCount(Node, IndexMode);
#endif
}

const char *Node_GetName(SFNode *Node)
{
	assert(Node);
#ifdef NODE_USE_POINTERS
	return Node->sgprivate->name;
#else
	if (Node->sgprivate->tag == TAG_ProtoNode) 
		return ((ProtoInstance*)Node)->proto_name;

	return GetNodeName(Node->sgprivate->tag);
#endif
}


M4Err Node_GetFieldIndex(SFNode *Node, u32 inField, u8 IndexMode, u32 *allField)
{
	assert(Node);
#ifdef NODE_USE_POINTERS
	return Node->sgprivate->get_field_index(Node, inField, IndexMode, allField);
#else
	switch (Node->sgprivate->tag) {
	case TAG_ProtoNode:
		return protoinst_get_field_ind(Node, inField, IndexMode, allField);
#ifdef M4_DEF_Script
	case TAG_Script:
		return Script_GetFieldIndex(Node, inField, IndexMode, allField);
#endif
	default:
		return SFNode_GetFieldIndex(Node, inField, IndexMode, allField);
	}
#endif
}


u32 Node_FieldIsDefault(SFNode *Node, u32 Index)
{
	return 0;
}


M4Err Node_GetFieldByName(SFNode *node, char *name, FieldInfo *field)
{
	u32 i, count;
	assert(node);
	count = Node_GetNumFields(node, FCM_ALL);
	
	memset(field, 0, sizeof(FieldInfo));
	for (i=0; i<count;i++) {
		Node_GetField(node, i, field);
		if (!stricmp(field->name, name)) return M4OK;
	}
	return M4BadParam;
}

LPSCENEGRAPH Node_GetParentGraph(SFNode *node)
{
	return (node ? node->sgprivate->scenegraph : NULL);
}


/* QUANTIZATION AND BIFS_Anim Info */
Bool Node_GetAQInfo(SFNode *Node, u32 FieldIndex, u8 *QType, u8 *AType, Float *b_min, Float *b_max, u32 *QT13_bits)
{
	assert(Node);

#ifdef NODE_USE_POINTERS
	return Node->sgprivate->get_aq_info(Node, FieldIndex, QType, AType, b_min, b_max, QT13_bits);
#else
	switch (Node->sgprivate->tag) {
	case TAG_ProtoNode:
		return Proto_GetAQInfo(Node, FieldIndex, QType, AType, b_min, b_max, QT13_bits);
	default:
		return SFNode_GetAQInfo(Node, FieldIndex, QType, AType, b_min, b_max, QT13_bits);
	}
#endif
}

SFNode *SG_FindNode(LPSCENEGRAPH sg, u32 nodeID)
{
	SFNode *node;
	assert(sg);
	node = SG_SearchForNodeByID(sg, nodeID);
	return node;
}

SFNode *SG_FindNodeByName(LPSCENEGRAPH sg, char *name)
{
	u32 i;
	assert(sg);
	for (i=0; i<sg->node_reg_size; i++) {
		if (!sg->node_registry[i]->sgprivate->NodeName) continue;
		if (!strcmp(sg->node_registry[i]->sgprivate->NodeName, name)) {
			return sg->node_registry[i];
		}
	}
	return NULL;
}

M4Err SG_DeleteAllProtos(LPSCENEGRAPH scene)
{
	if (!scene) return M4BadParam;
	while (ChainGetCount(scene->protos)) {
		PrototypeNode *p = ChainGetEntry(scene->protos, 0);
		SG_DeleteProto(p);
	}
	return M4OK;
}

void SG_SetProtoLoader(LPSCENEGRAPH scene, LPSCENEGRAPH (*GetExternProtoLib)(void *SceneCallback, MFURL *lib_url))
{
	if (!scene) return;
	scene->GetExternProtoLib = GetExternProtoLib;
}


M4Err Node_ModeFieldIndex(SFNode *node, u32 all_ind, u8 indexMode, u32 *outField)
{
	M4Err e;
	u32 i, count, temp;
	count = Node_GetNumFields(node, indexMode);
	for (i=0; i<count; i++) {
		e = Node_GetFieldIndex(node, i, indexMode, &temp);
		if (e) return e;
		if (temp==all_ind) {
			*outField = i;
			return M4OK;
		}
	}
	return M4BadParam;
}

void Node_ForceResetChildren(SFNode *node, Chain *list)
{
	while (ChainGetCount(list)) {
		SFNode *p = ChainGetEntry(list, 0);
		ChainDeleteEntry(list, 0);
		Node_Unregister(p, node);
	}
}

u32 SG_GetNextNodeID(LPSCENEGRAPH sg) 
{
	if (sg->node_reg_size == 0) return 1;
	/*nodes are sorted*/
	return sg->node_registry[sg->node_reg_size-1]->sgprivate->NodeID+1;
}

void Script_Load(SFNode *n)
{
	if (n->sgprivate->scenegraph->Script_Load) n->sgprivate->scenegraph->Script_Load(n);
}

u32 Node_GetParentCount(SFNode *node)
{
	return ChainGetCount(node->sgprivate->parentNodes);
}

SFNode *Node_GetParent(SFNode *node, u32 idx)
{
	return ChainGetEntry(node->sgprivate->parentNodes, idx);
}


static void dirty_parents(SFNode *node)
{
	u32 i, count;
	count = ChainGetCount(node->sgprivate->parentNodes);
	for (i=0; i<count; i++) {
		SFNode *p = ChainGetEntry(node->sgprivate->parentNodes, i);
		if (p->sgprivate->is_dirty & SG_CHILD_DIRTY) continue;
		p->sgprivate->is_dirty |= SG_CHILD_DIRTY;
		dirty_parents(p);
	}
}

void Node_SetDirty(SFNode *node, Bool invalidate_parents)
{
	if (!node) return;
	node->sgprivate->is_dirty |= SG_NODE_DIRTY;
	if (invalidate_parents) dirty_parents(node);
}

void Node_ClearDirty(SFNode *node)
{
	if (node) node->sgprivate->is_dirty = 0;
}

u16 Node_GetDirty(SFNode *node)
{
	if (node) return node->sgprivate->is_dirty;
	return 0;
}

static void Node_DirtyChildren(SFNode *node, u16 val)
{
	u32 i, count;
	FieldInfo info;
	if (!node) return;
	
	node->sgprivate->is_dirty = val;
	count = Node_GetNumFields(node, FCM_ALL);
	for (i=0; i<count; i++) {
		Node_GetField(node, i, &info);
		if (info.fieldType==FT_SFNode) Node_DirtyChildren(*(SFNode **)info.far_ptr, val);
		else if (info.fieldType==FT_MFNode) {
			Chain *list = *(Chain **) info.far_ptr;
			u32 j, n;
			n = ChainGetCount(list);
			for (j=0; j<n; j++) 
				Node_DirtyChildren((SFNode *)ChainGetEntry(list, j), val);
		}
	}
}

void Node_SetDirtyChildren(SFNode *node)
{
	Node_DirtyChildren(node, SG_NODE_DIRTY | SG_CHILD_DIRTY);
}

void Node_ClearDirtyChildren(SFNode *node, Bool skip_if_clean)
{
	if (skip_if_clean && node && !node->sgprivate->is_dirty) return;
	Node_DirtyChildren(node, 0);
}

void Node_SetDirtyFlag(SFNode *node, u16 val)
{
	if (node) node->sgprivate->is_dirty = val;
}

/*for conditionals*/
#include <gpac/m4_bifs.h>

/*very similar to proto cloning code, with the exception that DEF/USE are all ignored (node duplicated)*/
SFNode *NodeCloneNoDEF(LPSCENEGRAPH inScene, SFNode *orig, SFNode *cloned_parent)
{
	u32 i, j, count;
	SFNode *node, *child, *tmp;
	Chain *list, *list2;
	Route *r1, *r2;
	ProtoInstance *proto;
	PrototypeNode *proto_node;
	FieldInfo field_orig, field;

	/*this is not a mistake*/
	if (!orig) return NULL;

	/*create a node*/
	if (orig->sgprivate->tag == TAG_ProtoNode) {
		proto_node = ((ProtoInstance *)orig)->proto_interface;
		/*create the instance but don't load the code -c we MUST wait for ISed routes to be cloned before*/
		node = Proto_CreateNode(inScene, proto_node, (ProtoInstance *) orig);
	} else {
		node = SG_NewNode(inScene, orig->sgprivate->tag);
	}

	count = Node_GetNumFields(orig, FCM_ALL);

#ifdef M4_DEF_Script
	if (orig->sgprivate->tag==TAG_Script) Script_PrepareClone(node, orig);
#endif

	/*copy each field*/
	for (i=0; i<count; i++) {
		Node_GetField(orig, i, &field_orig);

		/*get target ptr*/
		Node_GetField(node, i, &field);

		assert(field.eventType==field_orig.eventType);
		assert(field.fieldType==field_orig.fieldType);

		/*duplicate it*/
		switch (field.fieldType) {
		case FT_SFNode:
			child = NodeCloneNoDEF(inScene, (void *) (* ((SFNode **) field_orig.far_ptr)), node);
			*((SFNode **) field.far_ptr) = child;
			break;
		case FT_MFNode:
			list = *( (Chain **) field_orig.far_ptr);
			list2 = *( (Chain **) field.far_ptr);

			for (j=0; j<ChainGetCount(list); j++) {
				tmp = ChainGetEntry(list, j);
				child = NodeCloneNoDEF(inScene, tmp, node);
				ChainAddEntry(list2, child);
			}
			break;
		case FT_SFTime:
			SG_CopyField(field.far_ptr, field_orig.far_ptr, field.fieldType);
			if (!inScene->GetSceneTime) break;
			/*update SFTime that must be updated when cloning the node*/
			if (orig->sgprivate->tag == TAG_ProtoNode) {
				if (Proto_FieldIsSFTimeOffset(orig, &field_orig))
					*((SFTime *)field.far_ptr) += inScene->GetSceneTime(inScene->SceneCallback);
			} else if (!stricmp(field_orig.name, "startTime") || !stricmp(field_orig.name, "startTime") ) {
				*((SFTime *)field.far_ptr) += inScene->GetSceneTime(inScene->SceneCallback);
			}
			break;
		default:
			SG_CopyField(field.far_ptr, field_orig.far_ptr, field.fieldType);
			break;
		}
	}

	if (cloned_parent) Node_Register(node, cloned_parent);

	/*init node before creating ISed routes so the eventIn handler are in place*/
#ifdef M4_DEF_Conditional
	if (node->sgprivate->tag == TAG_Conditional) BIFS_SetupConditionalClone(node, orig);
	else 
#endif
		if (node->sgprivate->tag != TAG_ProtoNode) Node_Init(node);


	proto = inScene->pOwningProto;
	if (proto) {
		/*create Routes for ISed fields*/
		for (i=0; i<ChainGetCount(proto->proto_interface->sub_graph->Routes); i++) {
			r1 = ChainGetEntry(proto->proto_interface->sub_graph->Routes, i);
			r2 = NULL;
			/*locate only ISed routes*/
			if (!r1->IS_route) continue;
			
			/*eventOut*/
			if (r1->FromNode == orig) {
				r2 = SG_NewRoute(inScene, node, r1->FromFieldIndex, (SFNode *) proto, r1->ToFieldIndex);
				r2->IS_route = 1;
			}
			/*eventIn or exposedField*/
			else if (r1->ToNode == orig) {
				r2 = SG_NewRoute(inScene, (SFNode *) proto, r1->FromFieldIndex, node, r1->ToFieldIndex);
				r2->IS_route = 1;
			}
			/*activate the route now so that proto instanciation works properly, otherwise
			we may load scripts with wrong field values*/		
			if (r2) ActivateRoute(r2);
		}
	}
	
#ifdef M4_DEF_Script
	/*remember scripts*/
	if (node->sgprivate->tag == TAG_Script) ChainAddEntry(proto->scripts_to_load, node);
#endif

	/*this is a proto node, init our internal stuff*/
	if (node->sgprivate->tag == TAG_ProtoNode) {
		node->sgprivate->PreDestroyNode = NULL;
		node->sgprivate->privateStack = NULL;
		node->sgprivate->RenderNode = RenderProtoInstance;
		/*load code*/
		proto_instanciate(((ProtoInstance *)node)->proto_interface, (ProtoInstance *)node);
	}
	return node;
}

SFNode *Node_Clone(LPSCENEGRAPH inScene, SFNode *orig)
{
	return NodeCloneNoDEF(inScene, orig, NULL);
}
