/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_scenegraph_dev.h>
#include <gpac/intern/m4_node_tables.h>

void Node_Setup(SFNode *p, u32 tag)
{
	SAFEALLOC(p->sgprivate, sizeof(NodePriv));
	p->sgprivate->tag = tag;
	p->sgprivate->is_dirty = SG_NODE_DIRTY;
	p->sgprivate->outRoutes = NewChain();
	p->sgprivate->parentNodes = NewChain();
}

NodePriv *Node_GetPriv(SFNode *node)
{
	return node->sgprivate;
}

SFNode *NewSFNode()
{
	SFNode *newnode = malloc(sizeof(SFNode));
	Node_Setup(newnode, TAG_UndefinedNode);
	return newnode;
}

u32 Node_GetTag(SFNode*p)
{
	return p->sgprivate->tag;
}

u32 Node_GetID(SFNode*p)
{
	return p->sgprivate->NodeID;
}

const char *Node_GetDefName(SFNode*p)
{
	return p->sgprivate->NodeName;
}


void *Node_GetPrivate(SFNode*p)
{
	return p->sgprivate->privateStack;
}
void Node_SetPrivate(SFNode*p, void *pr)
{
	p->sgprivate->privateStack = pr;
}

static u32 IsInternal(SFNode *p)
{
	switch (p->sgprivate->tag) {
#ifdef M4_DEF_ColorInterpolator
	case TAG_ColorInterpolator: return 1;
#endif
#ifdef M4_DEF_CoordinateInterpolator
	case TAG_CoordinateInterpolator: return 1;
#endif
#ifdef M4_DEF_CoordinateInterpolator2D
	case TAG_CoordinateInterpolator2D: return 1;
#endif
#ifdef M4_DEF_NormalInterpolator
	case TAG_NormalInterpolator: return 1;
#endif
#ifdef M4_DEF_OrientationInterpolator
	case TAG_OrientationInterpolator: return 1;
#endif
#ifdef M4_DEF_PositionInterpolator
	case TAG_PositionInterpolator: return 1;
#endif
#ifdef M4_DEF_PositionInterpolator2D
	case TAG_PositionInterpolator2D: return 1;
#endif
#ifdef M4_DEF_QuantizationParameter
	case TAG_QuantizationParameter: return 1;
#endif
#ifdef M4_DEF_ScalarInterpolator
	case TAG_ScalarInterpolator: return 1;
#endif
#ifdef M4_DEF_Script
	case TAG_Script: return 1;
#endif
#ifdef M4_DEF_Valuator
	case TAG_Valuator: return 1;
#endif
	default: return 0;
	}
}

M4Err Node_SetRenderFunction(SFNode *p, void (*RenderNode)(SFNode *node, void *render_stack) )
{
	if (IsInternal(p)) return M4BadParam;
	p->sgprivate->RenderNode = RenderNode;
	return M4OK;
}

M4Err Node_SetPreDestroyFunction(SFNode *p, void (*PreDestroyNode)(struct _sfNode *node) )
{
	if (IsInternal(p)) return M4BadParam;
	p->sgprivate->PreDestroyNode = PreDestroyNode;
	return M4OK;
}


//parent base node used for type casting only
typedef struct
{
	BASE_NODE
	CHILDREN
} ParentNode;


SFBool *NewSFBool()
{
	SFBool *tmp = malloc(sizeof(SFBool));
	memset(tmp, 0, sizeof(SFBool));
	return tmp;
}
SFFloat *NewSFFloat()
{
	SFFloat *tmp = malloc(sizeof(SFFloat));
	memset(tmp, 0, sizeof(SFFloat));
	return tmp;
}
SFTime *NewSFTime()
{
	SFTime *tmp = malloc(sizeof(SFTime));
	memset(tmp, 0, sizeof(SFTime));
	return tmp;
}
SFInt32 *NewSFInt32()
{
	SFInt32 *tmp = malloc(sizeof(SFInt32));
	memset(tmp, 0, sizeof(SFInt32));
	return tmp;
}
SFString *NewSFString()
{
	SFString *tmp = malloc(sizeof(SFString));
	memset(tmp, 0, sizeof(SFString));
	return tmp;
}
SFVec3f *NewSFVec3f()
{
	SFVec3f *tmp = malloc(sizeof(SFVec3f));
	memset(tmp, 0, sizeof(SFVec3f));
	return tmp;
}
SFVec2f *NewSFVec2f()
{
	SFVec2f *tmp = malloc(sizeof(SFVec2f));
	memset(tmp, 0, sizeof(SFVec2f));
	return tmp;
}
SFColor *NewSFColor()
{
	SFColor *tmp = malloc(sizeof(SFColor));
	memset(tmp, 0, sizeof(SFColor));
	return tmp;
}
SFRotation *NewSFRotation()
{
	SFRotation *tmp = malloc(sizeof(SFRotation));
	memset(tmp, 0, sizeof(SFRotation));
	return tmp;
}
SFImage *NewSFImage()
{
	SFImage *tmp = malloc(sizeof(SFImage));
	memset(tmp, 0, sizeof(SFImage));
	return tmp;
}
SFURL *NewSFURL()
{
	SFURL *tmp = malloc(sizeof(SFURL));
	memset(tmp, 0, sizeof(SFURL));
	return tmp;
}
SFCommandBuffer *NewSFCommandBuffer()
{
	SFCommandBuffer *tmp = malloc(sizeof(SFCommandBuffer));
	memset(tmp, 0, sizeof(SFCommandBuffer));
	tmp->commandList = NewChain();
	return tmp;
}

MFBool *NewMFBool()
{
	MFBool *tmp = malloc(sizeof(MFBool));
	memset(tmp, 0, sizeof(MFBool));
	return tmp;
}
MFFloat *NewMFFloat()
{
	MFFloat *tmp = malloc(sizeof(MFFloat));
	memset(tmp, 0, sizeof(MFFloat));
	return tmp;
}
MFTime *NewMFTime()
{
	MFTime *tmp = malloc(sizeof(MFTime));
	memset(tmp, 0, sizeof(MFTime));
	return tmp;
}
MFInt32 *NewMFInt32()
{
	MFInt32 *tmp = malloc(sizeof(MFInt32));
	memset(tmp, 0, sizeof(MFInt32));
	return tmp;
}
MFString *NewMFString()
{
	MFString *tmp = malloc(sizeof(MFString));
	memset(tmp, 0, sizeof(MFString));
	return tmp;
}
MFVec3f *NewMFVec3f()
{
	MFVec3f *tmp = malloc(sizeof(MFVec3f));
	memset(tmp, 0, sizeof(MFVec3f));
	return tmp;
}
MFVec2f *NewMFVec2f()
{
	MFVec2f *tmp = malloc(sizeof(MFVec2f));
	memset(tmp, 0, sizeof(MFVec2f));
	return tmp;
}
MFColor *NewMFColor()
{
	MFColor *tmp = malloc(sizeof(MFColor));
	memset(tmp, 0, sizeof(MFColor));
	return tmp;
}
MFRotation *NewMFRotation()
{
	MFRotation *tmp = malloc(sizeof(MFRotation));
	memset(tmp, 0, sizeof(MFRotation));
	return tmp;
}
MFURL *NewMFURL()
{
	MFURL *tmp = malloc(sizeof(MFURL));
	memset(tmp, 0, sizeof(MFURL));
	return tmp;
}

void *SG_NewFieldPointer(u32 FieldType) 
{
	switch (FieldType) {
	case FT_SFBool: return NewSFBool();
	case FT_SFFloat: return NewSFFloat();
	case FT_SFTime: return NewSFTime();
	case FT_SFInt32: return NewSFInt32();
	case FT_SFString: return NewSFString();
	case FT_SFVec3f: return NewSFVec3f();
	case FT_SFVec2f: return NewSFVec2f();
	case FT_SFColor: return NewSFColor();
	case FT_SFRotation: return NewSFRotation();
	case FT_SFImage: return NewSFImage();
	case FT_MFBool: return NewMFBool();
	case FT_MFFloat: return NewMFFloat();
	case FT_MFTime: return NewMFTime();
	case FT_MFInt32: return NewMFInt32();
	case FT_MFString: return NewMFString();
	case FT_MFVec3f: return NewMFVec3f();
	case FT_MFVec2f: return NewMFVec2f();
	case FT_MFColor: return NewMFColor();
	case FT_MFRotation: return NewMFRotation();

	//used in proto and script 
	case FT_MFNode: 
	{
		return NewChain();
	}
	//used in commands
	case FT_SFCommandBuffer:
		return NewSFCommandBuffer();

	case FT_SFURL: 
		return NewSFURL();
	case FT_MFURL:
		return NewMFURL();
	}
	return NULL;
}

void MFInt32_Del(MFInt32 par)
{
	free(par.vals);
}

void MFInt_Del(MFInt par)
{
	free(par.vals);
}

void MFFloat_Del(MFFloat par)
{
	free(par.vals);
}

void MFBool_Del(MFBool par)
{
	free(par.vals);
}

void MFColor_Del(MFColor par)
{
	free(par.vals);
}

void MFRotation_Del(MFRotation par)
{
	free(par.vals);
}

void MFString_Del(MFString par)
{
	u32 i;
	for (i=0; i<par.count; i++) {
		if (par.vals[i]) free(par.vals[i]);
	}
	free(par.vals);
}

void MFTime_Del(MFTime par)
{
	free(par.vals);
}

void MFVec2f_Del(MFVec2f par)
{
	free(par.vals);
}

void MFVec3f_Del(MFVec3f par)
{
	free(par.vals);
}

void SFString_Del(SFString par)
{
	if (par.buffer) free(par.buffer);
}
void SFImage_Del(SFImage im)
{
	free(im.pixels);
}

void SFScript_Del(SFScript sc)
{
	if (sc.script_text) free(sc.script_text);
}

void SFCommandBuffer_Del(SFCommandBuffer cb)
{
	u32 i;
	for (i=ChainGetCount(cb.commandList); i>0; i--) {
		SGCommand *com = ChainGetEntry(cb.commandList, i-1);
		SG_DeleteCommand(com);
	}
	DeleteChain(cb.commandList);
	if (cb.buffer) free(cb.buffer);
}

void SFURL_Del(SFURL url)
{
	if (url.url) free(url.url);
}

void MFURL_Del(MFURL url)
{
	u32 i;
	for (i=0; i<url.count; i++) {
		SFURL_Del(url.vals[i]);
	}
	free(url.vals);
}
void MFScript_Del(MFScript sc)
{
	u32 i;
	for (i=0; i<sc.count; i++) {
		if (sc.vals[i].script_text) free(sc.vals[i].script_text);
	}
	free(sc.vals);
}


static void Node_on_add_children(SFNode *node)
{
	SFNode *child, *child_in;
	u32 i;
	FieldInfo field;
	SFParent *n = (SFParent *)node;

	
	Node_GetField(node, 2, &field);

	/*for each node in input*/
	while (ChainGetCount(n->addChildren)) {
		child = ChainGetEntry(n->addChildren, 0);
		ChainDeleteEntry(n->addChildren, 0);
		/*this is safe because the node is already registered (add_children)*/
		Node_Unregister(child, node);

		/*check the node is not in children list*/
		child_in = NULL;
		for (i=0; i<ChainGetCount(n->children); i++) {
			child_in = ChainGetEntry(n->children, i);
			if (child_in == child) break;
			child_in = NULL;
		}
		/*already in*/
		if (child_in) continue;
		
		/*OK add the node*/
		Node_Register(child, node);

		/*and add node*/
		ChainAddEntry(n->children, child);
	}
	/*signal children field is modified*/
	SG_NodeChanged(node, &field);
}

static void Node_on_remove_children(SFNode *node)
{
	SFNode *child, *child_in;
	FieldInfo field;
	u32 i;
	SFParent *n = (SFParent *)node;
	Node_GetField(node, 2, &field);

	/*for each node in input*/
	while (ChainGetCount(n->removeChildren)) {
		child = ChainGetEntry(n->removeChildren, 0);
		ChainDeleteEntry(n->removeChildren, 0);
		/*this is safe because the node is already registered (remove_children)*/
		Node_Unregister(child, node);

		/*check the node is in children list*/
		child_in = NULL;
		for (i=0; i<ChainGetCount(n->children); i++) {
			child_in = ChainGetEntry(n->children, i);
			if (child_in == child) break;
			child_in = NULL;
		}
		/*not in*/
		if (!child_in) continue;
		/*remove from parent*/
		ChainDeleteEntry(n->children, i);
		/*unregister the node (same handling if implicit or explicit DEF/USE*/
		Node_Unregister(child, node);
	}
	/*signal children field is modified*/
	SG_NodeChanged(node, &field);
}

void SetupChildrenNode(SFNode *pNode)
{
	ParentNode *par = (ParentNode *)pNode;
	par->children = NewChain();
	par->addChildren = NewChain();
	par->on_addChildren = Node_on_add_children;
	par->removeChildren = NewChain();
	par->on_removeChildren = Node_on_remove_children;
	pNode->sgprivate->is_dirty |= SG_CHILD_DIRTY;
}

void DestroyChildrenNode(SFNode *pNode)
{
	ParentNode *par = (ParentNode *)pNode;
	NodeList_Delete(par->children, pNode);
	NodeList_Delete(par->addChildren, pNode);
	NodeList_Delete(par->removeChildren, pNode);
}

void NodeList_Delete(Chain *children, SFNode *parent)
{
	Node_ResetChildren(parent, children);
	DeleteChain(children);
}

void Node_Delete(SFNode *node, SFNode *parent)
{
	if (node) Node_Unregister(node, parent);
}



void DestroyNode(SFNode *node)
{	
	if (node->sgprivate->tag == TAG_ProtoNode) {
		Proto_DeleteInstance((ProtoInstance *)node);
	} else if (node->sgprivate->tag == TAG_UndefinedNode) {
		SFNode_Delete(node);
	} else {
#ifdef NODE_USE_POINTERS
		node->sgprivate->node_del(node);
#else
		Node_Del(node);
#endif
	}	
}


void SFNode_Delete(SFNode *node)
{
	LPROUTE r;
	if (!node) return;
	if (node->sgprivate->NodeName) free(node->sgprivate->NodeName);
	//disconnect from routes
	while (ChainGetCount(node->sgprivate->outRoutes)) {
		r = ChainGetEntry(node->sgprivate->outRoutes, 0);
		ChainDeleteEntry(node->sgprivate->outRoutes, 0);
		r->FromNode = NULL;
	}
	DeleteChain(node->sgprivate->outRoutes);
	node->sgprivate->outRoutes = NULL;

	//destroy user defined stuff
	if (node->sgprivate->PreDestroyNode) {
		node->sgprivate->PreDestroyNode(node);
	}

	assert(! ChainGetCount(node->sgprivate->parentNodes));

	DeleteChain(node->sgprivate->parentNodes);
	free(node->sgprivate);
	free(node);
}


void SG_DeleteFieldPointer(void *field, u32 FieldType) 
{
	SFNode *node;

	switch (FieldType) {
	case FT_SFBool: 
	case FT_SFFloat:
	case FT_SFTime: 
	case FT_SFInt32:
	case FT_SFVec3f:
	case FT_SFVec2f:
	case FT_SFColor:
	case FT_SFRotation:
		break;
	case FT_SFString:
		if ( ((SFString *)field)->buffer) free(((SFString *)field)->buffer);
		break;
	case FT_SFImage:
		SFImage_Del(* ((SFImage *)field));
		break;

	case FT_SFNode: 
		node = *(SFNode **) field;
		if (node) DestroyNode(node);
		return;
	case FT_SFCommandBuffer:
		SFCommandBuffer_Del(*(SFCommandBuffer *)field);
		break;
	
	case FT_MFBool:
		MFBool_Del( * ((MFBool *) field));
		break;
	case FT_MFFloat: 
		MFFloat_Del( * ((MFFloat *) field));
		break;
	case FT_MFTime: 
		MFTime_Del( * ((MFTime *)field));
		break;
	case FT_MFInt32:
		MFInt32_Del( * ((MFInt32 *)field));
		break;
	case FT_MFString:
		MFString_Del( *((MFString *)field));
		break;
	case FT_MFVec3f:
		MFVec3f_Del( * ((MFVec3f *)field));
		break;
	case FT_MFVec2f:
		MFVec2f_Del( * ((MFVec2f *)field));
		break;
	case FT_MFColor:
		MFColor_Del( * ((MFColor *)field));
		break;
	case FT_MFRotation:
		MFRotation_Del( * ((MFRotation *)field));
		break;
	case FT_MFURL:
		MFURL_Del( * ((MFURL *) field));
		break;		
	//used only in proto since this field is created by default for regular nodes
	case FT_MFNode: 
		while (ChainGetCount((Chain *)field)) {
			node = ChainGetEntry((Chain *)field, 0);
			DestroyNode(node);
			ChainDeleteEntry((Chain *)field, 0);
		}
		DeleteChain((Chain *)field);
		return;

	default:
		assert(0);
		return;
	}
	//free pointer
	free(field);
}



Bool SFNode_IsInNDT(u32 NDT, u32 NodeTag)
{
	u32 BIFSV, NodeType;

	BIFSV = 1;

	while (BIFSV <= NUM_BIFS_VERSION) {
		NodeType = NDT_GetNodeType(NDT, NodeTag, BIFSV);
		//this is a bad child
		if (NodeType) return 1;
		BIFSV += 1;
	}
	return 0;
}

//checks and adds a child in the children list
M4Err Node_InsertChild(SFNode *parent, SFNode *new_child, s32 Position)
{
	u32 NDTType, tag;
	ProtoInstance *proto;
	ParentNode *node = (ParentNode *) parent;

	//get the NDT of the parent
	NDTType = Node_GetChildTable(parent);
	//this node CANNOT be a parent - TO DO : Check for protos
	if (!NDTType) return M4InvalidNode;

	tag = new_child->sgprivate->tag;
	//this is a proto, check that the rendering node is of the right type
	if (tag==TAG_ProtoNode) {
		proto = (ProtoInstance *) new_child;
		if (!proto->RenderingNode) return M4InvalidNode;
		tag = proto->RenderingNode->sgprivate->tag;
	}
	//check this is a good children
	if (!SFNode_IsInNDT(NDTType, tag)) return M4InvalidNode;

	if (Position == -1) {
		return ChainAddEntry(node->children, new_child);
	} else {
		return ChainInsertEntry(node->children, new_child, Position);
	}
}

/*UGLY and only used in V4Studio...*/
M4Err Node_RemoveChild(SFNode *parent, SFNode *toremove_child) 
{
	s32 ind;
	ParentNode *node = (ParentNode *) parent;

	ind = ChainDeleteItem(node->children, toremove_child);
	if (ind<0) return M4BadParam;
	/*V4Studio doesn't handle DEF/USE properly yet...*/
	/*Node_Unregister(toremove_child, parent);*/
	return M4OK;
}

Bool SG_IsSFField(u32 FieldType)
{
	switch (FieldType) {
	case FT_SFBool:
	case FT_SFFloat:
	case FT_SFTime:
	case FT_SFInt32:
	case FT_SFString:
	case FT_SFVec3f:
	case FT_SFVec2f:
	case FT_SFColor:
	case FT_SFRotation:
	case FT_SFImage:
	case FT_SFNode:
	case FT_SFURL:
	case FT_SFCommandBuffer:
		return 1;
	default:
		return 0;
	}
}

/*********************************************************************
		MF Fields manipulation (alloc, realloc, GetAt)
*********************************************************************/

//return the size of fixed fields (eg no buffer in the field)
u32 GetSFFieldSize(u32 FieldType)
{
	switch (FieldType) {
	case FT_SFBool:
	case FT_MFBool:
		return sizeof(SFBool);
	case FT_SFFloat:
	case FT_MFFloat:
		return sizeof(SFFloat);
	case FT_SFTime:
	case FT_MFTime:
		return sizeof(SFTime);
	case FT_SFInt32:
	case FT_MFInt32:
		return sizeof(SFInt32);
	case FT_SFVec3f:
	case FT_MFVec3f:
		return 3*sizeof(SFFloat);
	case FT_SFVec2f:
	case FT_MFVec2f:
		return 2*sizeof(SFFloat);
	case FT_SFColor:
	case FT_MFColor:
		return 3*sizeof(SFFloat);
	case FT_SFRotation:
	case FT_MFRotation:
		return 4*sizeof(SFFloat);

	//check if that works!!
	case FT_SFString:
	case FT_MFString:
		//ptr to char
		return sizeof(SFString);
	case FT_SFScript:
	case FT_MFScript:
		return sizeof(MFScript);
	case FT_SFURL:
	case FT_MFURL:
		return sizeof(MFURL);
	default:
		return 0;
	}
}

u32 SG_GetSFType(u32 FieldType)
{
	switch (FieldType) {
	case FT_SFBool:
	case FT_MFBool:
		return FT_SFBool;
	case FT_SFFloat:
	case FT_MFFloat:
		return FT_SFFloat;
	case FT_SFTime:
	case FT_MFTime:
		return FT_SFTime;
	case FT_SFInt32:
	case FT_MFInt32:
		return FT_SFInt32;
	case FT_SFVec3f:
	case FT_MFVec3f:
		return FT_SFVec3f;
	case FT_SFVec2f:
	case FT_MFVec2f:
		return FT_SFVec2f;
	case FT_SFColor:
	case FT_MFColor:
		return FT_SFColor;
	case FT_SFRotation:
	case FT_MFRotation:
		return FT_SFRotation;

	//check if that works!!
	case FT_SFString:
	case FT_MFString:
		//ptr to char
		return FT_SFString;
	case FT_SFScript:
	case FT_MFScript:
		return FT_SFScript;
	case FT_SFURL:
	case FT_MFURL:
		return FT_SFURL;
	case FT_SFNode:
	case FT_MFNode:
		return FT_SFNode;
	default:
		return FT_Unknown;
	}
}

const char *GetEventTypeName(u32 EventType)
{
	switch (EventType) {
	case ET_EventIn: return "eventIn";
	case ET_Field: return "field";
	case ET_ExposedField: return "exposedField";
	case ET_EventOut: return "eventOut";
	default: return "unknownEvent";
	}
}

const char *GetFieldTypeName(u32 FieldType)
{

	switch (FieldType) {
	case FT_SFBool: return "SFBool";
	case FT_SFFloat: return "SFFloat";
	case FT_SFTime: return "SFTime";
	case FT_SFInt32: return "SFInt32";
	case FT_SFString: return "SFString";
	case FT_SFVec3f: return "SFVec3f";
	case FT_SFVec2f: return "SFVec2f";
	case FT_SFColor: return "SFColor";
	case FT_SFRotation: return "SFRotation";
	case FT_SFImage: return "SFImage";
	case FT_SFNode: return "SFNode";
	case FT_SFVec4f: return "SFVec4f";
	case FT_MFBool: return "MFBool";
	case FT_MFFloat: return "MFFloat";
	case FT_MFTime: return "MFTime";
	case FT_MFInt32: return "MFInt32";
	case FT_MFString: return "MFString";
	case FT_MFVec3f: return "MFVec3f";
	case FT_MFVec2f: return "MFVec2f";
	case FT_MFColor: return "MFColor";
	case FT_MFRotation: return "MFRotation";
	case FT_MFImage: return "MFImage";
	case FT_MFNode: return "MFNode";
	case FT_MFVec4f: return "MFVec4f";
	case FT_SFURL: return "SFURL";
	case FT_MFURL: return "MFURL";
	case FT_SFCommandBuffer: return "SFCommandBuffer";
	case FT_SFScript: return "SFScript";
	case FT_MFScript: return "MFScript";
	default: return "UnknownType";
	}
}

u32 GetFieldTypeByName(char *fieldType)
{
	if (!stricmp(fieldType, "SFBool")) return FT_SFBool;
	else if (!stricmp(fieldType, "SFFloat")) return FT_SFFloat;
	else if (!stricmp(fieldType, "SFTime")) return FT_SFTime;
	else if (!stricmp(fieldType, "SFInt32")) return FT_SFInt32;
	else if (!stricmp(fieldType, "SFString")) return FT_SFString;
	else if (!stricmp(fieldType, "SFVec2f")) return FT_SFVec2f;
	else if (!stricmp(fieldType, "SFVec3f")) return FT_SFVec3f;
	else if (!stricmp(fieldType, "SFColor")) return FT_SFColor;
	else if (!stricmp(fieldType, "SFRotation")) return FT_SFRotation;
	else if (!stricmp(fieldType, "SFImage")) return FT_SFImage;
	else if (!stricmp(fieldType, "SFNode")) return FT_SFNode;

	else if (!stricmp(fieldType, "MFBool")) return FT_MFBool;
	else if (!stricmp(fieldType, "MFFloat")) return FT_MFFloat;
	else if (!stricmp(fieldType, "MFTime")) return FT_MFTime;
	else if (!stricmp(fieldType, "MFInt32")) return FT_MFInt32;
	else if (!stricmp(fieldType, "MFString")) return FT_MFString;
	else if (!stricmp(fieldType, "MFVec2f")) return FT_MFVec2f;
	else if (!stricmp(fieldType, "MFVec3f")) return FT_MFVec3f;
	else if (!stricmp(fieldType, "MFColor")) return FT_MFColor;
	else if (!stricmp(fieldType, "MFRotation")) return FT_MFRotation;
	else if (!stricmp(fieldType, "MFImage")) return FT_MFImage;
	else if (!stricmp(fieldType, "MFNode")) return FT_MFNode;

	return FT_Unknown;
}

//
//	Insert (+alloc) an MFField with a specified position for insertion and sets the ptr to the 
//	newly created slot
//	!! Doesnt work for MFNodes
//	InsertAt is the 0-based index for the new slot
M4Err MFField_Insert(void *mf, u32 FieldType, void **new_ptr, u32 InsertAt)
{
	char *buffer;
	u32 FieldSize, i, k;
	GenMFField *mffield = (GenMFField *)mf;

	if (SG_IsSFField(FieldType)) return M4BadParam;
	if (FieldType == FT_MFNode) return M4BadParam;

	FieldSize = GetSFFieldSize(FieldType);
	
	//field we can't copy
	if (!FieldSize) return M4BadParam;
	
	//first item ever
	if (!mffield->count || !mffield->array) {
		if (mffield->array) free(mffield->array);
		mffield->array = malloc(sizeof(char)*FieldSize);
		memset(mffield->array, 0, sizeof(char)*FieldSize);
		mffield->count = 1;
		if (new_ptr) *new_ptr = mffield->array;
		return M4OK;
	}

	//alloc 1+itemCount
	buffer = malloc(sizeof(char)*(1+mffield->count)*FieldSize);

	//append at the end
	if (InsertAt >= mffield->count) {
		memcpy(buffer, mffield->array, mffield->count * FieldSize);
		memset(buffer + mffield->count * FieldSize, 0, FieldSize);
		if (new_ptr) *new_ptr = buffer + mffield->count * FieldSize;
		free(mffield->array);
		mffield->array = buffer;
		mffield->count += 1;
		return M4OK;
	}
	//insert in the array
	k=0;
	for (i=0; i < mffield->count; i++) {
		if (InsertAt == i) {
			if (new_ptr) {
				*new_ptr = buffer + i*FieldSize;
				memset(*new_ptr, 0, sizeof(char)*FieldSize);
			}
			k = 1;
		}
		memcpy(buffer + (k+i) * FieldSize , mffield->array + i*FieldSize, FieldSize);
	}
	free(mffield->array);
	mffield->array = buffer;
	mffield->count += 1;
	return M4OK;
}

#define MAX_MFFIELD_ALLOC		5000000
M4Err MFField_Alloc(void *mf, u32 FieldType, u32 NbItems)
{
	u32 FieldSize;
	GenMFField *mffield = (GenMFField *)mf;

	if (SG_IsSFField(FieldType)) return M4BadParam;
	if (FieldType == FT_MFNode) return M4BadParam;

	FieldSize = GetSFFieldSize(FieldType);
	
	//field we can't copy
	if (!FieldSize) return M4BadParam;
	if (NbItems>MAX_MFFIELD_ALLOC) return M4IOErr;

	//erase everything
	if (mffield->array) free(mffield->array);
	mffield->array = NULL;
	if (NbItems) {
		mffield->array = malloc(sizeof(char)*FieldSize*NbItems);
		memset(mffield->array, 0, sizeof(char)*FieldSize*NbItems);
	}
	mffield->count = NbItems;
	return M4OK;
}

M4Err MFField_GetItem(void *mf, u32 FieldType, void **new_ptr, u32 ItemPos)
{
	u32 FieldSize;
	GenMFField *mffield = (GenMFField *)mf;

	*new_ptr = NULL;
	if (SG_IsSFField(FieldType)) return M4BadParam;
	if (FieldType == FT_MFNode) return M4BadParam;

	FieldSize = GetSFFieldSize(FieldType);
	
	//field we can't copy
	if (!FieldSize) return M4BadParam;
	if (ItemPos >= mffield->count) return M4BadParam;
	*new_ptr = mffield->array + ItemPos * FieldSize;
	return M4OK;
}


M4Err MFField_Append(void *mf, u32 FieldType, void **new_ptr)
{
	GenMFField *mffield = (GenMFField *)mf;
	return MFField_Insert(mf, FieldType, new_ptr, mffield->count+2);
}


//remove the specified item (0-based index)
M4Err MFField_Remove(void *mf, u32 FieldType, u32 RemoveFrom)
{
	char *buffer;
	u32 FieldSize, i, k;
	GenMFField *mffield = (GenMFField *)mf;

	FieldSize = GetSFFieldSize(FieldType);
	
	//field we can't copy
	if (!FieldSize) return M4BadParam;

	if (!mffield->count || RemoveFrom >= mffield->count) return M4BadParam;

	if (mffield->count == 1) {
		free(mffield->array);
		mffield->array = NULL;
		mffield->count = 0;
		return M4OK;
	}
	k=0;
	buffer = malloc(sizeof(char)*(mffield->count-1)*FieldSize);
	for (i=0; i<mffield->count; i++) {
		if (RemoveFrom == i) {
			k = 1;
		} else {
			memcpy(buffer + (i-k)*FieldSize, mffield->array + i*FieldSize, FieldSize);
		}
	}
	free(mffield->array);
	mffield->array = buffer;
	mffield->count -= 1;
	return M4OK;
}

M4Err MFField_Reset(void *mf, u32 FieldType)
{
	GenMFField *mffield = (GenMFField *)mf;

	//field we can't copy
	if (SG_IsSFField(FieldType)) return M4BadParam;
	if (!GetSFFieldSize(FieldType)) return M4BadParam;

	switch (FieldType) {
	case FT_MFString:
		MFString_Del( * ((MFString *) mf));
		break;
	case FT_MFURL:
		MFURL_Del( * ((MFURL *) mf));
		break;
	default:
		free(mffield->array);
		break;
	}

	mffield->array = NULL;
	mffield->count = 0;
	return M4OK;
}

/*special cloning with type-casting from SF/MF strings to URL conversion since proto URL doesn't exist
as a field type (it's just a stupid encoding trick) */
void SG_CopyCastField(void *dest, u32 dst_field_type, void *orig, u32 ori_field_type)
{
	SFURL *url;
	char tmp[50];
	u32 size, i, sf_type_ori, sf_type_dst;
	void *dst_field, *orig_field;
	if (!dest || !orig) return;
	
	switch (dst_field_type) {
	case FT_SFString:
		if (ori_field_type != FT_SFURL) return;
		url = ((SFURL *)orig);
		if (url->OD_ID>0) {
			sprintf(tmp, "%d", url->OD_ID);
			if ( ((SFString*)dest)->buffer) free(((SFString*)dest)->buffer);
			((SFString*)dest)->buffer = strdup(tmp);
		} else {
			if ( ((SFString*)dest)->buffer) free(((SFString*)dest)->buffer);
			((SFString*)dest)->buffer = strdup(url->url);
		}
		return;
	case FT_SFURL:
		if (ori_field_type != FT_SFString) return;
		url = ((SFURL *)dest);
		url->OD_ID = 0;
		if (url->url) free(url->url);
		if ( ((SFString*)orig)->buffer) 
			url->url = strdup(((SFString*)orig)->buffer);
		else 
			url->url = NULL;
		return;
	case FT_MFString:
	case FT_MFURL:
		break;
	default:
		return;
	}

	size = ((GenMFField *)orig)->count;
	MFField_Alloc(dest, dst_field_type, size);
	sf_type_ori = SG_GetSFType(ori_field_type);
	sf_type_dst = SG_GetSFType(dst_field_type);
	//duplicate all items
	for (i=0; i<size; i++) {
		MFField_GetItem(dest, dst_field_type, &dst_field, i);
		MFField_GetItem(orig, ori_field_type, &orig_field, i);
		SG_CopyCastField(dst_field, sf_type_dst, orig_field, sf_type_ori);
	}
	return;
}

void SG_CopyField(void *dest, void *orig, u32 field_type)
{
	u32 size, i, sf_type;
	void *dst_field, *orig_field;

	if (!dest || !orig) return;

	switch (field_type) {
	case FT_SFBool:
		memcpy(dest, orig, sizeof(SFBool));
		break;
	case FT_SFColor:
		memcpy(dest, orig, sizeof(SFColor));
		break;
	case FT_SFFloat:
		memcpy(dest, orig, sizeof(SFFloat));
		break;
	case FT_SFInt32:
		memcpy(dest, orig, sizeof(SFInt32));
		break;
	case FT_SFRotation:
		memcpy(dest, orig, sizeof(SFRotation));
		break;
	case FT_SFTime:
		memcpy(dest, orig, sizeof(SFTime));
		break;
	case FT_SFVec2f:
		memcpy(dest, orig, sizeof(SFVec2f));
		break;
	case FT_SFVec3f:
		memcpy(dest, orig, sizeof(SFVec3f));
		break;
	case FT_SFString:
		if ( ((SFString*)dest)->buffer) free(((SFString*)dest)->buffer);
		if ( ((SFString*)orig)->buffer )
			((SFString*)dest)->buffer = strdup(((SFString*)orig)->buffer);
		else
			((SFString*)dest)->buffer = NULL;
		break;
	case FT_SFURL:
		if ( ((SFURL *)dest)->url ) free( ((SFURL *)dest)->url );
		((SFURL *)dest)->OD_ID = ((SFURL *)orig)->OD_ID;
		if (((SFURL *)orig)->url) 
			((SFURL *)dest)->url = strdup(((SFURL *)orig)->url);
		else
			((SFURL *)dest)->url = NULL;
		break;
	case FT_SFImage:
		if (((SFImage *)dest)->pixels) free(((SFImage *)dest)->pixels);
		((SFImage *)dest)->width = ((SFImage *)orig)->width;
		((SFImage *)dest)->height = ((SFImage *)orig)->height;
		((SFImage *)dest)->numComponents  = ((SFImage *)orig)->numComponents;
		size = ((SFImage *)dest)->width * ((SFImage *)dest)->height * ((SFImage *)dest)->numComponents;
		((SFImage *)dest)->pixels = malloc(sizeof(char)*size);
		memcpy(((SFImage *)dest)->pixels, ((SFImage *)orig)->pixels, sizeof(char)*size);
		break;
	case FT_SFCommandBuffer:
		SFCommandBuffer_Del( *(SFCommandBuffer *)dest);
		((SFCommandBuffer *)dest)->commandList = NewChain();
		((SFCommandBuffer *)dest)->bufferSize = ((SFCommandBuffer *)orig)->bufferSize;
		((SFCommandBuffer *)dest)->buffer = malloc(sizeof(char)*((SFCommandBuffer *)orig)->bufferSize);
		memcpy(((SFCommandBuffer *)dest)->buffer, 
			((SFCommandBuffer *)orig)->buffer,
			sizeof(char)*((SFCommandBuffer *)orig)->bufferSize);
		break;

	/*simply copy text string*/
	case FT_SFScript:
		if (((SFScript*)dest)->script_text) free(((SFScript*)dest)->script_text);		
		((SFScript*)dest)->script_text = NULL;
		if ( ((SFScript*)orig)->script_text)
			((SFScript *)dest)->script_text = strdup( ((SFScript*)orig)->script_text );
		break;


	//MFFields
	case FT_MFBool:
	case FT_MFFloat:
	case FT_MFTime:
	case FT_MFInt32:
	case FT_MFString:
	case FT_MFVec3f:
	case FT_MFVec2f:
	case FT_MFColor:
	case FT_MFRotation:
	case FT_MFImage:
	case FT_MFURL:
	case FT_MFScript:
		size = ((GenMFField *)orig)->count;
		MFField_Reset(dest, field_type);
		MFField_Alloc(dest, field_type, size);
		sf_type = SG_GetSFType(field_type);
		//duplicate all items
		for (i=0; i<size; i++) {
			MFField_GetItem(dest, field_type, &dst_field, i);
			MFField_GetItem(orig, field_type, &orig_field, i);
			SG_CopyField(dst_field, orig_field, sf_type);
		}
		break;
	}
}


Bool SG_FieldsEqual(void *dest, void *orig, u32 field_type)
{
	u32 size, i, sf_type;
	void *dst_field, *orig_field;
	Bool changed = 0;

	if (!dest || !orig) return 0;

	switch (field_type) {
	case FT_SFBool:
		changed = memcmp(dest, orig, sizeof(SFBool));
		break;
	case FT_SFColor:
		if (((SFColor *)dest)->red != ((SFColor *)orig)->red) changed = 1;
		else if (((SFColor *)dest)->green != ((SFColor *)orig)->green) changed = 1;
		else if (((SFColor *)dest)->blue != ((SFColor *)orig)->blue) changed = 1;
		break;
	case FT_SFFloat:
		if ( (*(SFFloat *)dest) != (*(SFFloat *)orig) ) changed = 1;
		break;
	case FT_SFInt32:
		changed = memcmp(dest, orig, sizeof(SFInt32));
		break;
	case FT_SFRotation:
		if (((SFRotation *)dest)->xAxis != ((SFRotation *)orig)->xAxis) changed = 1;
		else if (((SFRotation *)dest)->yAxis != ((SFRotation *)orig)->yAxis) changed = 1;
		else if (((SFRotation *)dest)->zAxis != ((SFRotation *)orig)->zAxis) changed = 1;
		else if (((SFRotation *)dest)->angle != ((SFRotation *)orig)->angle) changed = 1;
		break;
	case FT_SFTime:
		if ( (*(SFTime *)dest) != (*(SFTime*)orig) ) changed = 1;
		break;
	case FT_SFVec2f:
		if (((SFVec2f *)dest)->x != ((SFVec2f *)orig)->x) changed = 1;
		else if (((SFVec2f *)dest)->y != ((SFVec2f *)orig)->y) changed = 1;
		break;
	case FT_SFVec3f:
		if (((SFVec3f *)dest)->x != ((SFVec3f *)orig)->x) changed = 1;
		else if (((SFVec3f *)dest)->y != ((SFVec3f *)orig)->y) changed = 1;
		else if (((SFVec3f *)dest)->z != ((SFVec3f *)orig)->z) changed = 1;
		break;
	case FT_SFString:
		if ( ((SFString*)dest)->buffer && ((SFString*)orig)->buffer) {
			changed = strcmp(((SFString*)dest)->buffer, ((SFString*)orig)->buffer);
		} else {
			changed = ( !((SFString*)dest)->buffer && !((SFString*)orig)->buffer) ? 0 : 1;
		}
		break;
	case FT_SFURL:
		if (((SFURL *)dest)->OD_ID > 0 || ((SFURL *)orig)->OD_ID > 0) {
			if ( ((SFURL *)orig)->OD_ID != ((SFURL *)dest)->OD_ID) changed = 1;
		} else {
			if ( ((SFURL *)orig)->url && ! ((SFURL *)dest)->url) changed = 1;
			else if ( ! ((SFURL *)orig)->url && ((SFURL *)dest)->url) changed = 1;
			else if ( strcmp( ((SFURL *)orig)->url , ((SFURL *)dest)->url) ) changed = 1;
		}
		break;
	case FT_SFImage:
	case FT_SFScript:
	case FT_SFCommandBuffer:
		changed = 1;
		break;

	//MFFields
	case FT_MFBool:
	case FT_MFFloat:
	case FT_MFTime:
	case FT_MFInt32:
	case FT_MFString:
	case FT_MFVec3f:
	case FT_MFVec2f:
	case FT_MFColor:
	case FT_MFRotation:
	case FT_MFImage:
	case FT_MFURL:
	case FT_MFScript:
		if ( ((GenMFField *)orig)->count != ((GenMFField *)dest)->count) changed = 1;
		else {
			size = ((GenMFField *)orig)->count;
			sf_type = SG_GetSFType(field_type);
			for (i=0; i<size; i++) {
				MFField_GetItem(dest, field_type, &dst_field, i);
				MFField_GetItem(orig, field_type, &orig_field, i);
				if (! SG_FieldsEqual(dst_field, orig_field, sf_type) ) {
					changed = 1;
					break;
				}
			}
		}
		break;
	}
	return changed ? 0 : 1;
}


void SFColor_fromHSV(SFColor *col)
{
	Float _r, _g, _b;
	s32 hh = (s32) (col->red/360.0f);
	Float hue = col->red - hh * 360.0f;
	Float h6 = hue/60.0f;
	s32 i = (s32) h6;
	Float f = h6-i;
	Float p1 = col->blue * (1.0f - col->green);
	Float p2 = col->blue * (1.0f - (col->green*f));
	Float p3 = col->blue * (1.0f - col->green * (1-f) );
	switch(i) {
	case 0:
		_r = col->blue;	
		_g = p3;	
		_b = p1;
		break;
	case 1:
		_r = p2;
		_g = col->blue;
		_b = p1;
		break;
	case 2:
		_r = p1;
		_g = col->blue;
		_b = p3;
		break;
	case 3:
		_r = p1;       
		_g = p2;
		_b = col->blue;
		break;
	case 4:
		_r = p3;
		_g = p1;
		_b = col->blue;
		break;
	case 5:
		_r = col->blue;
		_g = p1;
		_b = p2;
		break;
	default:
		_r = _g = _b = 0;
	}
	col->red = _r;
	col->green = _g;
	col->blue = _b;
}

void SFColor_toHSV(SFColor *col)
{
	Float h, s;
	Float _max = MAX(col->red, MAX(col->green, col->blue));
	Float _min = MIN(col->red, MAX(col->green, col->blue));

	s = ((_max == 0) ? 0 : ((_max-_min)/_max));
	if (s != 0) {
		Float rl = (_max - col->red) / (_max-_min);
		Float gl = (_max - col->green) / (_max-_min);
		Float bl = (_max - col->blue) / (_max-_min);
		if (_max == col->red) {
			if (_min == col->green) h = 60*(5+bl);
			else h = 60*(1-gl);
		} else if (_max == col->green) {
			if (_min == col->blue) h = 60*(1+rl);
			else h = 60*(3-bl);
		} else {
			if (_min == col->red) h = 60*(3+gl);
			else h = 60*(5-rl);
		}
	} else {
		h = 0;
	}
	col->red = h;
	col->green = s;
	col->blue = _max;
}
