/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_render_tools.h>

void mx2d_set_coefs(M4Matrix2D *_this, Float *from_6)
{
	if (!_this) return;
	_this->m[0] = from_6[0];
	_this->m[1] = from_6[1];
	_this->m[2] = from_6[2];
	_this->m[3] = from_6[3];
	_this->m[4] = from_6[4];
	_this->m[5] = from_6[5];
}

void mx2d_add_matrix(M4Matrix2D *_this, M4Matrix2D *from)
{
	M4Matrix2D bck;
	if (!_this || !from) return;
	mx2d_copy(bck, *_this);
	_this->m[0] = from->m[0]*bck.m[0] + from->m[1]*bck.m[3];
	_this->m[1] = from->m[0]*bck.m[1] + from->m[1]*bck.m[4];
	_this->m[2] = from->m[0]*bck.m[2] + from->m[1]*bck.m[5] + from->m[2];
	_this->m[3] = from->m[3]*bck.m[0] + from->m[4]*bck.m[3];
	_this->m[4] = from->m[3]*bck.m[1] + from->m[4]*bck.m[4];
	_this->m[5] = from->m[3]*bck.m[2] + from->m[4]*bck.m[5] + from->m[5];
}

void mx2d_add_translation(M4Matrix2D *_this, Float cx, Float cy)
{
	M4Matrix2D tmp;
	if (!_this || (!cx && !cy) ) return;
	mx2d_init(tmp);
	tmp.m[2] = cx;
	tmp.m[5] = cy;
	mx2d_add_matrix(_this, &tmp);
}


void mx2d_add_rotation(M4Matrix2D *_this, Float cx, Float cy, Float angle)
{
	M4Matrix2D tmp;
	if (!_this) return;
	mx2d_init(tmp);

	mx2d_add_translation(_this, -cx, -cy);
	
	tmp.m[0] = (Float) cos(angle);
	tmp.m[4] = tmp.m[0];
	tmp.m[3] = (Float) sin(angle);
	tmp.m[1] = -1 * tmp.m[3];
	mx2d_add_matrix(_this, &tmp);

	mx2d_add_translation(_this, cx, cy);
}

void mx2d_add_scale(M4Matrix2D *_this, Float scale_x, Float scale_y)
{
	M4Matrix2D tmp;
	if (!_this || ((scale_x==1.0) && (scale_y==1.0)) ) return;
	mx2d_init(tmp);
	tmp.m[0] = scale_x;
	tmp.m[4] = scale_y;
	mx2d_add_matrix(_this, &tmp);
}

void mx2d_add_scale_at(M4Matrix2D *_this, Float scale_x, Float scale_y, Float cx, Float cy, Float angle)
{
	M4Matrix2D tmp;
	if (!_this) return;
	mx2d_init(tmp);
	if (angle) {
		mx2d_add_rotation(_this, cx, cy, -angle);
	}
	tmp.m[0] = scale_x;
	tmp.m[4] = scale_y;
	mx2d_add_matrix(_this, &tmp);
	if (angle) mx2d_add_rotation(_this, cx, cy, angle);
}

Float mx2d_get_determinent(M4Matrix2D *_this)
{
	if (_this)
		return _this->m[0]*_this->m[4] - _this->m[1]*_this->m[3];
	return 0;
}

void mx2d_inverse(M4Matrix2D *_this)
{
	Float res;
	M4Matrix tmp;
	if(!_this) return;
	res = mx2d_get_determinent(_this);
	if (! res) {
		mx2d_init(*_this);
		return;
	}
	tmp.m[0] = _this->m[4]/res;
	tmp.m[1] = -1 *  _this->m[1]/res;
	tmp.m[2] =  ( _this->m[1]*_this->m[5] - _this->m[4]*_this->m[2]) / res;
	tmp.m[3] = -1 * _this->m[3]/res;
	tmp.m[4] =  _this->m[0]/res;
	tmp.m[5] = -1 * ( _this->m[0]*_this->m[5] - _this->m[3]*_this->m[2]) / res;
	mx2d_copy(*_this, tmp);
}


void mx2d_apply_coords(M4Matrix2D *_this, Float *x, Float *y)
{
	Float _x, _y;
	if (!_this || !x || !y) return;

	_x = *x * _this->m[0] + *y * _this->m[1] + _this->m[2];
	_y = *x * _this->m[3] + *y * _this->m[4] + _this->m[5];
	*x = _x;
	*y = _y;
}

void mx2d_apply_point(M4Matrix2D *_this, SFVec2f *pt)
{
	mx2d_apply_coords(_this, &pt->x, &pt->y);
}

void mx2d_apply_rect(M4Matrix2D *_this, M4Rect *rc)
{
	SFVec2f tl, tr, bl, br;
	tl.x = rc->x;
	tl.y = rc->y;
	tr.x = rc->x+rc->width;
	tr.y = rc->y;
	bl.x = rc->x;
	bl.y = rc->y - rc->height;
	br.x = rc->x+rc->width;
	br.y = rc->y - rc->height;

	mx2d_apply_point(_this, &tl);
	mx2d_apply_point(_this, &tr);
	mx2d_apply_point(_this, &bl);
	mx2d_apply_point(_this, &br);

	rc->x = tl.x;
	if (rc->x>tr.x) rc->x=tr.x;
	if (rc->x>bl.x) rc->x=bl.x;
	if (rc->x>br.x) rc->x=br.x;

	rc->y = tl.y;
	if (rc->y<tr.y) rc->y=tr.y;
	if (rc->y<bl.y) rc->y=bl.y;
	if (rc->y<br.y) rc->y=br.y;
	
	rc->width = tl.x;
	if (rc->width<tr.x) rc->width=tr.x;
	if (rc->width<bl.x) rc->width=bl.x;
	if (rc->width<br.x) rc->width=br.x;

	rc->height = tl.y;
	if (rc->height>tr.y) rc->height=tr.y;
	if (rc->height>bl.y) rc->height=bl.y;
	if (rc->height>br.y) rc->height=br.y;

	rc->height = rc->y - rc->height;
	rc->width -= rc->x;
	assert(rc->height>=0);
	assert(rc->width>=0);
}

void mx2d_from_mx(M4Matrix2D *mat2D, M4Matrix *mat)
{
	mx_init(*mat2D);
	mat2D->m[0] = mat->m[0];
	mat2D->m[1] = mat->m[4];
	mat2D->m[2] = mat->m[12];
	mat2D->m[3] = mat->m[1];
	mat2D->m[4] = mat->m[5];
	mat2D->m[5] = mat->m[13];
}

void mx_apply_rect(M4Matrix *mat, M4Rect *rc)
{
	M4Matrix2D mat2D;
	mx2d_from_mx(&mat2D, mat);
	mx2d_apply_rect(&mat2D, rc);
}

void mx_add_matrix(M4Matrix *mat, M4Matrix *mul)
{
    M4Matrix tmp;
	mx_init(tmp);

    tmp.m[0] = mat->m[0]*mul->m[0] + mat->m[4]*mul->m[1] + mat->m[8]*mul->m[2];
    tmp.m[4] = mat->m[0]*mul->m[4] + mat->m[4]*mul->m[5] + mat->m[8]*mul->m[6];
    tmp.m[8] = mat->m[0]*mul->m[8] + mat->m[4]*mul->m[9] + mat->m[8]*mul->m[10];
    tmp.m[12]= mat->m[0]*mul->m[12] + mat->m[4]*mul->m[13] + mat->m[8]*mul->m[14] + mat->m[12];
    tmp.m[1] = mat->m[1]*mul->m[0] + mat->m[5]*mul->m[1] + mat->m[9]*mul->m[2];
    tmp.m[5] = mat->m[1]*mul->m[4] + mat->m[5]*mul->m[5] + mat->m[9]*mul->m[6];
    tmp.m[9] = mat->m[1]*mul->m[8] + mat->m[5]*mul->m[9] + mat->m[9]*mul->m[10];
    tmp.m[13]= mat->m[1]*mul->m[12] + mat->m[5]*mul->m[13] + mat->m[9]*mul->m[14] + mat->m[13];
    tmp.m[2] = mat->m[2]*mul->m[0] + mat->m[6]*mul->m[1] + mat->m[10]*mul->m[2];
    tmp.m[6] = mat->m[2]*mul->m[4] + mat->m[6]*mul->m[5] + mat->m[10]*mul->m[6];
    tmp.m[10]= mat->m[2]*mul->m[8] + mat->m[6]*mul->m[9] + mat->m[10]*mul->m[10];
    tmp.m[14]= mat->m[2]*mul->m[12] + mat->m[6]*mul->m[13] + mat->m[10]*mul->m[14] + mat->m[14];
	memcpy(mat->m, tmp.m, sizeof(Float)*16);
}

void mx_add_translation(M4Matrix *mat, Float tx, Float ty, Float tz)
{
	Float tmp[3];
	u32 i;
	tmp[0] = mat->m[12];
	tmp[1] = mat->m[13];
	tmp[2] = mat->m[14];
	for (i=0; i<3; i++)
		tmp[i] += (tx*mat->m[i] + ty * mat->m[i+4] + tz * mat->m[i + 8]);
	mat->m[12] = tmp[0];
	mat->m[13] = tmp[1];
	mat->m[14] = tmp[2];
}

void mx_add_scale(M4Matrix *mat, Float sx, Float sy, Float sz)
{
	Float tmp[3];
	u32 i, j;

	tmp[0] = sx; 
	tmp[1] = sy; 
	tmp[2] = sz; 

	for (i=0; i<3; i++) {
		for (j=0; j<3; j++) {
			mat->m[i*4 + j] = mat->m[j+4 * i] * tmp[i];
		}
	}
}

void mx_add_rotation(M4Matrix *mat, Float angle, Float x, Float y, Float z)
{
	M4Matrix tmp;
	Float xx, yy, zz, xy, xz, yz;
	Float nor = (Float) sqrt(x*x + y*y + z*z);
	Float cos_a = (Float) cos(angle);
	Float sin_a = (Float) sin(angle);
	Float icos_a = 1.0f - cos_a;
	if (nor) { x /= nor; y /= nor; z /= nor; }
	xx = x*x; yy = y*y; zz = z*z; xy = x*y; xz = x*z; yz = y*z;
	mx_init(tmp);
    tmp.m[0] = icos_a*xx + cos_a;
    tmp.m[1] = xy*icos_a + z*sin_a;
    tmp.m[2] = xz*icos_a - y*sin_a;
    
	tmp.m[4] = xy*icos_a - z*sin_a;
    tmp.m[5] = icos_a*yy + cos_a;
    tmp.m[6] = yz*icos_a + x*sin_a;

	tmp.m[8] = xz*icos_a + y*sin_a;
    tmp.m[9] = yz*icos_a - x*sin_a;
    tmp.m[10]= icos_a*zz + cos_a;

	mx_add_matrix(mat, &tmp);
}

void mx_from_mx2d(M4Matrix *mat, M4Matrix2D *mat2D)
{
	mx_init(*mat);
	mat->m[0] = mat2D->m[0];
	mat->m[4] = mat2D->m[1];
	mat->m[12] = mat2D->m[2];
	mat->m[1] = mat2D->m[3];
	mat->m[5] = mat2D->m[4];
	mat->m[13] = mat2D->m[5];
}

void mx_inverse(M4Matrix *mx)
{
    Float det;
	M4Matrix rev;
	mx_init(rev);

	assert(! ((mx->m[3] != 0.0) || (mx->m[7] != 0.0) || (mx->m[11] != 0.0) || (mx->m[15] != 1.0)) );

	det = mx->m[0] * mx->m[5] * mx->m[10] + mx->m[1] * mx->m[6] * mx->m[8] + mx->m[2] * mx->m[4] * mx->m[9]
		- mx->m[2] * mx->m[5] * mx->m[8] - mx->m[1] * mx->m[4] * mx->m[10] - mx->m[0] * mx->m[6] * mx->m[9];

	if (det * det < M4_EPSILON_FLOAT) return;

	/* Calculate inverse(A) = adj(A) / det(A) */
	det = 1.0f / det;
	rev.m[0] = (Float)  ((mx->m[5] * mx->m[10] - mx->m[6] * mx->m[9]) * det);
	rev.m[4] = (Float) -((mx->m[4] * mx->m[10] - mx->m[6] * mx->m[8]) * det);
	rev.m[8] = (Float)  ((mx->m[4] * mx->m[9] - mx->m[5] * mx->m[8]) * det);
	rev.m[1] = (Float) -((mx->m[1] * mx->m[10] - mx->m[2] * mx->m[9]) * det);
	rev.m[5] = (Float)  ((mx->m[0] * mx->m[10] - mx->m[2] * mx->m[8]) * det);
	rev.m[9] = (Float) -((mx->m[0] * mx->m[9] - mx->m[1] * mx->m[8]) * det);
	rev.m[2] = (Float)  ((mx->m[1] * mx->m[6] - mx->m[2] * mx->m[5]) * det);
	rev.m[6] = (Float) -((mx->m[0] * mx->m[6] - mx->m[2] * mx->m[4]) * det);
	rev.m[10] = (Float) ((mx->m[0] * mx->m[5] - mx->m[1] * mx->m[4]) * det);

	/* do translation part*/
	rev.m[12] = -( mx->m[12] * rev.m[0] + mx->m[13] * rev.m[4] + mx->m[14] * rev.m[8] );
	rev.m[13] = -( mx->m[12] * rev.m[1] + mx->m[13] * rev.m[5] + mx->m[14] * rev.m[9] );
	rev.m[14] = -( mx->m[12] * rev.m[2] + mx->m[13] * rev.m[6] + mx->m[14] * rev.m[10] );
#undef PRECISION_LIMIT
	mx_copy(*mx, rev);
}


void mx_apply_vec(M4Matrix *mx, SFVec3f *pt)
{
	SFVec3f res;
	res.x = pt->x * mx->m[0] + pt->y * mx->m[4] + pt->y * mx->m[8] + mx->m[12];
	res.y = pt->x * mx->m[1] + pt->y * mx->m[5] + pt->z * mx->m[9] + mx->m[13];
	res.z = pt->x * mx->m[2] + pt->y * mx->m[6] + pt->z * mx->m[10] + mx->m[14];
	*pt = res;
}

void mx_ortho(M4Matrix *mx, Float left, Float right, Float bottom, Float top, Float z_near, Float z_far)
{
	mx_init(*mx);
	mx->m[0] = 2/(right-left);
	mx->m[5] = 2/(top-bottom);
	mx->m[10] = -2/(z_far-z_near);
	mx->m[12] = (right+left)/(right-left);
	mx->m[13] = (top+bottom)/(top-bottom);
	mx->m[14] = (z_far+z_near)/(z_far-z_near);
	mx->m[15] = 1.0;
}

void mx_perspective(M4Matrix *mx, Float fieldOfView, Float aspectRatio, Float z_near, Float z_far)
{
	Float f = (Float) (cos(fieldOfView/2) / sin(fieldOfView/2));
	mx_init(*mx);
	mx->m[0] = f/aspectRatio;
	mx->m[5] = f;
	mx->m[10] = (z_far+z_near)/(z_near-z_far);
	mx->m[11] = -1;
	mx->m[14] = 2*z_near*z_far/(z_near-z_far);
	mx->m[15] = 0;
}

void mx_lookat(M4Matrix *mx, SFVec3f eye, SFVec3f center, SFVec3f upVector)
{
	SFVec3f f, s, u;
	
	f = vec_diff(&center, &eye);
	vec_norm(&f);
	vec_norm(&upVector);

	s = vec_cross(&f, &upVector);
	u = vec_cross(&s, &f);
	mx_init(*mx);
	
	mx->m[0] = s.x;
	mx->m[1] = u.x;
	mx->m[2] = -f.x;
	mx->m[4] = s.y;
	mx->m[5] = u.y;
	mx->m[6] = -f.y;
	mx->m[8] = s.z;
	mx->m[9] = u.z;
	mx->m[10] = -f.z;

	mx_add_translation(mx, -eye.x, -eye.y, -eye.z);
}



void mx_apply_bbox(M4Matrix *mx, M4BBox *b)
{
	Float var;
	mx_apply_vec(mx, &b->min_edge);
	mx_apply_vec(mx, &b->max_edge);

	if (b->min_edge.x > b->max_edge.x) 
	{
		var = b->min_edge.x; b->min_edge.x = b->max_edge.x; b->max_edge.x = var;
	}
	if (b->min_edge.y > b->max_edge.y) 
	{
		var = b->min_edge.y; b->min_edge.y = b->max_edge.y; b->max_edge.y = var;
	}
	if (b->min_edge.z > b->max_edge.z) 
	{
		var = b->min_edge.z; b->min_edge.z = b->max_edge.z; b->max_edge.z = var;
	}
	bbox_refresh(b);
}

/*color matrix stuff*/

static void cmat_identity(M4ColorMatrix *_this)
{
	M4ColorMatrix mat;
	cmat_init(&mat);
	_this->identity = memcmp(_this->m, mat.m, sizeof(Float)*20) ? 0 : 1;
}

void cmat_init(M4ColorMatrix *_this)
{
	if (!_this) return;
	memset(_this->m, 0, sizeof(Float)*20);
	_this->m[0] = _this->m[6] = _this->m[12] = _this->m[18] = 1;
	_this->identity = 1;
}
void cmat_set_all(M4ColorMatrix *_this, Float *coefs)
{
	if (!_this || !coefs) return;
	memcpy(_this->m, coefs, sizeof(Float)*20);
	cmat_identity(_this);
}
void cmat_set(M4ColorMatrix *_this, 
				 Float c1, Float c2, Float c3, Float c4, Float c5,
				 Float c6, Float c7, Float c8, Float c9, Float c10,
				 Float c11, Float c12, Float c13, Float c14, Float c15,
				 Float c16, Float c17, Float c18, Float c19, Float c20)
{
	if (!_this) return;
	_this->m[0] = c1; _this->m[1] = c2; _this->m[2] = c3; _this->m[3] = c4; _this->m[4] = c5;
	_this->m[5] = c6; _this->m[6] = c7; _this->m[7] = c8; _this->m[8] = c9; _this->m[9] = c10;
	_this->m[10] = c11; _this->m[11] = c12; _this->m[12] = c13; _this->m[13] = c14; _this->m[14] = c15;
	_this->m[15] = c16; _this->m[16] = c17; _this->m[17] = c18; _this->m[18] = c19; _this->m[19] = c20;
	cmat_identity(_this);
}

void cmat_copy(M4ColorMatrix *_this, M4ColorMatrix *from)
{
	if (!_this || !from) return;
	memcpy(_this->m, from->m, sizeof(Float)*20);
	cmat_identity(_this);
}
M4ColorMatrix *cmat_clone(M4ColorMatrix *from)
{
	M4ColorMatrix *clone;
	if (!from) return NULL;
	clone = malloc(sizeof(M4ColorMatrix));
	cmat_copy(clone, from);
	return clone;
}

void cmat_multiply(M4ColorMatrix *_this, M4ColorMatrix *w)
{
	Float res[20];
	if (!_this || !w || w->identity) return;
	if (_this->identity) {
		cmat_copy(_this, w);
		return;
	}

	res[0] = _this->m[0]*w->m[0] + _this->m[1]*w->m[5] + _this->m[2]*w->m[10] + _this->m[3]*w->m[15];
	res[1] = _this->m[0]*w->m[1] + _this->m[1]*w->m[6] + _this->m[2]*w->m[11] + _this->m[3]*w->m[16];
	res[2] = _this->m[0]*w->m[2] + _this->m[1]*w->m[7] + _this->m[2]*w->m[12] + _this->m[3]*w->m[17];
	res[3] = _this->m[0]*w->m[3] + _this->m[1]*w->m[8] + _this->m[2]*w->m[13] + _this->m[3]*w->m[18];
	res[4] = _this->m[0]*w->m[4] + _this->m[1]*w->m[9] + _this->m[2]*w->m[14] + _this->m[3]*w->m[19] + _this->m[4];
	
	res[5] = _this->m[5]*w->m[0] + _this->m[6]*w->m[5] + _this->m[7]*w->m[10] + _this->m[8]*w->m[15];
	res[6] = _this->m[5]*w->m[1] + _this->m[6]*w->m[6] + _this->m[7]*w->m[11] + _this->m[8]*w->m[16];
	res[7] = _this->m[5]*w->m[2] + _this->m[6]*w->m[7] + _this->m[7]*w->m[12] + _this->m[8]*w->m[17];
	res[8] = _this->m[5]*w->m[3] + _this->m[6]*w->m[8] + _this->m[7]*w->m[13] + _this->m[8]*w->m[18];
	res[9] = _this->m[5]*w->m[4] + _this->m[6]*w->m[9] + _this->m[7]*w->m[14] + _this->m[8]*w->m[19] + _this->m[9];
	
	res[10] = _this->m[10]*w->m[0] + _this->m[11]*w->m[5] + _this->m[12]*w->m[10] + _this->m[13]*w->m[15];
	res[11] = _this->m[10]*w->m[1] + _this->m[11]*w->m[6] + _this->m[12]*w->m[11] + _this->m[13]*w->m[16];
	res[12] = _this->m[10]*w->m[2] + _this->m[11]*w->m[7] + _this->m[12]*w->m[12] + _this->m[13]*w->m[17];
	res[13] = _this->m[10]*w->m[3] + _this->m[11]*w->m[8] + _this->m[12]*w->m[13] + _this->m[13]*w->m[18];
	res[14] = _this->m[10]*w->m[4] + _this->m[11]*w->m[9] + _this->m[12]*w->m[14] + _this->m[13]*w->m[19] + _this->m[14];
	
	res[15] = _this->m[15]*w->m[0] + _this->m[16]*w->m[5] + _this->m[17]*w->m[10] + _this->m[18]*w->m[15];
	res[16] = _this->m[15]*w->m[1] + _this->m[16]*w->m[6] + _this->m[17]*w->m[11] + _this->m[18]*w->m[16];
	res[17] = _this->m[15]*w->m[2] + _this->m[16]*w->m[7] + _this->m[17]*w->m[12] + _this->m[18]*w->m[17];
	res[18] = _this->m[15]*w->m[3] + _this->m[16]*w->m[8] + _this->m[17]*w->m[13] + _this->m[18]*w->m[18];
	res[19] = _this->m[15]*w->m[4] + _this->m[16]*w->m[9] + _this->m[17]*w->m[14] + _this->m[18]*w->m[19] + _this->m[19];
	
	cmat_set_all(_this, res);
}

#define CLIP_COLOR(val)	if (val<0.0) { val=0; } else if (val>1.0) { val=1.0; }

M4Color cmat_apply(M4ColorMatrix *_this, M4Color col)
{
	Float a, r, g, b;
	Float _a, _r, _g, _b;
	if (!_this || _this->identity) return col;

	a = M4C_A(col); a /= 255;
	r = M4C_R(col); r /= 255;
	g = M4C_G(col); g /= 255;
	b = M4C_B(col); b /= 255;

	_r = r * _this->m[0] + g * _this->m[1] + b * _this->m[2] + a * _this->m[3] + _this->m[4];
	_g = r * _this->m[5] + g * _this->m[6] + b * _this->m[7] + a * _this->m[8] + _this->m[9];
	_b = r * _this->m[10] + g * _this->m[11] + b * _this->m[12] + a * _this->m[13] + _this->m[14];
	_a = r * _this->m[15] + g * _this->m[16] + b * _this->m[17] + a * _this->m[18] + _this->m[19];

	CLIP_COLOR(_a);
	CLIP_COLOR(_r);
	CLIP_COLOR(_g);
	CLIP_COLOR(_b);
	return MAKE_ARGB_FLOAT(_a, _r, _g, _b);
}


#define ConvexCompare(delta)	\
    ( (delta.x > 0) ? -1 :		\
      (delta.x < 0) ?	1 :		\
      (delta.y > 0) ? -1 :		\
      (delta.y < 0) ?	1 :	\
      0 )

#define ConvexGetPointDelta(delta, pprev, pcur )			\
    /* Given a previous point 'pprev', read a new point into 'pcur' */	\
    /* and return delta in 'delta'.				    */	\
    pcur = pts[iread++];						\
    delta.x = pcur.x - pprev.x;					\
    delta.y = pcur.y - pprev.y;					\

#define ConvexCross(p, q) p.x*q.y - p.y*q.x;

#define ConvexCheckTriple						\
    if ( (thisDir = ConvexCompare(dcur)) == -curDir ) {			\
	  ++dirChanges;							\
	  /* if ( dirChanges > 2 ) return NotConvex;		     */ \
    }									\
    curDir = thisDir;							\
    cross = ConvexCross(dprev, dcur);					\
    if ( cross > 0 ) { \
		if ( angleSign == -1 ) return M4_PolyComplex;		\
		angleSign = 1;					\
	}							\
    else if (cross < 0) {	\
		if (angleSign == 1) return M4_PolyComplex;		\
		angleSign = -1;				\
	}						\
    pSecond = pThird;		\
    dprev.x = dcur.x;		\
    dprev.y = dcur.y;							\

u32 polygon2D_check_convexity(M4Point2D *pts, u32 len)
{
	s32 curDir, thisDir = 0, dirChanges = 0, angleSign = 0;
	u32 iread;
    Float cross;
	M4Point2D pSecond, pThird, pSaveSecond;
	M4Point2D dprev, dcur;

    /* Get different point, return if less than 3 diff points. */
    if (len < 3 ) return M4_PolyConvexLine;
    iread = 1;
	ConvexGetPointDelta(dprev, (pts[0]), pSecond);
    pSaveSecond = pSecond;
	/*initial direction */
    curDir = ConvexCompare(dprev);
    while ( iread < len) {
		/* Get different point, break if no more points */
		ConvexGetPointDelta(dcur, pSecond, pThird );
		if ( (dcur.x == 0.0f) && (dcur.y == 0.0f) ) continue;
		/* Check current three points */
		ConvexCheckTriple;
    }

    /* Must check for direction changes from last vertex back to first */
	/* Prepare for 'ConvexCheckTriple' */
    pThird = pts[0];
    dcur.x = pThird.x - pSecond.x;
    dcur.y = pThird.y - pSecond.y;
    if ( ConvexCompare(dcur) ) ConvexCheckTriple;

    /* and check for direction changes back to second vertex */
    dcur.x = pSaveSecond.x - pSecond.x;
    dcur.y = pSaveSecond.y - pSecond.y;
	/* Don't care about 'pThird' now */
    ConvexCheckTriple;			

    /* Decide on polygon type given accumulated status */
    if ( dirChanges > 2 ) return M4_PolyComplex;
    if ( angleSign > 0 ) return M4_PolyConvexCCW;
    if ( angleSign < 0 ) return M4_PolyConvexCW;
    return M4_PolyConvexLine;
}


Bool plane_exists_intersection(M4Plane *plane, M4Plane *with)
{
	SFVec3f cross = vec_cross(&with->normal, &plane->normal);
	return vec_lensq(&cross) > M4_EPSILON_FLOAT;
}

Bool plane_intersect_line(M4Plane *plane, SFVec3f *linepoint, SFVec3f *linevec, SFVec3f *outPoint)
{
	Float t, t2;
	t2 = vec_dot(&plane->normal, linevec);
	if (t2 == 0) return 0;
	t = - (vec_dot(&plane->normal, linepoint) + plane->d) / t2;
	*outPoint = vec_scale(linevec, t);
	*outPoint = vec_add(linepoint, outPoint);
	return 1;
}

Bool plane_intersect_plane(M4Plane *plane, M4Plane *with, SFVec3f *linepoint, SFVec3f *linevec)
{
	Float fn00 = vec_len(&plane->normal);
	Float fn01 = vec_dot(&plane->normal, &with->normal);
	Float fn11 = vec_len(&with->normal);
	Float det = fn00*fn11 - fn01*fn01;
	if (fabs(det) > M4_EPSILON_FLOAT) {
		Float fc0, fc1;
		SFVec3f v1, v2;
		det = 1.0f / det;
		fc0 = (fn11*-plane->d + fn01*with->d) * det;
		fc1 = (fn00*-with->d + fn01*plane->d) * det;
		*linevec = vec_cross(&plane->normal, &with->normal);
		v1 = vec_scale(&plane->normal, fc0);
		v2 = vec_scale(&with->normal, fc1);
		*linepoint = vec_add(&v1, &v2);
		return 1;
	}
	return 0;
}

Bool plane_intersect_planes(M4Plane *plane, M4Plane *p1, M4Plane *p2, SFVec3f *outPoint)
{
	SFVec3f lp, lv;
	if (plane_intersect_plane(plane, p1, &lp, &lv))
		return plane_intersect_line(p2, &lp, &lv, outPoint);
	return 0;
}

/*we should only need a full matrix product for frustrum setup*/
static void projection_add_matrix(M4Matrix *mat, M4Matrix *mul)
{
    M4Matrix tmp;
	mx_init(tmp);
    tmp.m[0] = mat->m[0]*mul->m[0] + mat->m[4]*mul->m[1] + mat->m[8]*mul->m[2] + mat->m[12]*mul->m[3];
    tmp.m[1] = mat->m[1]*mul->m[0] + mat->m[5]*mul->m[1] + mat->m[9]*mul->m[2] + mat->m[13]*mul->m[3];
    tmp.m[2] = mat->m[2]*mul->m[0] + mat->m[6]*mul->m[1] + mat->m[10]*mul->m[2] + mat->m[14]*mul->m[3];
    tmp.m[3] = mat->m[3]*mul->m[0] + mat->m[7]*mul->m[1] + mat->m[11]*mul->m[2] + mat->m[15]*mul->m[3];
    tmp.m[4] = mat->m[0]*mul->m[4] + mat->m[4]*mul->m[5] + mat->m[8]*mul->m[6] + mat->m[12]*mul->m[7];
    tmp.m[5] = mat->m[1]*mul->m[4] + mat->m[5]*mul->m[5] + mat->m[9]*mul->m[6] + mat->m[13]*mul->m[7];
    tmp.m[6] = mat->m[2]*mul->m[4] + mat->m[6]*mul->m[5] + mat->m[10]*mul->m[6] + mat->m[14]*mul->m[7];
    tmp.m[7] = mat->m[3]*mul->m[4] + mat->m[7]*mul->m[5] + mat->m[11]*mul->m[6] + mat->m[15]*mul->m[7];
    tmp.m[8] = mat->m[0]*mul->m[8] + mat->m[4]*mul->m[9] + mat->m[8]*mul->m[10] + mat->m[12]*mul->m[11];
    tmp.m[9] = mat->m[1]*mul->m[8] + mat->m[5]*mul->m[9] + mat->m[9]*mul->m[10] + mat->m[13]*mul->m[11];
    tmp.m[10] = mat->m[2]*mul->m[8] + mat->m[6]*mul->m[9] + mat->m[10]*mul->m[10] + mat->m[14]*mul->m[11];
    tmp.m[11] = mat->m[3]*mul->m[8] + mat->m[7]*mul->m[9] + mat->m[11]*mul->m[10] + mat->m[15]*mul->m[11];
    tmp.m[12] = mat->m[0]*mul->m[12] + mat->m[4]*mul->m[13] + mat->m[8]*mul->m[14] + mat->m[12]*mul->m[15];
    tmp.m[13] = mat->m[1]*mul->m[12] + mat->m[5]*mul->m[13] + mat->m[9]*mul->m[14] + mat->m[13]*mul->m[15];
    tmp.m[14] = mat->m[2]*mul->m[12] + mat->m[6]*mul->m[13] + mat->m[10]*mul->m[14] + mat->m[14]*mul->m[15];
    tmp.m[15] = mat->m[3]*mul->m[12] + mat->m[7]*mul->m[13] + mat->m[11]*mul->m[14] + mat->m[15]*mul->m[15];
	memcpy(mat->m, tmp.m, sizeof(Float)*16);
}

void m4_frustum_from_matrix(M4Frustum *frus, M4Matrix *mx)
{
	u32 i;

	frus->planes[M4F_LEFT_PLANE].normal.x = mx->m[3] + mx->m[0];
	frus->planes[M4F_LEFT_PLANE].normal.y = mx->m[7] + mx->m[4];
	frus->planes[M4F_LEFT_PLANE].normal.z = mx->m[11] + mx->m[8];
	frus->planes[M4F_LEFT_PLANE].d = mx->m[15] + mx->m[12];

	frus->planes[M4F_RIGHT_PLANE].normal.x = mx->m[3] - mx->m[0];
	frus->planes[M4F_RIGHT_PLANE].normal.y = mx->m[7] - mx->m[4];
	frus->planes[M4F_RIGHT_PLANE].normal.z = mx->m[11] - mx->m[8];
	frus->planes[M4F_RIGHT_PLANE].d = mx->m[15] - mx->m[12];

	frus->planes[M4F_BOTTOM_PLANE].normal.x = mx->m[3] + mx->m[1];
	frus->planes[M4F_BOTTOM_PLANE].normal.y = mx->m[7] + mx->m[5];
	frus->planes[M4F_BOTTOM_PLANE].normal.z = mx->m[11] + mx->m[9];
	frus->planes[M4F_BOTTOM_PLANE].d = mx->m[15] + mx->m[13];

	frus->planes[M4F_TOP_PLANE].normal.x = mx->m[3] - mx->m[1];
	frus->planes[M4F_TOP_PLANE].normal.y = mx->m[7] - mx->m[5];
	frus->planes[M4F_TOP_PLANE].normal.z = mx->m[11] - mx->m[9];
	frus->planes[M4F_TOP_PLANE].d = mx->m[15] - mx->m[13];

	frus->planes[M4F_FAR_PLANE].normal.x = mx->m[3] - mx->m[2];
	frus->planes[M4F_FAR_PLANE].normal.y = mx->m[7] - mx->m[6];
	frus->planes[M4F_FAR_PLANE].normal.z = mx->m[11] - mx->m[10];
	frus->planes[M4F_FAR_PLANE].d = mx->m[15] - mx->m[14];

	frus->planes[M4F_NEAR_PLANE].normal.x = mx->m[3] + mx->m[2];
	frus->planes[M4F_NEAR_PLANE].normal.y = mx->m[7] + mx->m[6];
	frus->planes[M4F_NEAR_PLANE].normal.z = mx->m[11] + mx->m[10];
	frus->planes[M4F_NEAR_PLANE].d = mx->m[15] + mx->m[14];

	for (i=0; i<6; ++i) {
		Float len = (Float)(1.0f / vec_len(&frus->planes[i].normal));
		frus->planes[i].normal = vec_scale(&frus->planes[i].normal, len);
		frus->planes[i].d *= len;

		/*compute p-vertex idx*/
		frus->p_idx[i] = plane_get_p_vertex_idx(&frus->planes[i]);
	}
}

void frustum_update(M4Frustum *frus, M4Matrix *projection, M4Matrix *modelview)
{
	Float vlen, h, w;
	SFVec3f corner, center;
	M4Matrix mx;

	if (!frus->is_ortho) {
		mx_perspective(projection, frus->fieldOfView, frus->aspectRatio, frus->z_near, frus->z_far);
		mx_lookat(modelview, frus->position, frus->target, frus->up);
	}
	mx_copy(mx, *projection);
	projection_add_matrix(&mx, modelview);
	m4_frustum_from_matrix(frus, &mx);

	/*compute bsphere*/
	if (frus->is_ortho) {
		M4BBox b;
		/*since our frustrum is symetric, just pick 2 points*/
		b.max_edge = frustum_farrightup(frus);
		b.min_edge = frustum_farleftdown(frus);
		/*frustum should be centered, but in any case...*/
		b.min_edge.z = b.max_edge.z = (frus->z_near+frus->z_far) / 2;
		bbox_refresh(&b);
		frus->center = b.center;
		frus->radius = b.radius;
	} else {
		vlen = frus->z_far - frus->z_near;
		h = vlen * (Float) tan(frus->fieldOfView * 0.5f);
		w = h*frus->aspectRatio;
		center.x = 0.0f; center.y = 0.0f; center.z = frus->z_near + vlen * 0.5f;
		corner.x = w; corner.y = h; corner.z = vlen;
		corner = vec_diff(&corner, &center);
		frus->radius = vec_len(&corner);
		frus->center = vec_diff(&frus->target, &frus->position);
		vec_norm(&frus->center);
		frus->center = vec_scale(&frus->center, 0.5f*vlen+frus->z_near);
		frus->center = vec_add(&frus->center, &frus->position);

		center = frustum_farrightup(frus);
		center = frustum_farrightdown(frus);
		center = frustum_farleftup(frus);
		center = frustum_farleftdown(frus);
	}
}


SFVec3f frustum_farleftup(M4Frustum *frus)
{
	SFVec3f p;
	plane_intersect_planes(&frus->planes[M4F_FAR_PLANE], &frus->planes[M4F_TOP_PLANE], &frus->planes[M4F_LEFT_PLANE], &p);
	return p;
}

SFVec3f frustum_farleftdown(M4Frustum *frus)
{
	SFVec3f p;
	plane_intersect_planes(&frus->planes[M4F_FAR_PLANE], &frus->planes[M4F_BOTTOM_PLANE], &frus->planes[M4F_LEFT_PLANE], &p);
	return p;
}

SFVec3f frustum_farrightup(M4Frustum *frus)
{
	SFVec3f p;
	plane_intersect_planes(&frus->planes[M4F_FAR_PLANE], &frus->planes[M4F_TOP_PLANE], &frus->planes[M4F_RIGHT_PLANE], &p);
	return p;
}

SFVec3f frustum_farrightdown(M4Frustum *frus)
{
	SFVec3f p;
	plane_intersect_planes(&frus->planes[M4F_FAR_PLANE], &frus->planes[M4F_BOTTOM_PLANE], &frus->planes[M4F_RIGHT_PLANE], &p);
	return p;
}
