/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_render_dev.h>

/*user interaction event storage*/
typedef struct _user_event
{
	u32 event_type;
	Float x, y;
} UserEvent;



void SR_SimulationTick(LPSCENERENDER sr);


M4Err SR_SetOutputSize(LPSCENERENDER sr, u32 Width, u32 Height)
{
	M4Err e;

	SR_Lock(sr, 1);
	/*FIXME: need to check for max resolution*/
	sr->width = Width;
	sr->height = Height;
	e = sr->visual_renderer->RecomputeAR(sr->visual_renderer);
	SR_Lock(sr, 0);
	return e;
}

void SR_SetFullScreen(LPSCENERENDER sr)
{
	M4Err e;

	/*move to FS*/
	sr->fullscreen = !sr->fullscreen;
	e = sr->video_out->SetFullScreen(sr->video_out, sr->fullscreen, &sr->width, &sr->height);
	if (e) {
		M4USER_MESSAGE(sr->client, "VideoRenderer", "Cannot switch to fullscreen", e);
		sr->fullscreen = 0;
		sr->video_out->SetFullScreen(sr->video_out, 0, &sr->width, &sr->height);
	}
	e = sr->visual_renderer->RecomputeAR(sr->visual_renderer);
}

/*this is needed for:
- audio: since the audio renderer may not be threaded, it must be reconfigured by another thread otherwise 
we lock the audio plugin
- video: this is typical to OpenGL&co: multithreaded is forbidden, so resizing/fullscreen MUST be done by the same 
thread accessing the HW ressources
*/
void SR_ReconfigTask(LPSCENERENDER sr)
{
	u32 width,height;

	SR_Lock(sr, 1);
	
	/*reconfig audio if needed*/
	if (sr->audio_renderer) AR_Reconfig(sr->audio_renderer);

	/*resize visual if needed*/
	switch (sr->msg_type) {
	/*scene size has been overriden*/
	case 1:
		assert(!(sr->override_size_flags & 2));
		sr->override_size_flags |= 2;
		width = sr->scene_width;
		height = sr->scene_height;
		sr->has_size_info = 1;
		SR_SetSize(sr, width, height);
		M4USER_SETSIZE(sr->client, width, height);
		break;
	/*fullscreen on/off request*/
	case 2:
		SR_SetFullScreen(sr);
		sr->draw_next_frame = 1;
		break;
	/*aspect ratio modif*/
	case 3:
		sr->visual_renderer->RecomputeAR(sr->visual_renderer);
		break;
	/*size changed*/
	case 4:
		SR_SetOutputSize(sr, sr->new_width, sr->new_height);
		sr->new_width = sr->new_height = 0;
		break;
	/*size changed from scene cfg: resize window first*/
	case 5:
		{
			M4Event evt;
			/*send resize event*/
			evt.type = M4E_NEEDRESIZE;
			evt.size.width = sr->new_width;
			evt.size.height = sr->new_height;
			sr->video_out->PushEvent(sr->video_out, &evt);
			SR_SetOutputSize(sr, sr->new_width, sr->new_height);
			sr->new_width = sr->new_height = 0;
		}
		break;
	}
	/*3D driver changed message, recheck extensions*/
	if (sr->msg_type && sr->reset_graphics) sr->visual_renderer->GraphicsReset(sr->visual_renderer);
	sr->msg_type = 0;
	SR_Lock(sr, 0);
}

M4Err SR_RenderFrame(LPSCENERENDER sr)
{	
	SR_ReconfigTask(sr);
	/*render*/
	SR_SimulationTick(sr);
	return M4OK;
}

u32 SR_RenderRun(void *par)
{	
	LPSCENERENDER sr = (LPSCENERENDER) par;
	sr->video_th_state = 1;

	while (sr->video_th_state == 1) {
		/*sleep or render*/
		if (sr->is_hidden) 
			Sleep(sr->frame_duration);
		else
			SR_RenderFrame(sr);
	}

	/*destroy video out*/
	sr->video_out->Shutdown(sr->video_out);
	PM_ShutdownInterface(sr->video_out);
	sr->video_out = NULL;

	sr->video_th_state = 3;
	return 0;
}

/*forces graphics redraw*/
void SR_ResetGraphics(LPSCENERENDER sr)
{	
	if (sr) {
		sr->reset_graphics = 1;
		sr->draw_next_frame = 1;
	}
}

void SR_ResetFrameRate(SceneRenderer *sr)
{
	u32 i;
	for (i=0; i<FPS_COMPUTE_SIZE; i++) sr->frame_time[i] = 0;
	sr->current_frame = 0;
}

void SR_SetFontEngine(SceneRenderer *sr);
void SR_OnEvent(void *cbck, M4Event *event);


static LPSCENERENDER SR_New(M4User *client)
{
	char *sOpt;
	M4GLConfig cfg, *gl_cfg;
	LPSCENERENDER tmp = (LPSCENERENDER) malloc(sizeof(SceneRenderer));
	if (!tmp) return NULL;
	memset(tmp, 0, sizeof(SceneRenderer));
	tmp->client = client;
	tmp->clear_color.red = tmp->clear_color.green = tmp->clear_color.blue = 0;

	/*load renderer to check for GL flag*/
	sOpt = IF_GetKey(client->config, "Rendering", "RendererName");
	if (sOpt) {
		if (!PM_LoadInterfaceByName(client->plugins, sOpt, M4_RENDERER_INTERFACE, (void **) &tmp->visual_renderer) ) {
			tmp->visual_renderer = NULL;
			sOpt = NULL;
		}
	}
	if (!tmp->visual_renderer) {
		u32 i, count;
		count = PM_GetPluginsCount(client->plugins);
		for (i=0; i<count; i++) {
			if (PM_LoadInterface(client->plugins, i, M4_RENDERER_INTERFACE, (void **) &tmp->visual_renderer)) break;
			tmp->visual_renderer = NULL;
		}
		if (tmp->visual_renderer) IF_SetKey(client->config, "Rendering", "RendererName", tmp->visual_renderer->plugin_name);
	}

	if (!tmp->visual_renderer) {
		free(tmp);
		return NULL;
	}

	memset(&cfg, 0, sizeof(cfg));
	cfg.double_buffered = 1;
	gl_cfg = tmp->visual_renderer->bNeedsGL ? &cfg : NULL;

	/*load video out*/
	sOpt = IF_GetKey(client->config, "Video", "DriverName");
	if (sOpt) {
		if (!PM_LoadInterfaceByName(client->plugins, sOpt, M4_VIDEO_OUTPUT_INTERFACE, (void **) &tmp->video_out)) {
			tmp->video_out = NULL;
			sOpt = NULL;
		} else {
			tmp->video_out->evt_cbk_hdl = tmp;
			tmp->video_out->on_event = SR_OnEvent;
			/*init hw*/
			if (tmp->video_out->SetupHardware(tmp->video_out, client->os_window_handler, client->dont_override_window_proc, gl_cfg) != M4OK) {
				PM_ShutdownInterface(tmp->video_out);
				tmp->video_out = NULL;
			}
		}
	}

	if (!tmp->video_out) {
		u32 i, count;
		count = PM_GetPluginsCount(client->plugins);
		for (i=0; i<count; i++) {
			if (!PM_LoadInterface(client->plugins, i, M4_VIDEO_OUTPUT_INTERFACE, (void **) &tmp->video_out)) continue;
			tmp->video_out->evt_cbk_hdl = tmp;
			tmp->video_out->on_event = SR_OnEvent;
			/*init hw*/
			if (tmp->video_out->SetupHardware(tmp->video_out, client->os_window_handler, client->dont_override_window_proc, gl_cfg)==M4OK) {
				IF_SetKey(client->config, "Video", "DriverName", tmp->video_out->plugin_name);
				break;
			}
			PM_ShutdownInterface(tmp->video_out);
			tmp->video_out = NULL;
		}
	}

	if (!tmp->video_out ) {
		PM_ShutdownInterface(tmp->visual_renderer);
		free(tmp);
		return NULL;
	}

	/*and init*/
	if (tmp->visual_renderer->LoadRenderer(tmp->visual_renderer, tmp) != M4OK) {
		PM_ShutdownInterface(tmp->visual_renderer);
		tmp->video_out->Shutdown(tmp->video_out);
		PM_ShutdownInterface(tmp->video_out);
		free(tmp);
		return NULL;
	}

	tmp->mx = NewMutex();
	tmp->textures = NewChain();
	tmp->frame_rate = 30.0;	
	tmp->frame_duration = 33;
	tmp->time_nodes = NewChain();
	tmp->events = NewChain();
	tmp->ev_mx = NewMutex();
	
	SR_ResetFrameRate(tmp);	
	/*set font engine if any*/
	SR_SetFontEngine(tmp);

	tmp->base_curve_resolution = DEFAULT_CURVE_RESOLUTION;

	tmp->interaction_level = M4_InteractNormal | M4_InteractInputSensor | M4_InteractWM;
	return tmp;
}

LPSCENERENDER NewSceneRender(M4User *client, Bool self_threaded, Bool no_audio, MPEG4CLIENT term)
{
	LPSCENERENDER tmp = SR_New(client);
	if (!tmp) return NULL;
	tmp->term = term;

	if (!no_audio) tmp->audio_renderer = LoadAudioRenderer(client);	

	/*run threaded*/
	if (self_threaded) {
		tmp->VisualThread = NewThread();
		TH_Run(tmp->VisualThread, SR_RenderRun, tmp);
	}
	/*set default size if owning output*/
	if (!tmp->client->os_window_handler) SR_SetSize(tmp, 100, 100);
	return tmp;
}


void SR_Delete(LPSCENERENDER sr)
{
	if (!sr) return;

	SR_Lock(sr, 1);
	if (sr->VisualThread) {
		sr->video_th_state = 2;
		while (sr->video_th_state!=3) Sleep(10);
		TH_Delete(sr->VisualThread);
	}

	if (sr->video_out) {
		sr->video_out->Shutdown(sr->video_out);
		PM_ShutdownInterface(sr->video_out);
	}
	sr->visual_renderer->UnloadRenderer(sr->visual_renderer);
	PM_ShutdownInterface(sr->visual_renderer);

	if (sr->audio_renderer) AR_Delete(sr->audio_renderer);

	MX_P(sr->ev_mx);
	while (ChainGetCount(sr->events)) {
		UserEvent *ev = ChainGetEntry(sr->events, 0);
		ChainDeleteEntry(sr->events, 0);
		free(ev);
	}
	MX_V(sr->ev_mx);
	MX_Delete(sr->ev_mx);
	DeleteChain(sr->events);

	if (sr->font_engine) {
		sr->font_engine->shutdown_font_engine(sr->font_engine);
		PM_ShutdownInterface(sr->font_engine);
	}
	DeleteChain(sr->textures);
	DeleteChain(sr->time_nodes);

	SR_Lock(sr, 0);
	MX_Delete(sr->mx);
	free(sr);
}

void SR_SetFrameRate(LPSCENERENDER sr, Double fps)
{
	if (fps) {
		sr->frame_rate = fps;
		sr->frame_duration = (u32) (1000 / fps);
		SR_ResetFrameRate(sr);	
	}
}

void SR_Pause(LPSCENERENDER sr, Bool DoPause)
{
	if (!sr || !sr->audio_renderer) return;
	/*step mode*/
	if (DoPause==2) {
		sr->step_mode = 1;
		/*resume for next step*/
		if (sr->paused && sr->term) M4T_Pause(sr->term, 0);
	} else if (sr->paused != DoPause) {
		if (sr->audio_renderer) AR_Pause(sr->audio_renderer, DoPause, sr->step_mode);
		sr->paused = DoPause;
	}
}

u32 SR_GetTime(LPSCENERENDER sr)
{
	return AR_GetTime(sr->audio_renderer);
}

static M4Err SR_SetSceneSize(LPSCENERENDER sr, u32 Width, u32 Height)
{
	if (!Width || !Height) {
		sr->has_size_info = 0;
		if (sr->override_size_flags) {
			/*specify a small size to detect biggest bitmap but not 0 in case only audio..*/
			sr->scene_height = 20;
			sr->scene_width = 320;
		} else {
			sr->scene_height = 240;
			sr->scene_width = 320;
		}
	} else {
		sr->has_size_info = 1;
		sr->scene_height = Height;
		sr->scene_width = Width;
	}
	return M4OK;
}


M4Err SR_SetSceneGraph(LPSCENERENDER sr, LPSCENEGRAPH scene_graph)
{
	u32 width, height;

	if (!sr) return M4BadParam;

	SR_Lock(sr, 1);

	if (sr->audio_renderer) AR_ResetSources(sr->audio_renderer);
	MX_P(sr->ev_mx);
	while (ChainGetCount(sr->events)) {
		UserEvent *ev = ChainGetEntry(sr->events, 0);
		ChainDeleteEntry(sr->events, 0);
		free(ev);
	}
	
	/*reset main surface*/
	sr->visual_renderer->SceneReset(sr->visual_renderer);

	/*set current graph*/
	sr->scene = scene_graph;
	
	if (scene_graph) {
		sr->reset_graphics = 1;

		/*get pixel size if any*/
		SG_GetSizeInfo(sr->scene, &width, &height);
		/*WATCHOUT: since BIFS coord system origin is the center of the screen we need EVEN diemnsions
		in order to unambiguously compute pixel areas in both systems (BIFS and video mem, which is center on top-left)
		we use a smaller size to avoid render surprises...*/
		if (width%2) width--;
		if (height%2) height--;

		/*set scene size only if different, otherwise keep scaling/FS*/
		if ( !width || (sr->scene_width!=width) || !height || (sr->scene_height!=height)) {
			sr->is_pixel_metrics = SG_UsePixelMetrics(sr->scene);
			SR_SetSceneSize(sr, width, height);
			/*get actual size in pixels*/
			width = sr->scene_width;
			height = sr->scene_height;

			/*only notify user if we are attached to a window*/
			if (sr->client->os_window_handler) {
				M4USER_SETSIZE(sr->client, width, height);
				sr->override_size_flags &= ~2;
			} else {
				/*signal size changed*/
				SR_SetSize(sr,width, height);
			}
		}
	}

	SR_ResetFrameRate(sr);	
	MX_V(sr->ev_mx);
	SR_Lock(sr, 0);
	return M4OK;
}

void SR_Lock(LPSCENERENDER sr, Bool doLock)
{
	if (doLock)
		MX_P(sr->mx);
	else {
		MX_V(sr->mx);
	}
}

void SR_RefreshWindow(LPSCENERENDER sr)
{
	if (sr) sr->draw_next_frame = 1;
}

M4Err SR_SizeChanged(LPSCENERENDER sr, u32 NewWidth, u32 NewHeight)
{
	if (!NewWidth || !NewHeight) {
		sr->override_size_flags &= ~2;
		return M4OK;
	}
	SR_Lock(sr, 1);
	sr->new_width = NewWidth;
	sr->new_height = NewHeight;
	sr->msg_type = 4;
	SR_Lock(sr, 0);
	return M4OK;
}

M4Err SR_SetSize(LPSCENERENDER sr, u32 NewWidth, u32 NewHeight)
{
	if (!NewWidth || !NewHeight) {
		sr->override_size_flags &= ~2;
		return M4OK;
	}
	SR_Lock(sr, 1);
	sr->new_width = NewWidth;
	sr->new_height = NewHeight;
	sr->msg_type = 5;
	SR_Lock(sr, 0);
	/*force video resize*/
	if (!sr->VisualThread) SR_RenderFrame(sr);
	return M4OK;
}

void SR_ReloadConfig(LPSCENERENDER sr)
{
	const char *sOpt, *dr_name;

	/*changing drivers needs exclusive access*/
	SR_Lock(sr, 1);
	
	sOpt = IF_GetKey(sr->client->config, "Rendering", "BackColor");
	if (sOpt) SR_SetOption(sr, M4O_ClearColor, atoi(sOpt));
	
	sOpt = IF_GetKey(sr->client->config, "Rendering", "ForceSceneSize");
	if (sOpt && ! stricmp(sOpt, "yes")) {
		sr->override_size_flags = 1;
	} else {
		sr->override_size_flags = 0;
	}

	sOpt = IF_GetKey(sr->client->config, "Rendering", "AntiAlias");
	if (sOpt) {
		if (! stricmp(sOpt, "None")) SR_SetOption(sr, M4O_Antialias, M4_AL_None);
		else if (! stricmp(sOpt, "Text")) SR_SetOption(sr, M4O_Antialias, M4_AL_Text);
		else SR_SetOption(sr, M4O_Antialias, M4_AL_All);
	} else {
		IF_SetKey(sr->client->config, "Rendering", "AntiAlias", "All");
		SR_SetOption(sr, M4O_Antialias, M4_AL_All);
	}

	sOpt = IF_GetKey(sr->client->config, "Rendering", "StressMode");
	SR_SetOption(sr, M4O_StressMode, (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0);

	sOpt = IF_GetKey(sr->client->config, "Rendering", "FastRender");
	SR_SetOption(sr, M4O_HighSpeed, (sOpt && !stricmp(sOpt, "yes") ) ? 1 : 0);

	sOpt = IF_GetKey(sr->client->config, "Rendering", "BoundingVolume");
	if (sOpt) {
		if (! stricmp(sOpt, "Box")) SR_SetOption(sr, M4O_BoundingVolume, M4_Bounds_Box);
		else if (! stricmp(sOpt, "Sphere")) SR_SetOption(sr, M4O_BoundingVolume, M4_Bounds_Sphere);
		else SR_SetOption(sr, M4O_BoundingVolume, M4_Bounds_None);
	} else {
		IF_SetKey(sr->client->config, "Rendering", "BoundingVolume", "None");
		SR_SetOption(sr, M4O_BoundingVolume, M4_Bounds_None);
	}

	sOpt = IF_GetKey(sr->client->config, "FontEngine", "DriverName");
	if (sOpt && sr->font_engine) {
		dr_name = sr->font_engine->plugin_name;
		if (stricmp(dr_name, sOpt)) SR_SetFontEngine(sr);
	}

	sOpt = IF_GetKey(sr->client->config, "Rendering", "DisableZoomPan");
	if (sOpt && !stricmp(sOpt, "yes")) 
		sr->interaction_level &= ~M4_InteractZoomPan;
	else
		sr->interaction_level |= M4_InteractZoomPan;

	sr->draw_next_frame = 1;

	SR_Lock(sr, 0);
}

M4Err SR_SetOption(LPSCENERENDER sr, u32 type, u32 value)
{
	switch (type) {
	case M4O_AudioVolume: AR_SetVolume(sr->audio_renderer, value); break;
	case M4O_AudioPan: AR_SetPan(sr->audio_renderer, value); break;
	case M4O_OverrideSize:
		SR_Lock(sr, 1);
		sr->override_size_flags = value ? 1 : 0;
		sr->draw_next_frame = 1;
		SR_Lock(sr, 0);
		break;
	case M4O_ClearColor:
		sr->clear_color.red = ((Float) M4C_R(value))/0xFF;
		sr->clear_color.green = ((Float) M4C_G(value))/0xFF;
		sr->clear_color.blue = ((Float) M4C_B(value))/0xFF;
		break;
	case M4O_StressMode: sr->stress_mode = value; break;
	case M4O_Antialias: sr->antiAlias = value; break;
	case M4O_HighSpeed: sr->high_speed = value; break;
	case M4O_BoundingVolume: sr->draw_bvol = value; break;
	case M4O_AspectRatio: 
		sr->aspect_ratio = value; 
		sr->msg_type = 3;
		break;
	case M4O_InteractLevel: sr->interaction_level = value; return M4OK;
	case M4O_ForceRedraw:
		SR_Lock(sr, 1);
		sr->reset_graphics = 1;
		SR_Lock(sr, 0);
		break;
	case M4O_Fullscreen:
		if (sr->fullscreen != value) sr->msg_type = 2;
		return M4OK;
	case M4O_OriginalView:
	{
		M4Err ret = sr->visual_renderer->SetOption(sr->visual_renderer, type, value);
		SR_SetSize(sr, sr->scene_width, sr->scene_height);
		return ret;
	}
	case M4O_Visible:
		sr->is_hidden = !value;
		if (sr->video_out->PushEvent) {
			M4Event evt;
			evt.type = M4E_SHOWHIDE;
			evt.show.is_visible = value;
			return sr->video_out->PushEvent(sr->video_out, &evt);
		}
		return M4OK;

	case M4O_ReloadConfig:
		SR_ReloadConfig(sr);
	default: return sr->visual_renderer->SetOption(sr->visual_renderer, type, value);
	}

	sr->draw_next_frame = 1; 
	return M4OK;
}

u32 SR_GetOption(LPSCENERENDER sr, u32 type)
{
	switch (type) {
	case M4O_OverrideSize: return (sr->override_size_flags & 1) ? 1 : 0;
	case M4O_IsOver:
	{
		if (sr->interaction_sensors) return 0;
		if (ChainGetCount(sr->time_nodes)) return 0;
		return 1;
	}
	case M4O_ClearColor:
		return MAKE_ARGB_FLOAT(1.0, sr->clear_color.red, sr->clear_color.green, sr->clear_color.blue);
	case M4O_StressMode: return sr->stress_mode;
	case M4O_Antialias: return sr->antiAlias;
	case M4O_HighSpeed: return sr->high_speed;
	case M4O_AspectRatio: return sr->aspect_ratio;
	case M4O_Fullscreen: return sr->fullscreen;
	case M4O_InteractLevel: return sr->interaction_level;
	case M4O_Visible: return !sr->is_hidden;
	default: return sr->visual_renderer->GetOption(sr->visual_renderer, type);
	}
}

void SR_MapCoordinates(LPSCENERENDER sr, s32 X, s32 Y, Float *bifsX, Float *bifsY)
{
	*bifsX = (Float) X;
	*bifsY = (Float) Y;
}


void SR_SetFontEngine(SceneRenderer *sr)
{
	const char *sOpt;
	u32 i, count;
	FontRaster *ifce;

	ifce = NULL;
	sOpt = IF_GetKey(sr->client->config, "FontEngine", "DriverName");
	if (sOpt) {
		if (!PM_LoadInterfaceByName(sr->client->plugins, sOpt, M4_FONT_RASTER_INTERFACE, (void **) &ifce)) 
			ifce = NULL;
	}

	if (!ifce) {
		count = PM_GetPluginsCount(sr->client->plugins);
		for (i=0; i<count; i++) {
			if (PM_LoadInterface(sr->client->plugins, i, M4_FONT_RASTER_INTERFACE, (void **) &ifce)) {
				IF_SetKey(sr->client->config, "FontEngine", "DriverName", ifce->plugin_name);
				sOpt = ifce->plugin_name;
				break;
			}
		}
	}
	if (!ifce) return;

	/*cannot init font engine*/
	if (ifce->init_font_engine(ifce) != M4OK) {
		PM_ShutdownInterface(ifce);
		return;
	}


	/*shutdown current*/
	SR_Lock(sr, 1);
	if (sr->font_engine) {
		sr->font_engine->shutdown_font_engine(sr->font_engine);
		PM_ShutdownInterface(sr->font_engine);
	}
	sr->font_engine = ifce;

	/*success*/
	IF_SetKey(sr->client->config, "FontEngine", "DriverName", sOpt);
		
	sr->draw_next_frame = 1;
	SR_Lock(sr, 0);
}

M4Err SR_GetScreenBuffer(LPSCENERENDER sr, M4VideoSurface *framebuffer)
{
	M4Err e;
	if (!sr || !framebuffer) return M4BadParam;
	MX_P(sr->mx);
	e = sr->visual_renderer->GetScreenBuffer(sr->visual_renderer, framebuffer);
	if (e != M4OK) MX_V(sr->mx);
	return e;
}

M4Err SR_ReleaseScreenBuffer(LPSCENERENDER sr, M4VideoSurface *framebuffer)
{
	M4Err e;
	if (!sr || !framebuffer) return M4BadParam;
	e = sr->visual_renderer->ReleaseScreenBuffer(sr->visual_renderer, framebuffer);
	MX_V(sr->mx);
	return e;
}

Float SR_GetCurrentFPS(LPSCENERENDER sr, Bool absoluteFPS)
{
	Float fps;
	u32 ind, num, frames, run_time;

	/*start from last frame and get first frame time*/
	ind = sr->current_frame;
	frames = 0;
	run_time = sr->frame_time[ind];
	for (num=0; num<FPS_COMPUTE_SIZE; num++) {
		if (absoluteFPS) {
			run_time += sr->frame_time[ind];
		} else {
			run_time += MAX(sr->frame_time[ind], sr->frame_duration);
		}
		frames++;
		if (frames==FPS_COMPUTE_SIZE) break;
		if (!ind) {
			ind = FPS_COMPUTE_SIZE;
		} else {
			ind--;
		}
	}
	if (!run_time) return (Float) sr->frame_rate;
	fps = (Float) (1000*frames);
	fps /= (Float) (run_time);
	return fps;
}

void SR_RegisterTimeNode(LPSCENERENDER sr, TimeNode *tn)
{
	/*may happen with DEF/USE */
	if (tn->is_registered) return;
	if (tn->needs_unregister) return;
	ChainAddEntry(sr->time_nodes, tn);
	tn->is_registered = 1;
}
void SR_UnregisterTimeNode(LPSCENERENDER sr, TimeNode *tn)
{
	ChainDeleteItem(sr->time_nodes, tn);
}


void SR_UserInput(LPSCENERENDER sr, M4EventMouse *event)
{
	UserEvent *ev;

	if (sr->term && (sr->interaction_level & M4_InteractInputSensor) && (event->type!=M4E_VKEYDOWN) && (event->type!=M4E_VKEYUP))
		M4T_MouseInput(sr->term, event);

	if (!(sr->interaction_level & M4_InteractNormal)) return;

	switch (event->type) {
	case M4E_MOUSEMOVE:
	{
		u32 i;
		MX_P(sr->ev_mx);
		for (i=0; i<ChainGetCount(sr->events); i++) {
			ev = ChainGetEntry(sr->events, i);
			if (ev->event_type == M4E_MOUSEMOVE) {
				sr->visual_renderer->MapCoordsToAR(sr->visual_renderer, event->x, event->y, &ev->x, &ev->y);
				MX_V(sr->ev_mx);
				return;
			}
		}
		MX_V(sr->ev_mx);
	}
	default:
		ev = malloc(sizeof(UserEvent));
		ev->event_type = event->type;
		MX_P(sr->ev_mx);
		/*apply AR to coordinates*/
		sr->visual_renderer->MapCoordsToAR(sr->visual_renderer, event->x, event->y, &ev->x, &ev->y);
		ChainAddEntry(sr->events, ev);
		MX_V(sr->ev_mx);
		break;
	}
}

SFNode *SR_PickNode(LPSCENERENDER sr, s32 X, s32 Y)
{
	return NULL;
}

void SR_SimulationTick(LPSCENERENDER sr)
{	
	u32 in_time, end_time, i, count;
	SFNode *top_node;

	SR_Lock(sr, 1);

	if (!sr->scene) {
		SR_Lock(sr, 0);
		Sleep((u32) ( (Double) 1000/sr->frame_rate) );
		return;
	}

	in_time = M4_GetSysClock();
	if (sr->reset_graphics) sr->draw_next_frame = 1;

	top_node = SG_GetRootNode(sr->scene);

	/*process pending user events*/
	MX_P(sr->ev_mx);
	while (ChainGetCount(sr->events)) {
		UserEvent *ev = ChainGetEntry(sr->events, 0);
		ChainDeleteEntry(sr->events, 0);
		sr->visual_renderer->ExecuteEvent(sr->visual_renderer, ev->event_type, ev->x, ev->y);
		free(ev);
	}
	MX_V(sr->ev_mx);

	/*execute all routes before updating textures, otherwise nodes inside composite texture may never see their
	dirty flag set*/
	SG_ActivateRoutes(sr->scene);

	/*update all textures*/
	count = ChainGetCount(sr->textures);
	for (i=0; i<count; i++) {
		TextureHandler *st = ChainGetEntry(sr->textures, i);
		/*signal graphics reset before updating*/
		if (sr->reset_graphics && st->hwtx) sr->visual_renderer->TextureHWReset(st);
		st->update_texture_fcnt(st);
	}

	/*if invalidated, draw*/
	if (sr->draw_next_frame) {
		sr->draw_next_frame = 0;
		if (sr->reset_graphics) 
			Node_SetDirtyChildren(top_node);
		sr->visual_renderer->DrawScene(sr->visual_renderer, top_node);

		if (sr->stress_mode) {
			sr->draw_next_frame = 1;
			sr->reset_graphics = 1;
		} else {
			sr->reset_graphics = 0;
		}
	}

	/*update all timed nodes*/
	for (i=0; i<ChainGetCount(sr->time_nodes); i++) {
		TimeNode *tn = ChainGetEntry(sr->time_nodes, i);
		if (!tn->needs_unregister) tn->UpdateTimeNode(tn);
		if (tn->needs_unregister) {
			tn->is_registered = 0;
			tn->needs_unregister = 0;
			ChainDeleteEntry(sr->time_nodes, i);
			i--;
			continue;
		}
	}

	/*release all textures - we must release them to handle a same OD being used by several textures*/
	count = ChainGetCount(sr->textures);
	for (i=0; i<count; i++) {
		TextureHandler *st = ChainGetEntry(sr->textures, i);
		texture_release_stream(st);
	}
	end_time = M4_GetSysClock() - in_time;

	SR_Lock(sr, 0);

	sr->current_frame = (sr->current_frame+1) % FPS_COMPUTE_SIZE;
	sr->frame_time[sr->current_frame] = end_time;

	/*step mode on, pause and return*/
	if (sr->step_mode) {
		sr->step_mode = 0;
		if (sr->term) M4T_Pause(sr->term, 1);
		return;
	}
	/*not threaded, let the owner decide*/
	if (!sr->VisualThread) return;


	/*compute sleep time till next frame, otherwise we'll kill the CPU*/
	i=1;
	while (i * sr->frame_duration < end_time) i++;
	in_time = i * sr->frame_duration - end_time;
	Sleep(in_time);
}
