/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / codec pack plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/m4_decoder.h>

#define DEFAULT_VIDEO_CM_SIZE			4
/*default critical size in CU of composition memory for audio*/
#define DEFAULT_VIDEO_CM_TRIGGER		1


#include "./divx/decore.h"
#include <gpac/m4_author.h>

/*beerk*/
static int odivx_num_instances = 0;

typedef struct
{
	Bool codec_loaded;
	/*no support for scalability in XVID yet*/
	u16 ES_ID;
	u32 width, height, out_size;
	u32 cb_size, cb_trig;
	u32 y_size, uv_size;

	Bool first_frame;
	M4Err last_err;
	char *dsi;
	u32 dsi_size;
} ODIVXDec;

#define ODIVXCTX()	ODIVXDec *ctx = (ODIVXDec *) (ifcg->privateStack)


static M4Err ODIVX_AttachStream(DecoderInterface *ifcg, u16 ES_ID, unsigned char *decSpecInfo, u32 decSpecInfoSize, u16 DependsOnES_ID, u32 objectTypeIndication, Bool UpStream)
{
	M4VDecoderSpecificInfo dsi;
	M4Err e;
	DEC_PARAM par;
	ODIVXCTX();

	if (ctx->ES_ID && ctx->ES_ID!=ES_ID) return M4NotSupported;
	if (!decSpecInfoSize || !decSpecInfo || ctx->codec_loaded) return M4NonCompliantBitStream;

	/*decode DSI*/
	e = M4V_GetConfig(decSpecInfo, decSpecInfoSize, &dsi);
	if (e) return e;

	if (!dsi.width || !dsi.height) return M4NonCompliantBitStream;

	memset(&par, 0, sizeof(DEC_PARAM));
	par.x_dim = dsi.width;
	par.y_dim = dsi.height;
	par.color_depth = 0;
	par.output_format = YUV12;
	par.dither = 0;

	if (DecodeInit((u32) ctx, DEC_OPT_INIT, &par, NULL) != DEC_OK) return M4NonCompliantBitStream;

	ctx->width = par.x_dim;
	ctx->height = par.y_dim;
	ctx->dsi = malloc(sizeof(char) * decSpecInfoSize);
	memcpy(ctx->dsi, decSpecInfo, sizeof(char) * decSpecInfoSize);
	ctx->dsi_size = decSpecInfoSize;
	ctx->first_frame = 1;

	ctx->codec_loaded = 1;
	ctx->ES_ID = ES_ID;	
	/*output in YV12 only - let the player handle conversion*/
	ctx->out_size = ctx->width * ctx->height * 3 / 2;
	ctx->y_size = ctx->width * ctx->height;
	ctx->uv_size = ctx->width * ctx->height / 4;
	ctx->last_err = M4OK;
	return M4OK;
}
static M4Err ODIVX_DetachStream(DecoderInterface *ifcg, u16 ES_ID)
{
	ODIVXCTX();
	if (ctx->ES_ID != ES_ID) return M4BadParam;
	if (ctx->codec_loaded) DecodeInit((u32) ctx, DEC_OPT_RELEASE, NULL, NULL);
	ctx->codec_loaded = 0;
	ctx->ES_ID = 0;
	ctx->width = ctx->height = ctx->out_size = 0;
	if (ctx->dsi) free(ctx->dsi);
	ctx->dsi = NULL;
	ctx->dsi_size = 0;
	ctx->last_err = M4OK;
	return M4OK;
}
static M4Err ODIVX_GetCapabilities(DecoderInterface *ifcg, CapObject *capability)
{
	ODIVXCTX();

	switch (capability->CapCode) {
	case CAP_WIDTH:
		capability->cap.valueINT = ctx->width;
		break;
	case CAP_HEIGHT:
		capability->cap.valueINT = ctx->height;
		break;
	case CAP_PITCH:
		capability->cap.valueINT = ctx->width;
		break;
	case CAP_OUTPUTSIZE:
		capability->cap.valueINT = ctx->out_size;
		break;
	case CAP_COLORMODE:
		capability->cap.valueINT = M4PF_YV12;
		break;
	/*no postprocessing API for xvid*/
	case CAP_VID_POSTPROC:
		capability->cap.valueINT = 0;
		break;
	case CAP_BUFFER_MIN:
		capability->cap.valueINT = ctx->cb_trig;
		break;
	case CAP_BUFFER_MAX:
		capability->cap.valueINT = ctx->cb_size;
		break;
	/*no support for shape coding*/
	case CAP_VID_SHAPE:
		capability->cap.valueINT = 0;
		break;
	/*by default we use 4 bytes padding (otherwise it happens that XviD crashes on some videos...)*/
	case CAP_PADDING_BYTES:
		capability->cap.valueINT = 0;
		break;
	/*not known at our level...*/
	case CAP_CU_DURATION:
	default:
		capability->cap.valueINT = 0;
		break;
	}
	return M4OK;
}
static M4Err ODIVX_SetCapabilities(DecoderInterface *ifcg, CapObject capability)
{
	ODIVXCTX();
	switch (capability.CapCode) {
	case CAP_WAIT_RAP:
		ctx->first_frame = 1;
		ctx->last_err = M4OK;
		return M4OK;
	default:
		/*return unsupported to avoid confusion by the player (like color space changing ...) */
		return M4NotSupported;
	}
}
static M4Err ODIVX_Process(DecoderInterface *ifcg, 
		unsigned char *inBuffer, u32 inBufferLength,
		u16 ES_ID,
		unsigned char *outBuffer, u32 *outBufferLength,
		Bool isRAP, u8 PaddingBits, u32 mmlevel)
{
	AVPicture pict;
	u32 i, uv_w, half_h;
	ODIVXCTX();

	/*check not using scalabilty*/
	if (ES_ID != ctx->ES_ID) return M4BadParam;

	if (*outBufferLength < ctx->out_size) {
		*outBufferLength = ctx->out_size;
		return M4BufferTooSmall;
	}
	/*safety: if the codec cannot decode a frame forbid any further decode*/
//	if (ctx->last_err) return ctx->last_err;

	if (ctx->first_frame) {
		char *data = malloc(sizeof(char) * (ctx->dsi_size + inBufferLength));
		memcpy(data, ctx->dsi, sizeof(char) * ctx->dsi_size);
		memcpy(data + ctx->dsi_size, inBuffer, sizeof(char) * inBufferLength);
		if (!DecodeFrame(data, ctx->dsi_size + inBufferLength, (unsigned char *) &pict, 0) ) {
			*outBufferLength = 0;
			return (ctx->last_err = M4NonCompliantBitStream);
		}
		free(data);
		ctx->first_frame = 0;
	} else {
		if (!DecodeFrame(inBuffer, inBufferLength, (unsigned char *) &pict, 0) ) {
			*outBufferLength = 0;
			return (ctx->last_err = M4NonCompliantBitStream);
		}
	}

	/*dispatch nothing if seeking or droping*/
	switch (mmlevel) {
	case MM_LEVEL_SEEK:
	case MM_LEVEL_DROP:
		*outBufferLength = 0;
		break;
	default:
		*outBufferLength = ctx->out_size;
		/*copy YUV data (realign to real YV12, DivX is using padding blocks)*/
		for (i=0; i<ctx->height; i++) {
			char *src = pict.data[0] + pict.linesize[0]*i;
			char *dst = outBuffer + ctx->width*i;
			memcpy(dst, src, sizeof(char) * ctx->width);
		}
		outBuffer += ctx->y_size;

		half_h = ctx->height/2;
		uv_w = ctx->width/2;
		for (i=0; i<half_h; i++) {
			char *src = pict.data[1] + pict.linesize[1]*i;
			char *dst = outBuffer + i*uv_w;
			memcpy(dst, src, sizeof(char) * uv_w);
		}
		outBuffer += ctx->uv_size;
		for (i=0; i<half_h; i++) {
			char *src = pict.data[2] + pict.linesize[2]*i;
			char *dst = outBuffer + i*uv_w;
			memcpy(dst, src, sizeof(char) * uv_w);
		}
		break;
	}
	return M4OK;
}

static u32 ODIVX_CanHandleStream(DecoderInterface *dec, u32 StreamType, u32 ObjectType, unsigned char *decSpecInfo, u32 decSpecInfoSize, u32 PL)
{
	if (!ObjectType) {
		if (StreamType==M4ST_VISUAL) return 1;
		return 0;
	}

	if ((StreamType==M4ST_VISUAL) && (ObjectType==0x20)) return 1;
	return 0;
}

static const char *ODIVX_GetCodecName(DecoderInterface *dec) { return "OpenDivX";}

DecoderInterface *NewDecoderInterface()
{
	ODIVXDec *dec;
	DecoderInterface *ifce;
	ifce = malloc(sizeof(DecoderInterface));
	memset(ifce, 0, sizeof(DecoderInterface));
	ifce->CanHandleStream = ODIVX_CanHandleStream;
	M4_REG_PLUG(ifce, M4MEDIADECODERINTERFACE, "OpenDivX Decoder", "gpac distribution", 0)


	odivx_num_instances++;
	dec = (ODIVXDec *) malloc(sizeof(ODIVXDec));
	memset(dec, 0, sizeof(ODIVXDec));

	dec->cb_size = DEFAULT_VIDEO_CM_SIZE;
	dec->cb_trig = DEFAULT_VIDEO_CM_TRIGGER;

	/*setup our own interface*/	
	ifce->Codec_AttachStream = ODIVX_AttachStream;
	ifce->Codec_DetachStream = ODIVX_DetachStream;
	ifce->Codec_GetCapabilities = ODIVX_GetCapabilities;
	ifce->Codec_SetCapabilities = ODIVX_SetCapabilities;
	ifce->Codec_Process = ODIVX_Process;
	ifce->GetCodecName = ODIVX_GetCodecName;
	
	ifce->privateStack = dec;
	return ifce;
}

void DeleteDecoderInterface(DecoderInterface *ifcg)
{
	ODIVXCTX();
	if (ctx->codec_loaded) DecodeInit((u32) ctx, DEC_OPT_RELEASE, NULL, NULL);
	odivx_num_instances--;
	free(ctx);
	free(ifcg);
}

Bool QueryInterface(u32 InterfaceType)
{
	switch (InterfaceType) {
	case M4MEDIADECODERINTERFACE:
		if (odivx_num_instances) return 0;
		return 1;
	default:
		return 0;
	}
}

void *LoadInterface(u32 InterfaceType)
{
	switch (InterfaceType) {
	case M4MEDIADECODERINTERFACE:
		return NewDecoderInterface();
	default:
		return NULL;
	}
}

void ShutdownInterface(void *ifce)
{
	DecoderInterface *ifcd = (DecoderInterface *)ifce;
	switch (ifcd->InterfaceType) {
	case M4MEDIADECODERINTERFACE:
		DeleteDecoderInterface(ifcd);
		break;
	}
}
