/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / DirectX audio and video render plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */


#include "dx_hw.h"

#include <process.h>

#define DSCONTEXT()		DSContext *ctx = (DSContext *)dr->opaque;

void DS_AudioRender(AudioOutput *dr);

void DS_GotoSleep(AudioOutput *dr);
void DS_GotoSleep_Notifs(AudioOutput *dr);


#define DS_DEFAULT_BUFFERS	8

static M4Err DS_SetupHardware(AudioOutput *dr, void *os_handle, u32 num_buffers, u32 num_buffer_per_sec)
{
	DWORD flags;
    HRESULT hr;

	DSCONTEXT();
	ctx->hWnd = (HWND) os_handle;
	/*check if we have created a HWND (this requires that video is handled by the DX plugin*/
	if (!ctx->hWnd) ctx->hWnd = DD_GetGlobalHWND();
	/*too bad, use desktop as window*/
	if (!ctx->hWnd) ctx->hWnd = GetDesktopWindow();

	ctx->force_config = (num_buffers && num_buffer_per_sec) ? 1 : 0;
	ctx->cfg_num_buffers = num_buffers;
	ctx->cfg_num_buffer_per_sec = num_buffer_per_sec;
	if (ctx->cfg_num_buffers <= 1) ctx->cfg_num_buffers = 2;

	if ( FAILED( hr = DirectSoundCreate( NULL, &ctx->pDS, NULL ) ) ) return M4IOErr;
	flags = DSSCL_EXCLUSIVE;
	if( FAILED( hr = IDirectSound_SetCooperativeLevel(ctx->pDS, ctx->hWnd, DSSCL_EXCLUSIVE) ) ) {
		SAFE_DS_RELEASE( ctx->pDS ); 
		return M4IOErr;
	}
	return M4OK;
}


void DS_ResetBuffer(DSContext *ctx)
{
    VOID *pLock = NULL;
    DWORD size;

    if( FAILED(IDirectSoundBuffer_Lock(ctx->pOutput, 0, ctx->total_buffer_size, &pLock, &size, NULL, NULL, 0 ) ) )
        return;
	memset(pLock, 0, (size_t) size);
	IDirectSoundBuffer_Unlock(ctx->pOutput, pLock, size, NULL, 0L); 
}

void DS_ReleaseBuffer(AudioOutput *dr)
{
	DSCONTEXT();

	/*stop playing and notif proc*/
	if (ctx->pOutput) IDirectSoundBuffer_Stop(ctx->pOutput);
	SAFE_DS_RELEASE(ctx->pOutput);
	
	/*use notif, shutdown notifier and event watcher*/
	if (ctx->use_notif) {
		CloseHandle(ctx->hEvent);
		ctx->hEvent = NULL;
	}
	ctx->use_notif = 0;
}

static void DS_Shutdown(AudioOutput *dr)
{
	DSCONTEXT();
	DS_ReleaseBuffer(dr);
	SAFE_DS_RELEASE(ctx->pDS ); 
}

/*we assume what was asked is what we got*/
static M4Err DS_ConfigureOutput(AudioOutput *dr, u32 *SampleRate, u32 *NbChannels, u32 *nbBitsPerSample)
{
    u32 i;
	HRESULT hr; 
	char *sOpt;
	DSBUFFERDESC dsbBufferDesc;
	IDirectSoundNotify *pNotify;
	DSCONTEXT();

	DS_ReleaseBuffer(dr);

	ctx->format.nChannels = *NbChannels;
	ctx->format.wBitsPerSample = *nbBitsPerSample;
	ctx->format.nSamplesPerSec = *SampleRate;
	ctx->format.cbSize = sizeof (WAVEFORMATEX);
	ctx->format.wFormatTag = WAVE_FORMAT_PCM;
	ctx->format.nBlockAlign = ctx->format.nChannels * ctx->format.wBitsPerSample / 8;
	ctx->format.nAvgBytesPerSec = ctx->format.nSamplesPerSec * ctx->format.nBlockAlign;

	if (!ctx->force_config) {
		ctx->buffer_size = ctx->format.nAvgBytesPerSec / 20;
		ctx->num_audio_buffer = DS_DEFAULT_BUFFERS;
	} else {
		ctx->num_audio_buffer = ctx->cfg_num_buffers;
		ctx->buffer_size = ctx->format.nAvgBytesPerSec / ctx->cfg_num_buffer_per_sec;
	}

	/*make sure we're aligned*/
	while (ctx->buffer_size % ctx->format.nBlockAlign) ctx->buffer_size++;
	ctx->total_buffer_size = ctx->buffer_size * ctx->num_audio_buffer;

	/*this gives weird results depending on sound cards*/
	ctx->use_notif = 0;
	sOpt = PMI_GetOpt(dr, "Audio", "UseNotification");
	if (sOpt && !stricmp(sOpt, "yes")) ctx->use_notif = 1;

	memset(&dsbBufferDesc, 0, sizeof(DSBUFFERDESC));
	dsbBufferDesc.dwSize = sizeof (DSBUFFERDESC);
	dsbBufferDesc.dwBufferBytes = ctx->total_buffer_size;
	dsbBufferDesc.lpwfxFormat = &ctx->format;
	dsbBufferDesc.dwFlags = DSBCAPS_GETCURRENTPOSITION2 | DSBCAPS_GLOBALFOCUS;
	if (ctx->use_notif) dsbBufferDesc.dwFlags |= DSBCAPS_CTRLPOSITIONNOTIFY;

	hr = IDirectSound_CreateSoundBuffer(ctx->pDS, &dsbBufferDesc, &ctx->pOutput, NULL );
	if (FAILED(hr)) {
retry:
		if (ctx->use_notif) PMI_SetOpt(dr, "Audio", "UseNotification", "no");
		ctx->use_notif = 0;
		dsbBufferDesc.dwFlags = DSBCAPS_GETCURRENTPOSITION2 | DSBCAPS_GLOBALFOCUS;
		hr = IDirectSound_CreateSoundBuffer(ctx->pDS, &dsbBufferDesc, &ctx->pOutput, NULL );
		if (FAILED(hr)) return M4IOErr;
	}

	ctx->buffer_length_ms = 1000 * ctx->buffer_size / ctx->format.nAvgBytesPerSec;
	ctx->total_length_in_ms = 1000 * ctx->total_buffer_size / ctx->format.nAvgBytesPerSec;
	ctx->next_write_pos = 0;

	if (ctx->use_notif) {
		hr = IDirectSoundBuffer_QueryInterface(ctx->pOutput, &IID_IDirectSoundNotify , (void **)&pNotify);
		if (hr == S_OK) {
			DSBPOSITIONNOTIFY *notify_positions;
			notify_positions = (DSBPOSITIONNOTIFY *)malloc(ctx->num_audio_buffer * sizeof(DSBPOSITIONNOTIFY));
			if (notify_positions) {
				/* Create event to handle DSound notifications */
				ctx->hEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
				/*setup the notification positions*/
				for( i = 0; i < ctx->num_audio_buffer; i++) {
					notify_positions[i].dwOffset = ctx->buffer_size * i;
					notify_positions[i].hEventNotify = ctx->hEvent;
				}

				/*Tell DirectSound when to notify us*/
				hr = IDirectSoundNotify_SetNotificationPositions(pNotify, ctx->num_audio_buffer, notify_positions);
				free(notify_positions);

				if (hr != S_OK) {
					IDirectSoundNotify_Release(pNotify);
					CloseHandle(ctx->hEvent);
					ctx->hEvent = NULL;
					SAFE_DS_RELEASE(ctx->pOutput);
					goto retry;
				}
			}
			IDirectSoundNotify_Release(pNotify);
		} else {
			ctx->use_notif = 0;
		}
	}
	if (ctx->use_notif) {
		dr->GotoSleep = DS_GotoSleep_Notifs;
		/*we're supposed to have "no delay" when using notifications, but we still indicate
		1ms to perform resync*/
		ctx->total_length_in_ms = 1;
	} else {
		dr->GotoSleep = DS_GotoSleep;
	}


	/*reset*/
	DS_ResetBuffer(ctx);
	/*play*/
	IDirectSoundBuffer_Play(ctx->pOutput, 0, 0, DSBPLAY_LOOPING );	
	return M4OK;
}

static Bool DS_RestoreBuffer(LPDIRECTSOUNDBUFFER pDSBuffer)
{
	DWORD dwStatus;
    IDirectSoundBuffer_GetStatus(pDSBuffer, &dwStatus );
    if( dwStatus & DSBSTATUS_BUFFERLOST ) {
		fprintf(stdout, "DS buffer lost\n");
		IDirectSoundBuffer_Restore(pDSBuffer);
	    IDirectSoundBuffer_GetStatus(pDSBuffer, &dwStatus);
		if( dwStatus & DSBSTATUS_BUFFERLOST ) return 1;
    }
	return 0;
}



void DS_AudioRender(AudioOutput *dr) 
{
	HRESULT hr;
    VOID *pLock;
    DWORD size;
	DSCONTEXT();

	/*restoring*/
    if (DS_RestoreBuffer(ctx->pOutput)) return;
	
	/*lock and fill from current pos*/
	pLock = NULL;
    if( FAILED( hr = IDirectSoundBuffer_Lock(ctx->pOutput, ctx->next_write_pos, ctx->buffer_size,  
			&pLock,  &size, NULL, NULL, 0L ) ) ) {
		fprintf(stdout, "Error locking DS buffer\n");
        return;
	}

	assert(size == ctx->buffer_size);

	dr->FillBuffer(dr->audio_renderer, pLock, size);

	/*update current pos*/
    ctx->next_write_pos += size; 
    ctx->next_write_pos %= ctx->total_buffer_size;
    if( FAILED( hr = IDirectSoundBuffer_Unlock(ctx->pOutput, pLock, size, NULL, 0)) ) {
		fprintf(stdout, "Error unlocking DS buffer\n");
	}
}


void DS_LockAndWrite(AudioOutput *dr)
{
	/*OK render*/
	DS_AudioRender(dr);
}

void DS_GotoSleep_Notifs(AudioOutput *dr)
{
	DSCONTEXT();

	if (ctx->use_notif) {
		WaitForSingleObject(ctx->hEvent, INFINITE);
		return;
	}
}
void DS_GotoSleep(AudioOutput *dr)
{
    DWORD play;
	u32 sleep_for;
	DSCONTEXT();

    if (IDirectSoundBuffer_GetCurrentPosition(ctx->pOutput, &play, NULL) != DS_OK ) {
		fprintf(stdout, "error getting DS buffer poitions\n");
		return;
	}
	/*if play pos is in next output buffer, sleep untill buffer is done playing*/
	if ((play > ctx->next_write_pos) && (ctx->next_write_pos + ctx->buffer_size > play) ) {
		sleep_for = ctx->buffer_length_ms;
	} else {
		/*get time till end of current buffer play*/
		while (play >= ctx->buffer_size) play -= ctx->buffer_size;
		sleep_for = ctx->buffer_length_ms * (ctx->buffer_size - play) / ctx->buffer_size;
	}
	Sleep(sleep_for);
}

static u32 DS_QueryOutputSampleRate(AudioOutput *dr, u32 desired_samplerate, u32 NbChannels, u32 nbBitsPerSample)
{
	/*all sample rates supported for now ... */
	return desired_samplerate;
}

static void DS_Pause(AudioOutput *dr, Bool DoFreeze)
{
	DSCONTEXT();
	if (DoFreeze) {
		IDirectSoundBuffer_Stop(ctx->pOutput);
	} else {
		IDirectSoundBuffer_Play(ctx->pOutput, 0, 0, DSBPLAY_LOOPING);
	}
}

static void DS_SetVolume(AudioOutput *dr, u32 Volume)
{
	LONG Vol;
	DSCONTEXT();
	if (Volume > 100) Volume = 100;
	Vol = Volume * (DSBVOLUME_MIN - DSBVOLUME_MAX) / 100;
	IDirectSoundBuffer_SetVolume(ctx->pOutput, Vol);
}

static void DS_SetPan(AudioOutput *dr, u32 Pan)
{
	LONG dspan;
	DSCONTEXT();

	if (Pan > 100) Pan = 100;
	if (Pan > 50) {
		dspan = DSBPAN_RIGHT * (Pan - 50) / 50;
	} else if (Pan < 50) {
		dspan = DSBPAN_LEFT * (50 - Pan) / 50;
	} else {
		dspan = 0;
	}
	IDirectSoundBuffer_SetPan(ctx->pOutput, dspan);
}


static void DS_SetPriority(AudioOutput *dr, u32 Priority)
{
}

static u32 DS_GetAudioDelay(AudioOutput *dr)
{
	DSCONTEXT();
//	if (ctx->use_notif) return 1;

	return ctx->total_length_in_ms;
}

void *NewAudioOutput()
{
	HRESULT hr;
	DSContext *ctx;
	AudioOutput *driv;

	if( FAILED( hr = CoInitialize(NULL) ) ) return NULL;

	
	ctx = malloc(sizeof(DSContext));
	memset(ctx, 0, sizeof(DSContext));

	driv = malloc(sizeof(AudioOutput));
	memset(driv, 0, sizeof(AudioOutput));
	M4_REG_PLUG(driv, M4_AUDIO_OUTPUT_INTERFACE, "DirectSound Audio Output", "gpac distribution", 0);

	driv->opaque = ctx;

	driv->SetupHardware = DS_SetupHardware;
	driv->Shutdown = DS_Shutdown;
	driv->ConfigureOutput = DS_ConfigureOutput;
	driv->SetVolume = DS_SetVolume;
	driv->SetPan = DS_SetPan;
	driv->Pause = DS_Pause;
	driv->SetPriority = DS_SetPriority;
	driv->GetAudioDelay = DS_GetAudioDelay;
	driv->LockAndWrite = DS_LockAndWrite;
	driv->GotoSleep = DS_GotoSleep;
	driv->QueryOutputSampleRate = DS_QueryOutputSampleRate;
	/*never threaded*/
	driv->SelfThreaded = 0;
	return driv;
}

void DeleteAudioOutput(void *ifce)
{
	AudioOutput *dr = (AudioOutput *)ifce;
	DSCONTEXT();

	free(ctx);
	free(ifce);
	CoUninitialize();
}

