/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / file downloader plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "file_dnload.h"

void HTTP_CheckDone(FileDownload *fd)
{
	u32 runtime;
	if (fd->plug->bytes_done == fd->plug->total_size) {
		fd->plug->error = M4EOF;
		SK_Delete(fd->sock);
		fd->sock = NULL;
		fd->plug->net_status = NM_Disconnected;
		FD_PostError(fd);
		return;
	}
	/*update state if not done*/
	runtime = M4_GetSysClock() - fd->start_time;
	if (!runtime) {
		fd->plug->bytes_per_sec = 0.0;
	} else {
		fd->plug->bytes_per_sec = 1000.0f * (fd->plug->bytes_done - fd->cached_size);
		fd->plug->bytes_per_sec /= runtime;
	}
}

static M4INLINE void http_data_recieved(FileDownload *fd, char *bytes, u32 nbBytes)
{
	if (fd->use_cache) {
		if (fd->cache) {
			fwrite(bytes, nbBytes, 1, fd->cache);
			fflush(fd->cache);
		}
		fd->plug->bytes_done += nbBytes;
		fd->plug->error = M4OK;
		fd->plug->OnData(fd->plug, NULL, nbBytes);
	} else {
		fd->plug->bytes_done += nbBytes;
		fd->plug->OnData(fd->plug, bytes, nbBytes);
	}
	HTTP_CheckDone(fd);
}


static M4INLINE u32 http_skip_space(char *val)
{
	u32 ret = 0;
	while (val[ret] == ' ') ret+=1;
	return ret;
}

static M4INLINE char *http_is_header(char *line, char *header_name)
{
	char *res;
	if (strncmp(line, header_name, strlen(header_name))) return NULL;
	res = line + strlen(header_name);
	while ((res[0] == ':') || (res[0] == ' ')) res+=1;
	return res;
}

void HTTP_SendRequest(FileDownload *fd)
{
	unsigned char range_buf[1024];
	char sHTTP[FD_BUFFER_SIZE];
	
	if (fd->cached_size) sprintf(range_buf, "Range: bytes=%d-\r\n", fd->cached_size);

	sprintf(sHTTP, "GET %s HTTP/1.0\r\n"
					"Host: %s\r\n"
					"Accept: */*\r\n" 
					"%s"
					"User-Agent: %s\r\n\r\n", 
					fd->remote_path, fd->server_name, 
					(fd->cached_size ? (const char *) range_buf: ""),
					FD_AGENT_NAME);

	fd->plug->error = SK_Send(fd->sock, sHTTP, strlen(sHTTP));
	if (fd->plug->error) {
		fd->plug->net_status = NM_Disconnected;
		SK_Delete(fd->sock);
		fd->sock = NULL;
	} else {
		fd->plug->net_status = NM_WaitingForAck;
	}
}

void HTTP_ProcessReply(FileDownload *fd)
{
	M4Err e;
	char sHTTP[FD_BUFFER_SIZE];
	unsigned char buf[1024];
	unsigned char comp[400];
	unsigned char *new_location;
	char *hdr;
	s32 bytesRead, res;
	s32 LinePos, Pos;
	u32 rsp_code, BodyStart, ContentLength, first_byte, last_byte, total_size, range, no_range;

	bytesRead = res = 0;

	new_location = NULL;
	while (1) {
		e = SK_Receive(fd->sock, sHTTP, FD_BUFFER_SIZE, bytesRead, &res);
		switch (e) {
		case M4NetworkEmpty:
			return;
		/*socket has been closed while configuring, retry (not sure if the server got the GET)*/
		case M4ConnectionClosed:
			SK_Delete(fd->sock);
			fd->sock = NULL;
			fd->plug->net_status = NM_Setup;
			return;
		case M4OK:
			break;
		default:
			goto exit;
		}		

		bytesRead += res;

		/*locate body start*/
		BodyStart = SP_FindPattern(sHTTP, 0, bytesRead, "\r\n\r\n");
		if (!BodyStart) {
			BodyStart = SP_FindPattern(sHTTP, 0, bytesRead, "\n\n");
			if (!BodyStart) continue;
			BodyStart += 2;
		} else {
			BodyStart += 4;
		}
		break;
	}

	if (bytesRead < 0) {
		e = M4SignalingFailure;
		goto exit;
	}

	LinePos = SP_GetOneLine(sHTTP, 0, bytesRead, buf, 1024);
	Pos = SP_GetComponent(buf, 0, " \t\r\n", comp, 400);

	//pb in the request
	if (strncmp("HTTP", comp, 4) != 0) {
		e = M4SignalingFailure;
		goto exit;
	}
	Pos = SP_GetComponent(buf, Pos, " \t\r\n", comp, 400);
	if (Pos <= 0) {
		e = M4SignalingFailure;
		goto exit;
	}
	rsp_code = (u32) atoi(comp);	


	no_range = range = ContentLength = first_byte = last_byte = total_size = 0;
	//parse header
	while (1) {
		LinePos = SP_GetOneLine(sHTTP, LinePos , bytesRead, buf, 1024);
		if (LinePos < 0) break;

		if ((hdr = http_is_header(buf, "Content-Length")) ) {			
			ContentLength = (u32) atoi(hdr);
		}
		else if ((hdr = http_is_header(buf, "Content-Range")) ) {			
			range = 1;
			if (!strncmp(hdr, "bytes", 5)) {
				hdr += 5;
				if (*hdr == ':') hdr += 1;
				hdr += http_skip_space(hdr);
				if (*hdr == '*') {
					sscanf(hdr, "*/%d", &total_size);
				} else {
					sscanf(hdr, "%d-%d/%d", &first_byte, &last_byte, &total_size);
				}
			}
		}
		else if ((hdr = http_is_header(buf, "Accept-Ranges"))) {
			if (strstr(hdr, "none")) no_range = 1;
		}
		else if ((hdr = http_is_header(buf, "Location"))) {
			new_location = strdup(hdr);
		}
	}
	if (no_range) first_byte = 0;

	if (fd->cached_size) {
		if (total_size && fd->cached_size >= total_size) {
			rsp_code = 200;
			ContentLength = total_size;
		}
		if (ContentLength && fd->cached_size == ContentLength) rsp_code = 200;
	}	

	switch (rsp_code) {
	case 200:
	case 201:
	case 202:
	case 206:
		e = M4OK;
		break;
	/*redirection: extract the new location*/
	case 301:
	case 302:
		if (!new_location || !strlen(new_location) ) {
			e = M4URLNotFound;
			goto exit;
		}
		while (
			(new_location[strlen(new_location)-1] == '\n') 
			|| (new_location[strlen(new_location)-1] == '\r')  )
			new_location[strlen(new_location)-1] = 0;

		/*reset and reconnect*/
		SK_Delete(fd->sock);
		fd->sock = NULL;
		fd->plug->net_status = NM_Setup;
		if (!strnicmp(new_location, "http://", 7)) {
			sscanf(new_location, "http://%[^:/]:%hd", fd->server_name, &fd->session_port);
		}
		hdr = strstr(new_location, "://");
		hdr += 3;
		hdr = strstr(hdr, "/");
		strcpy(fd->remote_path, hdr);
		free(new_location);
		return;
	case 404:
		/*try without cache*/
		if (fd->cached_size) {
			fd->cached_size = 0;
			SK_Delete(fd->sock);
			fd->sock = NULL;
			fd->plug->net_status = NM_Setup;
			return;
		}
		e = M4URLNotFound;
		goto exit;
	case 503:
	default:
		e = M4RemotePeerError;
		goto exit;
	}
	/*some servers may reply without content length, but we MUST have it*/
	if (!ContentLength) e = M4RemotePeerError;
	if (e) goto exit;


	/*done*/
	if (fd->cached_size 
		&& ( (total_size && fd->cached_size >= total_size) || (ContentLength && fd->cached_size == ContentLength)) ) {
		fd->plug->total_size = fd->plug->bytes_done = fd->cached_size;
		/*disconnect*/
		fd->plug->net_status = NM_Disconnected;
		SK_Delete(fd->sock);
		fd->sock = NULL;
		BodyStart = bytesRead;
		fd->plug->error = M4EOF;
		FD_PostError(fd);
	}
	/*no range header, Accep-Ranges deny or dumb server : restart*/
	else if (!range || !first_byte || (first_byte != fd->cached_size) ) {
		fd->cached_size = fd->plug->bytes_done = 0;
		fd->plug->total_size = ContentLength;
		if (fd->use_cache) {
			fd->cache = fopen(fd->cache_name, "wb");
			if (!fd->cache) {
				e = M4IOErr;
				goto exit;
			}
		}
		fd->plug->net_status = NM_Running;
	}
	/*resume*/
	else {
		fd->plug->total_size = ContentLength + fd->cached_size;
		if (fd->use_cache) {
			fd->cache = fopen(fd->cache_name, "ab");
			if (!fd->cache) {
				e = M4IOErr;
				goto exit;
			}
		}
		fd->plug->net_status = NM_Running;
		fd->plug->bytes_done = fd->cached_size;
	}

	fd->start_time = M4_GetSysClock();
	
	//we may have existing data in this buffer ...
	if (!e && (BodyStart < (u32) bytesRead)) 
		http_data_recieved(fd, sHTTP+BodyStart, bytesRead - BodyStart);

exit:
	if (e) {
		fd->plug->net_status = NM_Disconnected;
		fd->plug->error = e;
		SK_Delete(fd->sock);
		fd->sock = NULL;
		FD_PostError(fd);
	}
}

void HTTP_Read(FileDownload *fd)
{
	M4Err e;
	char buffer[FD_BUFFER_SIZE];
	u32 size = 0;

	e = SK_Receive(fd->sock, buffer, FD_BUFFER_SIZE, 0, &size);
	if (!size || e == M4NetworkEmpty) return;

	if (e) {
		SK_Delete(fd->sock);
		fd->sock = NULL;
		fd->plug->net_status = NM_Disconnected;
		fd->plug->error = e;
		FD_PostError(fd);
		return;
	}
	http_data_recieved(fd, buffer, size);

	if (size == FD_BUFFER_SIZE) {
		HTTP_Read(fd);
		return;
	}
}

void HTTP_Task(FileDownload *fd)
{
	switch (fd->plug->net_status) {
	/*setup download*/
	case NM_Setup:
		FD_ConnectTask(fd);
		break;
	case NM_Connected:
		HTTP_SendRequest(fd);
		break;
	case NM_WaitingForAck:
		HTTP_ProcessReply(fd);
		break;
	case NM_Running:
		HTTP_Read(fd);
		break;
	}
}

